<?php
// index.php — پنل وب + کنترل اجرای خودکار
declare(strict_types=1);

$BASE = __DIR__;
$cfg  = require $BASE . '/config.php';
require $BASE . '/app/bot.php';

$paths = $cfg['PATHS'];
ensure_dirs($paths);

/* ==== Helpers for background control ==== */
function is_daemon_running(array $paths): bool {
  $pidfile = $paths['PID'];
  if (!is_file($pidfile)) return false;
  $pid = (int)@file_get_contents($pidfile);
  if ($pid <= 0) return false;
  if (function_exists('posix_kill')) return @posix_kill($pid, 0);
  return true; // fallback
}
function start_daemon(array $paths): array {
  $php = PHP_BINDIR . '/php';
  if (!is_executable($php)) $php = 'php';
  $cmd = escapeshellcmd($php) . ' ' . escapeshellarg($paths['BASE'] . '/daemon.php') . ' > ' . escapeshellarg($paths['DAEMON_LOG']) . ' 2>&1 & echo $!';
  if (function_exists('proc_open')) {
    $des=[0=>['pipe','r'],1=>['pipe','w'],2=>['pipe','w']];
    $p=@proc_open($cmd,$des,$pipes,$paths['BASE']);
    if (is_resource($p)) { @proc_close($p); return ['ok'=>true,'msg'=>'Daemon start issued (proc_open).']; }
  }
  if (function_exists('shell_exec')) { @shell_exec($cmd); return ['ok'=>true,'msg'=>'Daemon start issued (shell_exec).']; }
  return ['ok'=>false,'msg'=>'هاست اجازه اجرای پس‌زمینه (proc_open/shell_exec) نمی‌دهد. از SSH اجرا کن: php daemon.php &'];
}
function stop_daemon(array $paths): array {
  $pidfile=$paths['PID']; if(!is_file($pidfile)) return ['ok'=>true,'msg'=>'در حال اجرا نبود.'];
  $pid=(int)@file_get_contents($pidfile); if($pid>0 && function_exists('posix_kill')) @posix_kill($pid, SIGTERM);
  @unlink($pidfile);
  return ['ok'=>true,'msg'=>'Daemon متوقف شد.'];
}

/* ==== Actions ==== */
$msg = '';
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
  if (isset($_POST['save_links'])) {
    $raw = trim($_POST['links'] ?? '');
    $arr = array_values(array_filter(array_map('trim', preg_split('/\r?\n/', $raw))));
    save_json($paths['LINKS'], $arr);
    $msg = '✅ لیست لینک‌ها ذخیره شد.';
  }
  if (isset($_POST['save_settings'])) {
    $defaults = $cfg['DEFAULTS'];
    $settings = [
      'http_timeout'       => max(1, (int)($_POST['http_timeout'] ?? $defaults['http_timeout'])),
      'connect_timeout'    => max(1, (int)($_POST['connect_timeout'] ?? $defaults['connect_timeout'])),
      'tls_handshake'      => isset($_POST['tls_handshake']),
      'cache_ttl_dns'      => max(60, (int)($_POST['cache_ttl_dns'] ?? $defaults['cache_ttl_dns'])),
      'cache_ttl_asn'      => max(60, (int)($_POST['cache_ttl_asn'] ?? $defaults['cache_ttl_asn'])),
      'cache_ttl_config'   => max(60, (int)($_POST['cache_ttl_config'] ?? $defaults['cache_ttl_config'])),
      'poll_interval'      => max(1, (int)($_POST['poll_interval'] ?? $defaults['poll_interval'])),
    ];
    save_json($paths['SETTINGS'], $settings);
    $msg = '✅ تنظیمات ذخیره شد.';
  }
  if (isset($_POST['run_now'])) {
    $summary = run_once($cfg);
    $msg = '✅ اجرا شد: ' . htmlspecialchars(json_encode($summary, JSON_UNESCAPED_UNICODE));
  }
  if (isset($_POST['start_daemon'])) {
    $res = start_daemon($paths);
    $msg = ($res['ok']?'✅ ':'❌ ').$res['msg'];
  }
  if (isset($_POST['stop_daemon'])) {
    $res = stop_daemon($paths);
    $msg = ($res['ok']?'✅ ':'❌ ').$res['msg'];
  }
}

$links    = load_json($paths['LINKS'], []);
$links_tx = implode("\n", $links);
$settings = load_json($paths['SETTINGS'], $cfg['DEFAULTS']);
$last     = load_json($paths['SUMMARY'], ['time'=>'-', 'inputs'=>0, 'unique_endpoints'=>0, 'groups'=>[]]);
$running  = is_daemon_running($paths);
?>
<!doctype html><meta charset="utf-8">
<title>Datacenter Checker Panel</title>
<style>
  body{font-family:Inter,IRANSans,sans-serif;background:#f7f9fb;margin:0;color:#0f172a}
  header{display:flex;align-items:center;justify-content:space-between;padding:16px;background:#0ea5e9;color:#fff}
  main{padding:24px;max-width:1100px;margin:0 auto}
  .grid{display:grid;grid-template-columns:1fr 1fr;gap:20px}
  .card{background:#fff;border-radius:14px;box-shadow:0 6px 24px rgba(2,6,23,.06);padding:18px}
  textarea,input[type=number]{width:100%;padding:10px;border:1px solid #e5e7eb;border-radius:10px;background:#fbfdff}
  .btn{display:inline-block;background:#0ea5e9;color:#fff;border:0;border-radius:10px;padding:10px 14px;cursor:pointer}
  .btn-outline{background:#fff;color:#0ea5e9;border:1px solid #0ea5e9}
  .row{display:flex;gap:10px;flex-wrap:wrap}
  .muted{color:#64748b;font-size:.9rem}
  .msg{margin:10px 0;color:#16a34a}
  table{width:100%;border-collapse:collapse}
  th,td{padding:8px;border-bottom:1px solid #eef2f7;text-align:start;font-size:.92rem}
  code{background:#f1f5f9;padding:2px 6px;border-radius:6px}
  .status{padding:6px 10px;border-radius:8px; background:#fff; color:#0f172a}
</style>

<header>
  <strong>Datacenter Checker Panel</strong>
  <div class="status">وضعیت خودکار: <?= $running ? '🟢 در حال اجرا' : '🔴 متوقف' ?></div>
</header>

<main>
  <?php if ($msg): ?><div class="msg"><?=htmlspecialchars($msg)?></div><?php endif; ?>

  <div class="grid">
    <div class="card">
      <h3>لینک‌های ورودی</h3>
      <p class="muted">هر خط یک لینک شامل کانفیگ‌ها (vless/vmess/trojan/ss یا JSON).</p>
      <form method="post">
        <textarea name="links" rows="10" placeholder="https://example.com/configs.txt"><?=htmlspecialchars($links_tx)?></textarea>
        <div class="row" style="margin-top:10px">
          <button class="btn" name="save_links">ذخیره لینک‌ها</button>
          <button class="btn" name="run_now">الان بررسی کن</button>
        </div>
      </form>
    </div>

    <div class="card">
      <h3>تنظیمات</h3>
      <form method="post">
        <label>HTTP Timeout (sec)</label>
        <input type="number" min="1" name="http_timeout" value="<?=htmlspecialchars((string)$settings['http_timeout'])?>">
        <label>TCP Connect Timeout (sec)</label>
        <input type="number" min="1" name="connect_timeout" value="<?=htmlspecialchars((string)$settings['connect_timeout'])?>">
        <label><input type="checkbox" name="tls_handshake" <?=!empty($settings['tls_handshake'])?'checked':''?>> اندازه‌گیری TLS Handshake</label>
        <div class="row">
          <div style="flex:1">
            <label>DNS Cache TTL (sec)</label>
            <input type="number" min="60" name="cache_ttl_dns" value="<?=htmlspecialchars((string)$settings['cache_ttl_dns'])?>">
          </div>
          <div style="flex:1">
            <label>ASN Cache TTL (sec)</label>
            <input type="number" min="60" name="cache_ttl_asn" value="<?=htmlspecialchars((string)$settings['cache_ttl_asn'])?>">
          </div>
        </div>
        <div class="row" style="margin-top:10px">
          <div style="flex:1">
            <label>Config Cache TTL (sec)</label>
            <input type="number" min="60" name="cache_ttl_config" value="<?=htmlspecialchars((string)$settings['cache_ttl_config'])?>">
          </div>
          <div style="flex:1">
            <label>Poll Interval (sec)</label>
            <input type="number" min="1" name="poll_interval" value="<?=htmlspecialchars((string)$settings['poll_interval'])?>">
          </div>
        </div>
        <div class="row" style="margin-top:10px">
          <button class="btn" name="save_settings">ذخیره تنظیمات</button>
        </div>
      </form>
    </div>
  </div>

  <div class="grid" style="margin-top:20px">
    <div class="card">
      <h3>اجرای خودکار</h3>
      <form method="post" class="row">
        <button class="btn" name="start_daemon" type="submit">Start</button>
        <button class="btn btn-outline" name="stop_daemon" type="submit">Stop</button>
      </form>
      <p class="muted">اگر Start خطای «اجازه اجرا ندارد» داد، از SSH اجرا کن: <code>php daemon.php &</code></p>
    </div>

    <div class="card">
      <h3>آخرین اجرا</h3>
      <table>
        <tr><th>زمان</th><td><?=htmlspecialchars($last['time'] ?? '-')?></td></tr>
        <tr><th>تعداد ورودی</th><td><?=htmlspecialchars((string)($last['inputs'] ?? 0))?></td></tr>
        <tr><th>اندپوینت یکتا</th><td><?=htmlspecialchars((string)($last['unique_endpoints'] ?? 0))?></td></tr>
      </table>
      <h4 style="margin-top:10px">گروه‌ها</h4>
      <table>
        <tr><th>دیتاسنتر</th><th>تعداد</th></tr>
        <?php foreach (($last['groups'] ?? []) as $k=>$v): ?>
          <tr><td><?=htmlspecialchars($k)?></td><td><?=$v?></td></tr>
        <?php endforeach; ?>
      </table>
    </div>

    <div class="card">
      <h3>خروجی‌ها</h3>
      <p class="muted">فایل‌ها داخل پوشه <code><?=htmlspecialchars(basename($paths['OUT_DIR']))?></code> ذخیره می‌شوند.</p>
      <ul>
        <?php foreach (glob($paths['OUT_DIR'].'/*.{txt,json}', GLOB_BRACE) as $f): $bn=basename($f); $url=dirname($_SERVER['PHP_SELF']).'/out/'.$bn; ?>
          <li><a href="<?=htmlspecialchars($url)?>" target="_blank"><?=htmlspecialchars($bn)?></a></li>
        <?php endforeach; ?>
      </ul>
    </div>
  </div>
</main>