<?php
/**
 * Y.php v3 — Attempts multiple header presets and saves full responses for debugging.
 * Usage: https://YOURDOMAIN/Api/77/Y.php
 * Optional query params:
 *   token=...     -> Authorization (if you already have token)
 *   cookie=...    -> Cookie header
 *   extra=H1: V1||H2: V2   -> extra headers
 *
 * Outputs (out/):
 *   <name>_attempt_1_resp.txt  (full raw response info)
 *   ... attempt_N ...
 *   configs_raw.txt
 *   vmess_decoded.jsonl
 *
 * Place on shared hosting — uses curl, json, zlib.
 */

declare(strict_types=1);
header('Content-Type: text/plain; charset=utf-8');

$ENDPOINTS = [
  'profiles' => 'https://newversion-app-data.com/api/v6/profiles',
  'settings' => 'https://newversion-app-data.com/api/v4/apps/settings',
];

$Q = array_change_key_case($_GET, CASE_LOWER);
$TOKEN  = $Q['token']  ?? '';
$COOKIE = $Q['cookie'] ?? '';
$EXTRA  = $Q['extra']  ?? '';

$baseDir = __DIR__;
$outDir  = $baseDir . '/out';
ensure_dir($outDir);

echo "Y.php v3 — trying multiple header presets...\n\n";

/* ---------- PRESETS (from least to most "app-like") ---------- */
$presets = [];

// 1) plain
$presets[] = [
  'name' => 'plain',
  'headers' => [
    'Accept: application/json',
    'Accept-Encoding: gzip',
    'User-Agent: curl/7.68.0',
  ],
];

// 2) generic mobile UA + language
$presets[] = [
  'name' => 'mobile_lang',
  'headers' => [
    'Accept: application/json',
    'Accept-Encoding: gzip',
    'User-Agent: Dalvik/2.1.0 (Linux; U; Android 12)',
    'Accept-Language: en-US,en;q=0.9',
  ],
];

// 3) app-like with package + version
$presets[] = [
  'name' => 'app_basic',
  'headers' => [
    'Accept: application/json',
    'Accept-Encoding: gzip',
    'User-Agent: Dalvik/2.1.0 (Android) OkHttp',
    'X-Package-Name: vpn.goldenvpn.unblock',
    'X-App-Version: 2.1.0',
    'Accept-Language: en-US,en;q=0.9',
    'Referer: https://newversion-app-data.com/',
  ],
];

// 4) include common app headers (Firebase/persistence keys seen in logs)
$presets[] = [
  'name' => 'app_firebase',
  'headers' => [
    'Accept: application/json',
    'Accept-Encoding: gzip',
    'User-Agent: Dalvik/2.1.0 (Android) OkHttp',
    'X-Package-Name: vpn.goldenvpn.unblock',
    'X-App-Version: 2.1.0',
    'X-Firebase-App: 1:250249565590:android:b4cef1cddc09a3c95c39b7',
    'X-Persistence-Key: [DEFAULT]',
    'Accept-Language: en-US,en;q=0.9',
  ],
];

// 5) maximal: include X-SID and other hints (from server response headers you pasted earlier x-sid: 2261)
$presets[] = [
  'name' => 'app_max',
  'headers' => [
    'Accept: application/json',
    'Accept-Encoding: gzip',
    'User-Agent: Dalvik/2.1.0 (Android) OkHttp',
    'X-Package-Name: vpn.goldenvpn.unblock',
    'X-App-Version: 2.1.0',
    'X-Firebase-App: 1:250249565590:android:b4cef1cddc09a3c95c39b7',
    'X-Persistence-Key: [DEFAULT]',
    'X-SID: 2261',
    'X-Requested-With: XMLHttpRequest',
    'Accept-Language: en-US,en;q=0.9',
    'Referer: https://newversion-app-data.com/',
    'Origin: https://newversion-app-data.com',
  ],
];

// apply user-supplied token/cookie/extra to all presets if provided
foreach ($presets as $i => $p) {
  $h = $p['headers'];
  if ($TOKEN !== '') {
    $hdr = (stripos($TOKEN, 'bearer ') === 0) ? $TOKEN : ('Bearer '.$TOKEN);
    $h[] = 'Authorization: '.$hdr;
  }
  if ($COOKIE !== '') $h[] = 'Cookie: '.$COOKIE;
  if ($EXTRA !== '') {
    foreach (explode('||', $EXTRA) as $line) {
      $line = trim($line);
      if ($line !== '') $h[] = $line;
    }
  }
  $presets[$i]['headers'] = $h;
}

/* ---------- try each endpoint × preset ---------- */
$foundLinks = [];
$saveDecoded = [];

foreach ($ENDPOINTS as $name => $url) {
  echo "== endpoint: $name -> $url\n";
  $attempt = 0;
  foreach ($presets as $p) {
    $attempt++;
    $pname = $p['name'];
    echo "  * attempt $attempt preset=$pname ... ";
    [$body, $info, $err] = http_get_with_response($url, $p['headers']);
    $code = (int)($info['http_code'] ?? 0);

    // save full response for inspection
    $fn = "{$outDir}/{$name}_attempt_{$attempt}_{$pname}.resp.txt";
    $s = "=== REQUEST HEADERS ===\n" . implode("\n", $p['headers']) . "\n\n";
    $s .= "=== CURL INFO ===\n" . json_encode($info, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE) . "\n\n";
    if ($err) $s .= "=== CURL ERROR ===\n$err\n\n";
    $s .= "=== HTTP STATUS: $code ===\n\n";
    $s .= "=== RESPONSE HEADERS & BODY ===\n";
    $s .= $body ?? '';
    @file_put_contents($fn, $s);
    echo "saved -> " . basename($fn) . " (HTTP $code)\n";

    if ($err) continue;
    if ($code !== 200) {
      // if 401/403, keep trying other presets
      continue;
    }

    // 200: gather text for extraction
    if (is_string($body) && $body !== '') {
      // attempt JSON parse and collect strings
      $all = [$body];
      $aj = json_decode($body, true);
      if (is_array($aj)) {
        @file_put_contents("{$outDir}/{$name}.json", json_encode($aj, JSON_UNESCAPED_UNICODE|JSON_PRETTY_PRINT|JSON_UNESCAPED_SLASHES));
        $all[] = implode("\n", collect_strings($aj));
      }
      $blob = implode("\n", $all);
      $links = extract_config_links($blob);
      if ($links) {
        foreach ($links as $l) { $foundLinks[] = $l; }
      }
    }
    // if 200 found, break preset loop to avoid duplicate attempts (but continue to next endpoint)
    break;
  } // presets
} // endpoints

$foundLinks = array_values(array_unique($foundLinks));
echo "\n=== summary ===\n";
echo "Total unique links found: " . count($foundLinks) . "\n";
@file_put_contents("{$outDir}/configs_raw.txt", implode("\n", $foundLinks) . (count($foundLinks) ? "\n" : ''));

if ($foundLinks) {
  echo "Saved configs_raw.txt\n";
  foreach ($foundLinks as $ln) {
    if (stripos($ln, 'vmess://') === 0) {
      $b64 = substr($ln, strlen('vmess://'));
      $j = safe_base64_json_decode($b64);
      if ($j !== null) $saveDecoded[] = json_encode($j, JSON_UNESCAPED_UNICODE|JSON_UNESCAPED_SLASHES);
    }
  }
  if ($saveDecoded) {
    @file_put_contents("{$outDir}/vmess_decoded.jsonl", implode("\n", $saveDecoded) . "\n");
    echo "Saved vmess_decoded.jsonl (" . count($saveDecoded) . ")\n";
  } else {
    echo "No vmess JSON decoded.\n";
  }
} else {
  echo "No links extracted. Check the saved response files in out/ — open the *_attempt_*.resp.txt files.\n";
  echo "Please paste here one of those *_attempt_*.resp.txt files (for example out/profiles_attempt_2_app_basic.resp.txt) and I'll tell you exactly which header/token is required.\n";
}

/* ================= helpers ================= */

function ensure_dir(string $d) { if (!is_dir($d)) @mkdir($d, 0775, true); }

function http_get_with_response(string $url, array $headers): array {
  $ch = curl_init($url);
  curl_setopt_array($ch, [
    CURLOPT_RETURNTRANSFER => true,
    CURLOPT_FOLLOWLOCATION => true,
    CURLOPT_TIMEOUT => 25,
    CURLOPT_CONNECTTIMEOUT => 10,
    CURLOPT_ENCODING => 'gzip,deflate',
    CURLOPT_HTTPHEADER => $headers,
    CURLOPT_HEADER => false,
    // capture response headers via CURLINFO_HEADER_OUT not needed; we save info + body
  ]);
  $resp = curl_exec($ch);
  $info = curl_getinfo($ch);
  $err  = curl_error($ch);
  curl_close($ch);
  // Build a readable response: include headers if server returned them via info (can't get full raw headers here easily),
  // but we saved curl_getinfo which includes content_type, total_time, etc., and body.
  $out = $resp;
  // optionally, try to pretty-print JSON responses for readability when saving
  $json = json_decode($resp, true);
  if (is_array($json)) $out = json_encode($json, JSON_UNESCAPED_UNICODE|JSON_PRETTY_PRINT|JSON_UNESCAPED_SLASHES);
  return [$out, $info, $err];
}

function collect_strings($node): array {
  $out = [];
  $stack = [$node];
  while ($stack) {
    $cur = array_pop($stack);
    if (is_array($cur)) {
      foreach ($cur as $k => $v) {
        if (is_string($k)) $out[] = $k;
        if (is_string($v)) $out[] = $v;
        elseif (is_array($v)) $stack[] = $v;
      }
    }
  }
  return array_values(array_unique(array_map('trim', $out)));
}

function extract_config_links(string $text): array {
  if (!preg_match_all('~\b(vless://\S+|vmess://[A-Za-z0-9_\-+/=]+|trojan://\S+|ss://\S+)~i', $text, $m)) {
    return [];
  }
  $links = array_map(function($s){ $s = trim($s); return rtrim($s, ",'\"\\)\\]}>"); }, $m[0]);
  return array_values(array_unique(array_filter($links, fn($s)=>strlen($s)>10)));
}

function safe_base64_json_decode(string $b64) {
  $b64 = trim($b64);
  $pad = strlen($b64) % 4;
  if ($pad>0) $b64 .= str_repeat('=', 4-$pad);
  $raw = base64_decode($b64, true);
  if ($raw === false) return null;
  $j = json_decode($raw, true);
  return is_array($j) ? $j : null;
}