<?php
// inspect_responses.php
// Put next to Y.php and the out/ folder. Run in browser or CLI.
// It summarizes all out/*_attempt_*.resp.txt files and highlights auth hints.

declare(strict_types=1);
header('Content-Type: text/plain; charset=utf-8');

$base = __DIR__;
$outdir = $base . '/out';
if (!is_dir($outdir)) {
    echo "No out/ directory found at $outdir\n";
    exit(1);
}

$files = glob($outdir . '/*_attempt_*.resp.txt');
if (!$files) {
    echo "No *_attempt_*.resp.txt files found in $outdir\n";
    echo "List directory:\n";
    foreach (scandir($outdir) as $f) echo "  - $f\n";
    exit(0);
}

echo "Found " . count($files) . " response files in out/\n\n";

$keywords = [
  'token','auth','authorization','bearer','api_key','x-api-key','www-authenticate',
  'signature','sign','device','device_id','install','session','cookie','login','expired','invalid'
];

foreach ($files as $f) {
  $s = @file_get_contents($f);
  if ($s === false) { echo "Could not read $f\n"; continue; }
  echo "=== FILE: " . basename($f) . " ===\n";

  // Try to extract CURL INFO block if present (we saved JSON in http_get_with_response)
  // We'll look for a JSON object near "CURL INFO" or try to find HTTP STATUS line.
  $httpCode = null;
  if (preg_match('/"http_code"\s*:\s*(\d{3})/i', $s, $m)) {
    $httpCode = intval($m[1]);
  } elseif (preg_match('/HTTP STATUS:\s*(\d{3})/i', $s, $m)) {
    $httpCode = intval($m[1]);
  }
  echo "HTTP code: " . ($httpCode ?? 'unknown') . "\n";

  // Show first 10 lines of the file for quick view
  $lines = explode("\n", $s);
  $preview = array_slice($lines, 0, 18);
  echo "---- preview (first 18 lines) ----\n";
  foreach ($preview as $ln) echo rtrim($ln, "\r\n") . "\n";
  echo "---- end preview ----\n";

  // Search for keywords and show matching context snippets
  $lower = mb_strtolower($s);
  $found = [];
  foreach ($keywords as $kw) {
    $pos = mb_strpos($lower, $kw);
    if ($pos !== false) {
      $snippet = mb_substr($s, max(0, $pos-60), 140);
      $found[$kw] = trim(preg_replace('/\s+/', ' ', $snippet));
    }
  }

  if ($found) {
    echo "Potential auth-related matches:\n";
    foreach ($found as $k => $snip) {
      echo " - $k : " . (strlen($snip)>200 ? substr($snip,0,200).'...' : $snip) . "\n";
    }
  } else {
    echo "No obvious auth keywords found in file.\n";
  }

  // Try to pretty print if body is JSON (search for first { ... })
  if (preg_match('/\{[\s\S]*\}/', $s, $jm)) {
    $firstJson = $jm[0];
    $decoded = json_decode($firstJson, true);
    if (is_array($decoded)) {
      echo "Response includes a JSON object (printing top-level keys):\n";
      foreach ($decoded as $k=>$v) {
        if (is_scalar($v)) {
          echo "  $k => " . (string)$v . "\n";
        } else {
          echo "  $k => " . gettype($v) . "\n";
        }
      }
    }
  }

  echo "\n\n";
}

echo "DONE. Next: paste the output above here.\n";