<?php
/**
 * Nexy.php — v2ray/Xray config aggregator (NO cache, strict health, keep outbounds AS-IS)
 * PHP 7.4+
 */

/* ====== USER SETTINGS ====== */
$LINKS = [
    'https://fitn1.ir/Api/Bionet/trun.php',
    'https://fitn1.ir/Api/Vip%20vpn/trun.php',
    'https://fitn1.ir/Api/Verde/trun.php',
    'https://fitn1.ir/Api/Zoro/trun.php',
    'https://fitn1.ir/Api/Unic/trun.php',
    'https://fitn1.ir/Api/Taxi/trun.php',
    'https://fitn1.ir/Api/Shieldify/Shieldify.php',
    'https://fitn1.ir/Api/Ox%20vpn/Ox.php',
    'https://fitn1.ir/Api/Lima/trun.php',
    'https://fitn1.ir/Api/Lemon/lemon.php',
    'https://fitn1.ir/Api/Rosa/Rosahome.php',
    'https://fitn1.ir/Api/Ash/trun.php',
    'https://fitn1.ir/Api/Bull/trun.php',
    'https://fitn1.ir/Api/Homa/homa.php'
];

$HTTP_TIMEOUT   = 10;  // seconds for pulling raw configs
$HEALTH_TIMEOUT = 4;   // seconds for health-check

// سوییچ FakeDNS (برای روشن/خاموش کردن استفاده از FakeDNS و destOverride=fakedns)
$ENABLE_FAKEDNS = true;

define('CHECK_REQUIRE_ALL', false);

header('Content-Type: application/json; charset=utf-8');

/** ---------- HTTP fetch (no cache) ---------- */
function http_get_nocache(string $url, int $timeout): ?string {
    $ctx = stream_context_create([
        'http' => [
            'timeout' => $timeout,
            'ignore_errors' => true,
            'header' =>
                "Cache-Control: no-cache, no-store, must-revalidate\r\n" .
                "Pragma: no-cache\r\n" .
                "User-Agent: Mozilla/5.0\r\n" .
                "Connection: close\r\n",
        ],
        'ssl'  => [
            'verify_peer' => false,
            'verify_peer_name' => false,
        ],
    ]);

    $data = @file_get_contents($url, false, $ctx);
    return ($data === false || $data === '') ? null : $data;
}

/** ---------- small helpers ---------- */
function is_vless(string $s): bool { return stripos($s, 'vless://') === 0; }
function is_vmess(string $s): bool { return stripos($s, 'vmess://') === 0; }

function remove_nulls(&$arr) {
    if (!is_array($arr)) return;
    foreach ($arr as $k => $v) {
        if (is_array($v)) {
            remove_nulls($arr[$k]);
        } elseif ($v === null) {
            unset($arr[$k]);
        }
    }
}

/** ---------- VLESS parser ---------- */
function parse_vless(string $uri): ?array {
    $p = parse_url($uri);
    if (!$p || empty($p['host']) || empty($p['user'])) return null;

    $host = $p['host'];
    $port = isset($p['port']) ? intval($p['port']) : 443;
    $uuid = $p['user'];
    $tag  = $p['fragment'] ?? 'server';

    $q = [];
    if (isset($p['query'])) parse_str($p['query'], $q);

    $network  = isset($q['type']) ? strtolower($q['type']) : 'tcp';
    $security = isset($q['security']) ? strtolower($q['security']) : 'none';

    $user = ['id' => $uuid, 'encryption' => 'none'];
    if (!empty($q['flow'])) $user['flow'] = $q['flow'];

    $out = [
        'tag' => $tag,
        'protocol' => 'vless',
        'settings' => [
            'vnext' => [[
                'address' => $host,
                'port'    => $port,
                'users'   => [ $user ]
            ]]
        ],
        'streamSettings' => [
            'network'  => $network,
            'security' => $security
        ],
        'mux' => [ 'enabled' => false ],
    ];

    if (isset($q['type'])) {
        if ($q['type'] === 'ws') {
            $path       = $q['path'] ?? '/';
            $hostHeader = $q['host'] ?? $host;
            $out['streamSettings']['wsSettings'] = [
                'path'    => $path,
                'headers' => ['Host' => $hostHeader],
            ];
        } elseif ($q['type'] === 'tcp') {
            if (!empty($q['headerType']) && $q['headerType'] === 'http') {
                $out['streamSettings']['tcpSettings'] = [
                    'header' => [
                        'type'    => 'http',
                        'request' => [
                            'version' => '1.1',
                            'method'  => 'HEAD',
                            'path'    => ['/'],
                            'headers' => [
                                'Host'            => [$q['host'] ?? 'example.com'],
                                'Connection'      => ['keep-alive'],
                                'Accept-Encoding' => ['gzip, deflate, br'],
                                'User-Agent'      => [
                                    'Mozilla/5.0 (Linux; Android 14; Pixel 7 Pro Build/XXXXX; wv) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/129.0.0.0 Mobile Safari/537.36'
                                ]
                            ]
                        ]
                    ]
                ];
            } else {
                $out['streamSettings']['tcpSettings'] = [ 'header' => [ 'type' => 'none' ] ];
            }
        } elseif ($q['type'] === 'grpc') {
            $out['streamSettings']['grpcSettings'] = [
                'serviceName' => $q['serviceName'] ?? 'grpc',
                'multiMode'   => false
            ];
        }
    }

    if (isset($q['security'])) {
        if ($q['security'] === 'reality') {
            $re = [
                'show'        => false,
                'fingerprint' => $q['fp'] ?? 'chrome',
                'serverName'  => $q['sni'] ?? $host,
                'publicKey'   => $q['pbk'] ?? '',
                'shortId'     => $q['sid'] ?? '',
                'spiderX'     => $q['spx'] ?? '/',
                'allowInsecure' => false
            ];
            if (!empty($q['pbk'])) $re['publicKey'] = $q['pbk'];
            if (!empty($q['sid'])) $re['shortId']   = $q['sid'];
            if (!empty($q['spx'])) $re['spiderX']   = $q['spx'];
            $out['streamSettings']['realitySettings'] = $re;
        } elseif ($security === 'tls') {
            $out['streamSettings']['tlsSettings'] = [
                'allowInsecure' => false,
                'fingerprint'   => $q['fp'] ?? 'chrome',
                'serverName'    => $q['sni'] ?? $host
            ];
        } else {
            $out['streamSettings']['security'] = 'none';
        }
    }

    if ($network === 'tcp') {
        if (!empty($q['headerType']) && $q['headerType'] === 'http') {
            $out['streamSettings']['tcpSettings'] = [
                'header' => [
                    'type'    => 'http',
                    'request' => [
                        'version' => '1.1',
                        'method'  => 'HEAD',
                        'path'    => ['/'],
                        'headers' => [
                            'Host'            => [$q['host'] ?? 'example.com'],
                            'Connection'      => ['keep-alive'],
                            'Accept-Encoding' => ['gzip, deflate, br'],
                            'User-Agent'      => [
                                'Mozilla/5.0 (Linux; Android 14; Pixel 7 Pro Build/XXXXX; wv) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/129.0.0.0 Mobile Safari/537.36'
                            ]
                        ]
                    ]
                ]
            ];
        } else {
            $out['streamSettings']['tcpSettings'] = [ 'header' => [ 'type' => 'none' ] ];
        }
    }

    remove_nulls($out);
    return $out;
}

/** ---------- VMESS parser ---------- */
function parse_vmess(string $uri): ?array {
    $b64  = substr($uri, 8);
    $json = base64_decode(strtr($b64, '-_', '+/'), true);
    if ($json === false) return null;
    $o = json_decode($json, true);
    if (!is_array($o) || empty($o['add']) || empty($o['id']) || empty($o['port'])) return null;

    $host = $o['add'];
    $port = intval($o['port']);
    $uuid = $o['id'];
    $tag  = $o['ps'] ?? 'vmess';

    $network  = strtolower($o['net'] ?? 'tcp');
    $security = !empty($o['tls']) ? 'tls' : 'none';

    $user = [
        'id'         => $uuid,
        'alterId'    => 0,
        'security'   => 'auto',
        'encryption' => 'none'
    ];

    $out = [
        'tag'       => $tag,
        'protocol'  => 'vmess',
        'settings'  => [
            'vnext' => [[
                'address' => $host,
                'port'    => $port,
                'users'   => [ $user ]
            ]]
        ],
        'streamSettings' => [
            'network'  => $network,
            'security' => $security
        ],
        'mux' => ['enabled' => false],
    ];

    if ($network === 'ws') {
        $path       = $o['path'] ?? '/';
        $hostHeader = $o['host'] ?? $host;
        $out['streamSettings']['wsSettings'] = [
            'path'    => $path,
            'headers' => ['Host' => $hostHeader],
        ];
    } elseif ($network === 'grpc') {
        $out['streamSettings']['grpcSettings'] = [
            'serviceName' => $o['path'] ?? 'grpc',
            'multiMode'   => false
        ];
    } elseif ($network === 'tcp') {
        $out['streamSettings']['tcpSettings'] = [ 'header' => ['type' => 'none'] ];
    }

    if ($security === 'tls') {
        $out['streamSettings']['tlsSettings'] = [
            'allowInsecure' => false,
            'serverName'    => $o['sni'] ?? ($o['host'] ?? $host),
            'fingerprint'   => 'chrome'
        ];
    }

    remove_nulls($out);
    return $out;
}

/** ---------- normalize outbound ---------- */
function normalize_outbound(array $ob): array {
    if (empty($ob['protocol']) && !empty($ob['proto'])) {
        $ob['protocol'] = $ob['proto'];
    }
    if (!isset($ob['settings']) || !is_array($ob['settings'])) {
        $ob['settings'] = [];
    }
    if (!isset($ob['streamSettings']) || !is_array($ob['streamSettings'])) {
        $ob['streamSettings'] = [];
    }
    if (!isset($ob['streamSettings']['network'])) {
        $ob['streamSettings']['network'] = 'tcp';
    }
    if (!isset($ob['streamSettings']['security'])) {
        $ob['streamSettings']['security'] = 'none';
    }
    if (!isset($ob['mux']) || !is_array($ob['mux'])) {
        $ob['mux'] = ['enabled' => false];
    }
    if (!isset($ob['tag']) || $ob['tag'] === '') {
        $ob['tag'] = $ob['protocol'] ?? 'server';
    }

    remove_nulls($ob);
    return $ob;
}

/** ---------- sockopt + padding + TLS helper ---------- */
function add_sockopt_to_outbound(array $ob): array {
    if (!isset($ob['streamSettings']) || !is_array($ob['streamSettings'])) {
        $ob['streamSettings'] = [];
    }

    // padding: false روی همه استریم‌ها
    if (!isset($ob['streamSettings']['padding']) || !is_array($ob['streamSettings']['padding'])) {
        $ob['streamSettings']['padding'] = [
            'enabled' => false,
        ];
    }

    // sockopt پیشرفته
    if (!isset($ob['streamSettings']['sockopt']) || !is_array($ob['streamSettings']['sockopt'])) {
        $ob['streamSettings']['sockopt'] = [];
    }
    $sock = &$ob['streamSettings']['sockopt'];

    if (!array_key_exists('tcpFastOpen', $sock))          $sock['tcpFastOpen']        = true;
    if (!array_key_exists('tcpNoDelay', $sock))           $sock['tcpNoDelay']         = true;
    if (!array_key_exists('mtu', $sock))                  $sock['mtu']                = 1280;
    if (!array_key_exists('tcpKeepAliveIdle', $sock))     $sock['tcpKeepAliveIdle']   = 20;
    if (!array_key_exists('tcpKeepAliveInterval', $sock)) $sock['tcpKeepAliveInterval'] = 5;
    if (!array_key_exists('tcpKeepAliveCount', $sock))    $sock['tcpKeepAliveCount']  = 3;
    if (!array_key_exists('tcpUserTimeout', $sock))       $sock['tcpUserTimeout']     = 150000;
    if (!array_key_exists('tcpMaxSeg', $sock))            $sock['tcpMaxSeg']          = 0;
    if (!array_key_exists('tcpBrutal', $sock))            $sock['tcpBrutal']          = true;
    if (!array_key_exists('tproxy', $sock))               $sock['tproxy']             = 'off';
    if (!array_key_exists('fragment', $sock) || !is_array($sock['fragment'])) {
        $sock['fragment'] = [
            'packets'  => 3,
            'length'   => 500,
            'interval' => 1,
        ];
    }

    // TLS: فعال کردن session resumption + session tickets (در صورت وجود TLS)
    $security = $ob['streamSettings']['security'] ?? null;
    if ($security === 'tls') {
        if (!isset($ob['streamSettings']['tlsSettings']) || !is_array($ob['streamSettings']['tlsSettings'])) {
            $ob['streamSettings']['tlsSettings'] = [];
        }
        if (!array_key_exists('enableSessionResumption', $ob['streamSettings']['tlsSettings'])) {
            $ob['streamSettings']['tlsSettings']['enableSessionResumption'] = true;
        }
        if (!array_key_exists('sessionTickets', $ob['streamSettings']['tlsSettings'])) {
            $ob['streamSettings']['tlsSettings']['sessionTickets'] = true;
        }
    }

    return $ob;
}

/** ---------- detect list-style array ---------- */
function is_list_array(array $a): bool {
    $i = 0;
    foreach ($a as $k => $_) {
        if ($k !== $i) return false;
        $i++;
    }
    return true;
}

/** ---------- basic outbound validation ---------- */
function outbound_is_valid(array $ob): bool {
    $proto = strtolower($ob['protocol'] ?? '');

    if ($proto === 'vless' || $proto === 'vmess') {
        if (empty($ob['settings']['vnext']) || !is_array($ob['settings']['vnext'])) return false;
        $first = $ob['settings']['vnext'][0] ?? null;
        if (!is_array($first)) return false;
        $addr = $first['address'] ?? '';
        $port = $first['port'] ?? 0;

        if (!is_string($addr) || $addr === '') return false;
        if (!is_int($port) && !ctype_digit((string)$port)) return false;
        $port = (int)$port;
        if ($port <= 0 || $port > 65535) return false;
    }

    if (!empty($ob['settings']) && is_array($ob['settings']) && is_list_array($ob['settings'])) {
        return false;
    }

    return true;
}

/** ---------- JSON extraction ---------- */
function extract_outbounds_from_json(string $json): array {
    $data = json_decode($json, true);
    if (!is_array($data)) return [];
    if (isset($data['outbounds']) && is_array($data['outbounds'])) {
        $outs = [];
        foreach ($data['outbounds'] as $ob) {
            if (!is_array($ob) || empty($ob['protocol'])) continue;
            $outs[] = normalize_outbound($ob);
        }
        return $outs;
    }
    if (isset($data['protocol'])) return [ normalize_outbound($data) ];
    return [];
}

/** ---------- smart extraction ---------- */
function smart_extract_outbounds(string $raw): array {
    $t = trim($raw);

    $outs = extract_outbounds_from_json($t);
    if ($outs) return $outs;

    $first = strpos($t, '{'); $last = strrpos($t, '}');
    if ($first !== false && $last !== false && $last > $first) {
        $chunk = substr($t, $first, $last - $first + 1);
        $outs = extract_outbounds_from_json($chunk);
        if ($outs) return $outs;
    }

    $multi = [];
    if (preg_match_all('/\{(?:[^{}]|(?R))*\}/s', $t, $m)) {
        foreach ($m[0] as $blk) {
            $o = extract_outbounds_from_json($blk);
            if ($o) $multi = array_merge($multi, $o);
        }
        if ($multi) return $multi;
    }

    $col = [];
    foreach (preg_split('/\r?\n+/', $t) as $line) {
        $line = trim($line);
        if ($line === '') continue;

        if (is_vless($line)) {
            $ob = parse_vless($line);
            if ($ob) $col[] = normalize_outbound($ob);
            continue;
        }
        if (is_vmess($line)) {
            $ob = parse_vmess($line);
            if ($ob) $col[] = normalize_outbound($ob);
            continue;
        }

        if (($line[0] ?? '') === '{' || ($line[0] ?? '') === '[') {
            $o = extract_outbounds_from_json($line);
            if ($o) $col = array_merge($col, $o);
        }
    }
    return $col;
}

/** ---------- unique tag ---------- */
function unique_tag(string $base, array &$seen): string {
    $tag = $base;
    $i   = 1;
    while (isset($seen[$tag])) {
        $tag = $base . $i;
        $i++;
    }
    $seen[$tag] = true;
    return $tag;
}

/** ---------- health-check helpers (جدید) ---------- */

/**
 * از outbound، host/port/security/SNI را استخراج می‌کند.
 */
function outbound_get_endpoint(array $ob): ?array {
    $addr = null;
    $port = null;

    if (!empty($ob['settings']['vnext'][0]['address'])) {
        $addr = $ob['settings']['vnext'][0]['address'];
        $port = $ob['settings']['vnext'][0]['port'] ?? null;
    } elseif (!empty($ob['settings']['servers'][0]['address'])) {
        $addr = $ob['settings']['servers'][0]['address'];
        $port = $ob['settings']['servers'][0]['port'] ?? null;
    }

    if ($addr === null || $port === null) {
        return null;
    }

    $security = $ob['streamSettings']['security'] ?? 'none';

    $sni = null;
    if (!empty($ob['streamSettings']['tlsSettings']['serverName'])) {
        $sni = $ob['streamSettings']['tlsSettings']['serverName'];
    } elseif (!empty($ob['streamSettings']['realitySettings']['serverName'])) {
        $sni = $ob['streamSettings']['realitySettings']['serverName'];
    }

    return [
        'host'     => $addr,
        'port'     => (int)$port,
        'security' => strtolower($security),
        'sni'      => $sni,
    ];
}

/**
 * هلث‌چک چندمرحله‌ای:
 * DNS → TCP → (TLS در صورت نیاز)
 */
function outbound_health(array $ob, int $timeout): array {
    $res = [
        'ok'      => false,
        'dns_ok'  => false,
        'tcp_ok'  => false,
        'tls_ok'  => null,
        'endpoint'=> null,
        'rtt_ms'  => null,
        'tries'   => 0,
    ];

    $ep = outbound_get_endpoint($ob);
    if ($ep === null) {
        return $res;
    }
    $res['endpoint'] = $ep;

    $host     = $ep['host'];
    $port     = $ep['port'];
    $security = $ep['security'];
    $sni      = $ep['sni'] ?: $host;

    // ۱) DNS
    $ip = $host;
    if (!filter_var($host, FILTER_VALIDATE_IP)) {
        $dnsStart = microtime(true);
        $ip = @gethostbyname($host);
        $res['tries']++;

        if (empty($ip) || $ip === $host) {
            // DNS fail
            return $res;
        }
        $res['dns_ok'] = true;
        // (در صورت نیاز می‌توان زمان DNS را هم ذخیره کرد)
    } else {
        $res['dns_ok'] = true;
    }

    // ۲) TCP Connect (دو تلاش برای Fail-Tolerance)
    $tcpStart = microtime(true);
    $errno = 0; $errstr = '';
    $sock = @fsockopen($ip, $port, $errno, $errstr, $timeout);
    $res['tries']++;
    if (!$sock) {
        $sock = @fsockopen($ip, $port, $errno, $errstr, $timeout);
        $res['tries']++;
        if (!$sock) {
            return $res;
        }
    }
    $tcpRtt = (microtime(true) - $tcpStart) * 1000.0;
    $res['tcp_ok'] = true;
    $res['rtt_ms'] = (int)round($tcpRtt);
    @fclose($sock);

    // ۳) TLS Handshake در صورت نیاز (TLS / Reality)
    if ($security === 'tls' || $security === 'reality') {
        $tlsCtx = stream_context_create([
            'ssl' => [
                'verify_peer'      => false,
                'verify_peer_name' => false,
                'SNI_enabled'      => true,
                'peer_name'        => $sni,
            ]
        ]);

        $tlsStart = microtime(true);
        $tls = @stream_socket_client(
            "ssl://{$ip}:{$port}",
            $errno,
            $errstr,
            $timeout,
            STREAM_CLIENT_CONNECT,
            $tlsCtx
        );
        $res['tries']++;
        if (!$tls) {
            $res['tls_ok'] = false;
            return $res;
        }
        $tlsRtt = (microtime(true) - $tlsStart) * 1000.0;
        $res['tls_ok'] = true;
        if ($res['rtt_ms'] === null || $tlsRtt < $res['rtt_ms']) {
            $res['rtt_ms'] = (int)round($tlsRtt);
        }
        @fclose($tls);
    }

    // نتیجه نهایی: همه‌ی تست‌های لازم باید OK باشند
    $res['ok'] = $res['dns_ok'] && $res['tcp_ok'] && ($res['tls_ok'] !== false);

    return $res;
}

/** ---------- (قدیمی) checkUrls / HTTP helpers؛ برای سازگاری نگه داشتیم ---------- */
function get_check_urls(array $ob): array {
    $urls = [];
    if (!empty($ob['settings']['checkUrls']) && is_array($ob['settings']['checkUrls'])) {
        foreach ($ob['settings']['checkUrls'] as $u) {
            $u = trim($u); if ($u === '') continue;
            if (!preg_match('~^https?://~i', $u)) $u = 'https://' . $u;
            $urls[] = $u;
        }
    }
    if (empty($urls)) {
        $urls = [
            "https://www.gstatic.com/generate_204",
            "http://cp.cloudflare.com/generate_204",
            "http://detectportal.firefox.com/success.txt",
        ];
    }
    $uniq = [];
    foreach ($urls as $u) {
        $u = preg_replace('~\s+~', '', $u);
        if ($u !== '') $uniq[$u] = true;
    }
    return array_keys($uniq);
}
function add_buster(string $url): string {
    $sep = (strpos($url, '?') === false) ? '?' : '&';
    return $url . $sep . '_cb=' . rawurlencode((string)microtime(true) . '_' . mt_rand(1000,9999));
}
function try_fetch_url_nocache(string $url, int $timeout): bool {
    $url = add_buster($url);
    $opts = [
        'http' => [
            'method'        => 'GET',
            'timeout'       => $timeout,
            'ignore_errors' => true,
            'header' =>
                "Cache-Control: no-cache, no-store, must-revalidate\r\n" .
                "Pragma: no-cache\r\n" .
                "User-Agent: Mozilla/5.0\r\n" .
                "Connection: close\r\n",
        ],
        'ssl' => [
            'verify_peer'      => false,
            'verify_peer_name' => false,
        ],
    ];
    $ctx = stream_context_create($opts);
    $data = @file_get_contents($url, false, $ctx);
    if ($data === false) return false;
    if (strlen($data) === 0) return false;
    return true;
}

/**
 * نسخه‌ی ساده برای سازگاری قدیمی؛ الان فقط از outbound_health استفاده می‌کند.
 */
function outbound_is_alive(array $ob, int $connTimeout, int $httpTimeout, bool $requireAll): bool {
    $h = outbound_health($ob, $connTimeout);
    return $h['ok'];
}

/** ---------- template ---------- */
function build_template(): array {
    global $ENABLE_FAKEDNS;

    $levels = (object)[];
    $levels->{'0'} = [
        "handshake"    => 12,
        "connIdle"     => 3600,
        "uplinkOnly"   => 0,
        "downlinkOnly" => 0,
    ];

    $template = [
        "log" => [
            "loglevel" => "warning",
            "dnsLog"   => false,
        ],

        "stats" => (object)[],

        "api" => [
            "tag"      => "api",
            "services" => ["StatsService"],
        ],

        "policy" => [
            "system" => [
                "statsInboundUplink"   => true,
                "statsInboundDownlink" => true,
            ],
            "levels" => $levels,
        ],

        "dns" => [
            "queryStrategy"         => "UseIP",
            "disableCache"          => false,
            "disableFallback"       => false,
            "disableFallbackIfMatch"=> true,
            "hosts" => [
                "domain:googleapis.cn"         => "googleapis.com",
                "cloudflare-dns.com"           => "1.1.1.1",
                "security.cloudflare-dns.com"  => "1.1.1.1",
                "dns.google"                   => "8.8.8.8",
            ],
            "servers" => [
                [
                    "address"     => "https://1.1.1.1/dns-query",
                    "outboundTag" => "dns-out",
                ],
                [
                    "address"     => "https://8.8.8.8/dns-query",
                    "outboundTag" => "dns-out",
                ],
                [
                    "address"     => "https://9.9.9.9/dns-query",
                    "outboundTag" => "dns-out",
                ],
                // سرورهای DNS با serveStale و timeoutMs
                [
                    "address"          => "1.1.1.1",
                    "outboundTag"     => "dns-out",
                    "timeoutMs"       => 800,
                    "serveStale"      => true,
                    "serveExpiredTTL" => 30,
                ],
                [
                    "address"      => "8.8.8.8",
                    "outboundTag"  => "dns-out",
                    "timeoutMs"    => 800,
                    "serveStale"   => true,
                ],
            ],
        ],

        "observatory" => [
            "subjectSelector"   => ["auto-group"],
            "probeURL"          => "http://detectportal.firefox.com/success.txt",
            "enableConcurrency" => true,
        ],

        "inbounds" => [
            [
                "tag"      => "socks",
                "port"     => 10808,
                "listen"   => "127.0.0.1",
                "protocol" => "socks",
                "sniffing" => [
                    "enabled"      => true,
                    "routeOnly"    => false,
                    "destOverride" => ["http","tls","quic"],
                ],
                "settings" => [
                    "auth"      => "noauth",
                    "udp"       => true,
                    "ip"        => "127.0.0.1",
                    "userLevel" => 0,
                ],
            ],
            [
                "tag"      => "dns-in",
                "port"     => 10853,
                "listen"   => "127.0.0.1",
                "protocol" => "dokodemo-door",
                "settings" => [
                    "network"        => "tcp,udp",
                    "followRedirect" => false,
                    "address"        => "1.1.1.1",
                    "port"           => 53,
                ],
            ],
        ],

        "outbounds" => [
            [
                "tag"      => "direct",
                "protocol" => "freedom",
                "settings" => [
                    "domainStrategy" => "UseIP",
                ],
            ],
            [
                "tag"      => "block",
                "protocol" => "blackhole",
                "settings" => (object)[],
            ],
            [
                "tag"      => "dns-out",
                "protocol" => "dns",
                "settings" => (object)[],
            ],
        ],

        "routing" => [
            "domainStrategy" => "IPIfNonMatch",
            "domainMatcher"  => "hybrid",
            "balancers" => [
                [
                    "tag"      => "auto-group",
                    "selector" => [],
                    "strategy" => ["type" => "leastPing"],
                ],
            ],
            "rules" => [
                [
                    "type"        => "field",
                    "inboundTag"  => ["dns-in"],
                    "outboundTag" => "dns-out",
                ],
                [
                    "type"        => "field",
                    "protocol"    => ["bittorrent"],
                    "outboundTag" => "block",
                ],
                [
                    "type"        => "field",
                    "port"        => "6881-6999",
                    "outboundTag" => "block",
                ],
                [
                    "type"        => "field",
                    "ip"          => [
                        "10.0.0.0/8",
                        "172.16.0.0/12",
                        "192.168.0.0/16",
                        "127.0.0.0/8",
                    ],
                    "outboundTag" => "direct",
                ],
                [
                    "type"        => "field",
                    "ip"          => ["1.1.1.1"],
                    "port"        => "53",
                    "outboundTag" => "dns-out",
                ],
                [
                    "type"        => "field",
                    "domain"      => [
                        "dns.google",
                        "cloudflare-dns.com",
                        "security.cloudflare-dns.com",
                    ],
                    "network"     => "tcp",
                    "balancerTag" => "auto-group",
                ],
                [
                    "type"        => "field",
                    "network"     => "tcp,udp",
                    "balancerTag" => "auto-group",
                ],
            ],
        ],
    ];

    // اگر FakeDNS روشن است، fakedns و destOverride=fakedns را اضافه کن
    if ($ENABLE_FAKEDNS) {
        $template['fakedns'] = [
            [
                "ipPool"   => "198.18.0.0/15",
                "poolSize" => 10000,
            ],
        ];

        if (isset($template['inbounds'][0]['sniffing']['destOverride'])
            && is_array($template['inbounds'][0]['sniffing']['destOverride'])) {
            if (!in_array('fakedns', $template['inbounds'][0]['sniffing']['destOverride'], true)) {
                $template['inbounds'][0]['sniffing']['destOverride'][] = 'fakedns';
            }
        }
    }

    return $template;
}

/**
 * Add advanced observatory config based on outbounds.
 */
function addObservatoryToConfig(array &$jsonConfig, array $preferredSelectors = []) {
    if (!isset($jsonConfig['outbounds']) || !is_array($jsonConfig['outbounds'])) {
        return;
    }

    $selectors = [];

    foreach ($jsonConfig['outbounds'] as $ob) {
        if (!isset($ob['tag'])) continue;
        $tag = $ob['tag'];

        if (in_array($tag, ['direct', 'dns-out', 'block'], true)) {
            continue;
        }

        $selectors[] = $tag;
    }

    $selectors = array_values(array_unique($selectors));

    if (!empty($preferredSelectors)) {
        $preferredSelectors = array_values(array_unique($preferredSelectors));
        $intersection       = array_values(array_intersect($preferredSelectors, $selectors));
        $selectors          = !empty($intersection) ? $intersection : $preferredSelectors;
    }

    if (empty($selectors)) {
        return;
    }

    $jsonConfig['observatory'] = [
        'subjectSelector'   => $selectors,
        'probeURL'          => 'http://detectportal.firefox.com/success.txt',
        'enableConcurrency' => true,
    ];
}

/** ---------- main aggregation (normal + ads) ---------- */

$ALLOWED_PROTOCOLS = ['vless', 'vmess', 'trojan', 'shadowsocks'];

$servers    = [];
$adsServers = [];
$seenTags   = ['direct' => true, 'block' => true, 'dns-out' => true];
$mainTags   = [];
$adsTags    = [];

/* ۱) سرورهای معمولی */
shuffle($LINKS);

foreach ($LINKS as $url) {
    $attempts = 0;
    while ($attempts < 2) {
        $attempts++;

        $raw = http_get_nocache($url, $HTTP_TIMEOUT);
        if ($raw === null) continue;

        $added = smart_extract_outbounds($raw);
        if (!$added) continue;

        foreach ($added as $ob) {
            $proto = strtolower($ob['protocol'] ?? '');
            if ($proto === '' || !in_array($proto, $ALLOWED_PROTOCOLS, true)) continue;
            if (!outbound_is_valid($ob)) continue;

            $ob = add_sockopt_to_outbound($ob);

            // هلث‌چک پیشرفته؛ فقط کانفیگ‌های سالم وارد می‌شوند
            $health = outbound_health($ob, $HEALTH_TIMEOUT);
            if (!$health['ok']) {
                continue;
            }

            $base      = $ob['tag'] ?? ($proto . '_');
            $ob['tag'] = unique_tag(preg_replace('~[^a-zA-Z0-9_\-]~', '_', $base . '_'), $seenTags);

            $ob['__health'] = $health;
            $ob['__ping']   = $health['rtt_ms'] ?? -1;

            $servers[]  = $ob;
            $mainTags[] = $ob['tag'];
        }

        break;
    }
}

/* ۲) سرورهای مخصوص ادز */
if (!empty($ADS_LINKS)) {
    shuffle($ADS_LINKS);

    foreach ($ADS_LINKS as $url) {
        $attempts = 0;
        while ($attempts < 2) {
            $attempts++;

            $raw = http_get_nocache($url, $HTTP_TIMEOUT);
            if ($raw === null) continue;

            $added = smart_extract_outbounds($raw);
            if (!$added) continue;

            foreach ($added as $ob) {
                $proto = strtolower($ob['protocol'] ?? '');
                if ($proto === '' || !in_array($proto, $ALLOWED_PROTOCOLS, true)) continue;
                if (!outbound_is_valid($ob)) continue;

                $ob = add_sockopt_to_outbound($ob);

                // هلث‌چک برای سرورهای Ads
                $health = outbound_health($ob, $HEALTH_TIMEOUT);
                if (!$health['ok']) {
                    continue;
                }

                $base      = $ob['tag'] ?? ($proto . '_');
                $ob['tag'] = unique_tag(preg_replace('~[^a-zA-Z0-9_\-]~', '_', $base . '_'), $seenTags);

                $ob['__health'] = $health;
                $ob['__ping']   = $health['rtt_ms'] ?? -1;

                $adsServers[] = $ob;
                $adsTags[]    = $ob['tag'];
            }

            break;
        }
    }
}

/* ---------- ست کردن allowInsecure روی همه Outbound ها ---------- */
function forceAllowInsecure(&$template) {
    if (!isset($template['outbounds']) || !is_array($template['outbounds'])) {
        return;
    }

    foreach ($template['outbounds'] as &$ob) {

        if (!isset($ob['streamSettings']) || !is_array($ob['streamSettings'])) {
            continue;
        }

        $ss =& $ob['streamSettings'];
        $security = $ss['security'] ?? '';

        /* --- TLS معمولی --- */
        if ($security === 'tls') {
            if (!isset($ss['tlsSettings']) || !is_array($ss['tlsSettings'])) {
                $ss['tlsSettings'] = [];
            }
            $ss['tlsSettings']['allowInsecure'] = true;
        }

        /* --- Reality --- */
        if ($security === 'reality') {
            if (!isset($ss['realitySettings']) || !is_array($ss['realitySettings'])) {
                $ss['realitySettings'] = [];
            }
            $ss['realitySettings']['allowInsecure'] = true;
        }
    }
}

/* ۳) ساخت template نهایی */
$template = build_template();

foreach ($servers as $ob) {
    $template['outbounds'][] = $ob;
}
foreach ($adsServers as $ob) {
    $template['outbounds'][] = $ob;
}

/* ست کردن allowInsecure روی همه Outbound ها */
forceAllowInsecure($template);

/* تنظیم بالانسرها */
if (!empty($mainTags)) {
    $template['routing']['balancers'][0]['selector'] = $mainTags;
} else {
    $template['routing']['balancers'][0]['selector'] = $adsTags;
}

if (!empty($adsTags)) {
    $template['routing']['balancers'][] = [
        'tag'      => 'auto-ads',
        'selector' => $adsTags,
        'strategy' => ['type' => 'leastPing'],
    ];
}

/* observatory برای همه سرورها (معمولی + ادز) */
$allTags = array_values(array_unique(array_merge($mainTags, $adsTags)));
addObservatoryToConfig($template, $allTags);

/* meta */
$template['__meta'] = [
    'health'  => ['requireAll' => CHECK_REQUIRE_ALL],
    'network' => ['ipv6Blocked' => false],
];

/* ۴) قوانین روتینگ برای ترافیک گوگل / یوتیوب / ادز → auto-ads */
if (!empty($adsTags)) {
    $googleDomains = [
        "googleads.g.doubleclick.net",
        "geosite:category-ads-all",
        "geosite:google",
        "pagead2.googlesyndication.com",
        "tpc.googlesyndication.com",
        "www.googleadservices.com",
        "adservice.google.com",
        "adservice.google.*",
        "stats.g.doubleclick.net",

        "www.googletagmanager.com",
        "www.google-analytics.com",
        "ssl.google-analytics.com",
        "app-measurement.com",
        "www.googletagservices.com",

        "fcm.googleapis.com",
        "firebaseinstallations.googleapis.com",
        "firebaseremoteconfig.googleapis.com",
        "crashlytics.googleapis.com",

        "play.google.com",
        "play.googleapis.com",
        "android.googleapis.com",
        "android.clients.google.com",

        "www.google.com",
        "www.googleapis.com",
        "oauth2.googleapis.com",
        "accounts.google.com",
        "gstaticadssl.l.google.com",
        "fonts.gstatic.com",
        "www.gstatic.com",
        "clients3.google.com",
        "clients4.google.com",
        "clients5.google.com",

        "www.youtube.com",
        "m.youtube.com",
        "youtubei.googleapis.com",
        "i.ytimg.com",
        "s.youtube.com",
        "yt3.ggpht.com",
        "googlevideo.com",
        "r.googlevideo.com",

        "google.com",
        "googleapis.com",
        "googleusercontent.com",
        "gvt1.com",
        "gvt2.com",
        "gstatic.com",
    ];

    $rules    = $template['routing']['rules'] ?? [];
    $newRules = [];

    // ۱) تمام دامین‌های گوگل / یوتیوب / Ads / Firebase / Play → auto-ads
    $newRules[] = [
        "type"        => "field",
        "network"     => "tcp,udp",
        "domain"      => $googleDomains,
        "balancerTag" => "auto-ads",
    ];

    // ۲) بقیه قوانین قبلی (شامل fallback → auto-group) بدون تغییر
    foreach ($rules as $rule) {
        $newRules[] = $rule;
    }

    $template['routing']['rules'] = $newRules;
}

echo json_encode($template, JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE);