<?php
/**
 * Nexy.php — v2ray/Xray config aggregator (NO cache, strict health, keep outbounds AS-IS)
 * PHP 7.4+
 *
 * ✅ ویژگی‌های اضافه‌شده (بدون حذف منطق قبلی):
 * - Endpoint فوری: همیشه از کش چاپ می‌کند (current یا last_good) و همزمان worker را بک‌گراند تریگر می‌کند
 * - Worker/Daemon داخل همین فایل: دانلود موازی لینک‌ها با timeout=5s، سقف 100 کانفیگ در هر 5 دقیقه، سقف 300 غیرتکراری در کش
 * - کش هر کانفیگ 15 دقیقه (TTL)
 * - لاگ کامل + قفل اجرای همزمان + self-heal
 *
 * Modes:
 * - HTTP:   /Nexy.php                => فوری چاپ از cache + تریگر worker
 * - HTTP:   /Nexy.php?mode=worker    => اجرای worker یک دور
 * - HTTP:   /Nexy.php?mode=status    => وضعیت (lock/cache/last run)
 * - CLI:    php Nexy.php --worker
 * - CLI:    php Nexy.php --daemon
 */

/* ====== USER SETTINGS ====== */
$LINKS = [
    'https://fitn1.ir/Api/Nest/trunsp.php',
    'https://fitn1.ir/Api/Speedray/trunsp.php',
    'https://fitn1.ir/Api/kaya/trunsp.php',
    'https://fitn1.ir/Api/Bionet/trunsp.php',
    'https://fitn1.ir/Api/Rosa/Rosa.php',
    'https://fitn1.ir/Api/Verde/trun.php',
    'https://fitn1.ir/Api/Bionet/trunsp.php',
    'https://fitn1.ir/Api/Asgard/trunsp.php',
    'https://fitn1.ir/Api/Ash/trun.php',
    'https://fitn1.ir/Api/AAA/Ss.php',
    'https://fitn1.ir/Api/Cap/trun.php',
    'https://fitn1.ir/Api/Rosa/Rosa.php'
];

// توجه: در Worker دانلود لینک‌ها با timeout=5s انجام می‌شود (طبق درخواست شما)
$HTTP_TIMEOUT   = 10;  // (برای سازگاری با کد فعلی) seconds for pulling raw configs
$HEALTH_TIMEOUT = 4;   // seconds for health-check

// سوییچ FakeDNS (برای روشن/خاموش کردن استفاده از FakeDNS و destOverride=fakedns)
$ENABLE_FAKEDNS = true;

define('CHECK_REQUIRE_ALL', false);

/* ====== NEW SETTINGS (Worker/Cache/Daemon) ====== */
define('NEXY_DATA_DIR', __DIR__ . '/data');
define('NEXY_CACHE_DIR', NEXY_DATA_DIR . '/cache');
define('NEXY_LOG_DIR',   NEXY_DATA_DIR . '/logs');

define('NEXY_CURRENT_FILE',   NEXY_CACHE_DIR . '/current.json');
define('NEXY_LASTGOOD_FILE',  NEXY_CACHE_DIR . '/last_good.json');
define('NEXY_ITEMS_FILE',     NEXY_CACHE_DIR . '/items.json');
define('NEXY_WINDOW_FILE',    NEXY_CACHE_DIR . '/window.json');
define('NEXY_STATE_FILE',     NEXY_CACHE_DIR . '/state.json');
define('NEXY_LOCK_FILE',      NEXY_CACHE_DIR . '/worker.lock');

define('NEXY_FETCH_TIMEOUT',  5);    // ✅ timeout هر لینک (ثانیه)
define('NEXY_FETCH_CONN_TIMEOUT', 3);
define('NEXY_FETCH_CONCURRENCY', 16);

define('NEXY_ITEM_TTL',       900);  // ✅ 15 دقیقه
define('NEXY_MAX_ITEMS',      300);  // ✅ سقف کش غیرتکراری
define('NEXY_MAX_NEW_PER_WINDOW', 100); // ✅ سقف جمع‌آوری در هر 5 دقیقه
define('NEXY_WINDOW_SECONDS', 300); // ✅ 5 دقیقه
define('NEXY_DAEMON_SLEEP',   8);   // برای daemon
define('NEXY_LOCK_TTL',       180); // lock self-heal (ثانیه)

/* =========================
   BOOT: dirs + router
   ========================= */
nexy_ensure_dirs();

/* ===== CLI Router ===== */
if (PHP_SAPI === 'cli') {
    $argv = $_SERVER['argv'] ?? [];
    if (in_array('--worker', $argv, true)) {
        nexy_log("CLI worker start");
        nexy_worker_run_once();
        nexy_log("CLI worker end");
        exit(0);
    }
    if (in_array('--daemon', $argv, true)) {
        nexy_log("CLI daemon start");
        nexy_daemon_loop();
        exit(0);
    }
    // default CLI: print status
    echo json_encode(nexy_status_payload(), JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT);
    exit(0);
}

/* ===== HTTP Router ===== */
$mode = isset($_GET['mode']) ? (string)$_GET['mode'] : '';
if ($mode === 'worker') {
    header('Content-Type: application/json; charset=utf-8');
    $out = nexy_worker_run_once();
    echo json_encode($out, JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE);
    exit;
}
if ($mode === 'status') {
    header('Content-Type: application/json; charset=utf-8');
    echo json_encode(nexy_status_payload(), JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE);
    exit;
}

/* ===== Default endpoint: فوری چاپ از کش + تریگر بک‌گراند ===== */
header('Content-Type: application/json; charset=utf-8');
nexy_endpoint_serve_fast_and_trigger_worker();
exit;

/* ===================================================================
   ======================= NEW: Worker/Cache/Daemon ===================
   =================================================================== */

function nexy_ensure_dirs(): void {
    foreach ([NEXY_DATA_DIR, NEXY_CACHE_DIR, NEXY_LOG_DIR] as $d) {
        if (!is_dir($d)) @mkdir($d, 0777, true);
    }
    if (!file_exists(NEXY_ITEMS_FILE)) nexy_write_json_atomic(NEXY_ITEMS_FILE, ['items' => []]);
    if (!file_exists(NEXY_STATE_FILE)) nexy_write_json_atomic(NEXY_STATE_FILE, ['last_worker' => 0, 'last_ok' => 0, 'last_error' => 0, 'last_msg' => '']);
}

function nexy_log(string $msg, array $ctx = []): void {
    $ts = date('Y-m-d H:i:s');
    $pid = function_exists('getmypid') ? getmypid() : 0;
    $line = "[$ts][$pid] $msg";
    if (!empty($ctx)) $line .= " " . json_encode($ctx, JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE);
    $line .= "\n";
    @file_put_contents(NEXY_LOG_DIR . '/nexy.log', $line, FILE_APPEND);
}

function nexy_read_json(string $path, $default) {
    $raw = @file_get_contents($path);
    if ($raw === false || $raw === '') return $default;
    $j = json_decode($raw, true);
    return is_array($j) ? $j : $default;
}

function nexy_write_json_atomic(string $path, $data): bool {
    $tmp = $path . '.tmp.' . (function_exists('getmypid') ? getmypid() : 0) . '.' . mt_rand(1000,9999);
    $raw = json_encode($data, JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE);
    if ($raw === false) return false;
    if (@file_put_contents($tmp, $raw) === false) return false;
    return @rename($tmp, $path);
}

function nexy_now(): int { return time(); }

function nexy_lock_acquire(&$fh = null): bool {
    if (file_exists(NEXY_LOCK_FILE)) {
        $st = @stat(NEXY_LOCK_FILE);
        if ($st && (nexy_now() - (int)$st['mtime']) > NEXY_LOCK_TTL) {
            @unlink(NEXY_LOCK_FILE);
        }
    }
    $fh = @fopen(NEXY_LOCK_FILE, 'c+');
    if (!$fh) return false;
    if (!@flock($fh, LOCK_EX | LOCK_NB)) {
        return false;
    }
    @ftruncate($fh, 0);
    @fwrite($fh, (string)nexy_now());
    @fflush($fh);
    return true;
}

function nexy_lock_release($fh): void {
    if ($fh) {
        @flock($fh, LOCK_UN);
        @fclose($fh);
    }
}

function nexy_status_payload(): array {
    $state = nexy_read_json(NEXY_STATE_FILE, []);
    $items = nexy_read_json(NEXY_ITEMS_FILE, ['items' => []]);
    $cnt = isset($items['items']) && is_array($items['items']) ? count($items['items']) : 0;
    return [
        'time' => nexy_now(),
        'lock_exists' => file_exists(NEXY_LOCK_FILE),
        'current_exists' => file_exists(NEXY_CURRENT_FILE),
        'last_good_exists' => file_exists(NEXY_LASTGOOD_FILE),
        'items_count' => $cnt,
        'state' => $state,
    ];
}

function nexy_endpoint_serve_fast_and_trigger_worker(): void {
    $payload = null;

    $cur = @file_get_contents(NEXY_CURRENT_FILE);
    if ($cur !== false && $cur !== '') {
        $payload = $cur;
    } else {
        $lg = @file_get_contents(NEXY_LASTGOOD_FILE);
        if ($lg !== false && $lg !== '') {
            $payload = $lg;
        }
    }

    if ($payload === null) {
        $template = build_template();
        $template['__meta']['bootstrap'] = true;
        $payload = json_encode($template, JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE);
    }

    echo $payload;

    if (function_exists('fastcgi_finish_request')) {
        @fastcgi_finish_request();
    } else {
        @ob_flush(); @flush();
    }

    nexy_trigger_worker_background();
}

function nexy_trigger_worker_background(): void {
    $fh = null;
    if (!nexy_lock_acquire($fh)) {
        return;
    }
    nexy_lock_release($fh);

    $php = defined('PHP_BINARY') ? PHP_BINARY : 'php';
    $cmd = $php . ' ' . escapeshellarg(__FILE__) . ' --worker';
    $okExec = false;

    if (function_exists('exec')) {
        @exec($cmd . ' > /dev/null 2>&1 &', $o, $rc);
        $okExec = true;
    } elseif (function_exists('shell_exec')) {
        @shell_exec($cmd . ' > /dev/null 2>&1 &');
        $okExec = true;
    }

    if ($okExec) {
        nexy_log("Triggered worker via CLI background");
        return;
    }

    $self = nexy_self_url();
    $url = $self . (strpos($self, '?') === false ? '?' : '&') . 'mode=worker&cb=' . rawurlencode((string)microtime(true));

    if (function_exists('curl_init')) {
        $ch = curl_init($url);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_TIMEOUT_MS, 300);
        curl_setopt($ch, CURLOPT_CONNECTTIMEOUT_MS, 200);
        curl_setopt($ch, CURLOPT_NOSIGNAL, true);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
        curl_setopt($ch, CURLOPT_SSL_VERIFYHOST, false);
        curl_setopt($ch, CURLOPT_HTTPHEADER, ['Connection: close', 'Cache-Control: no-cache']);
        @curl_exec($ch);
        @curl_close($ch);
        nexy_log("Triggered worker via HTTP curl (non-blocking)");
        return;
    }

    $parts = parse_url($url);
    if (!$parts || empty($parts['host'])) return;
    $scheme = $parts['scheme'] ?? 'http';
    $host = $parts['host'];
    $port = isset($parts['port']) ? (int)$parts['port'] : (($scheme === 'https') ? 443 : 80);
    $path = ($parts['path'] ?? '/') . (isset($parts['query']) ? ('?' . $parts['query']) : '');

    $fp = @fsockopen(($scheme === 'https' ? 'ssl://' : '') . $host, $port, $errno, $errstr, 0.2);
    if ($fp) {
        $req = "GET $path HTTP/1.1\r\nHost: $host\r\nConnection: close\r\nCache-Control: no-cache\r\n\r\n";
        @fwrite($fp, $req);
        @fclose($fp);
        nexy_log("Triggered worker via raw socket");
    }
}

function nexy_self_url(): string {
    $https = (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off');
    $scheme = $https ? 'https' : 'http';
    $host = $_SERVER['HTTP_HOST'] ?? 'localhost';
    $uri  = $_SERVER['SCRIPT_NAME'] ?? '';
    return $scheme . '://' . $host . $uri;
}

/**
 * ✅ تنها تابع مصرف Window (فقط یک بار)
 */
function nexy_window_take(int $want): int {
    $w = nexy_read_json(NEXY_WINDOW_FILE, ['start' => 0, 'count' => 0]);
    $now = nexy_now();

    if (empty($w['start']) || ($now - (int)$w['start']) >= NEXY_WINDOW_SECONDS) {
        $w['start'] = $now;
        $w['count'] = 0;
    }

    $remain = max(0, NEXY_MAX_NEW_PER_WINDOW - (int)$w['count']);
    $take = min($want, $remain);

    $w['count'] = (int)$w['count'] + $take;
    nexy_write_json_atomic(NEXY_WINDOW_FILE, $w);
    return $take;
}

function nexy_items_load(): array {
    $j = nexy_read_json(NEXY_ITEMS_FILE, ['items' => []]);
    if (!isset($j['items']) || !is_array($j['items'])) $j['items'] = [];
    return $j['items'];
}

function nexy_items_save(array $items): void {
    nexy_write_json_atomic(NEXY_ITEMS_FILE, ['items' => $items]);
}

function nexy_items_prune(array &$items): void {
    $now = nexy_now();
    foreach ($items as $fp => $it) {
        $t = (int)($it['t'] ?? 0);
        if ($t <= 0 || ($now - $t) > NEXY_ITEM_TTL) {
            unset($items[$fp]);
        }
    }
    if (count($items) > NEXY_MAX_ITEMS) {
        uasort($items, function($a, $b) {
            return (int)($a['t'] ?? 0) <=> (int)($b['t'] ?? 0);
        });
        while (count($items) > NEXY_MAX_ITEMS) {
            $k = array_key_first($items);
            if ($k === null) break;
            unset($items[$k]);
        }
    }
}

function nexy_fingerprint_outbound(array $ob): string {
    $x = $ob;
    unset($x['tag'], $x['__health'], $x['__ping']);
    nexy_deep_ksort($x);
    $raw = json_encode($x, JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE);
    return sha1((string)$raw);
}

function nexy_deep_ksort(&$a): void {
    if (!is_array($a)) return;
    ksort($a);
    foreach ($a as &$v) {
        if (is_array($v)) nexy_deep_ksort($v);
    }
}

/**
 * ✅ Worker (FIXED)
 */
function nexy_worker_run_once(): array {
    global $LINKS, $HEALTH_TIMEOUT;

    $fh = null;
    if (!nexy_lock_acquire($fh)) {
        return ['ok' => true, 'skipped' => true, 'reason' => 'worker_running', 'status' => nexy_status_payload()];
    }

    $state = nexy_read_json(NEXY_STATE_FILE, ['last_worker' => 0, 'last_ok' => 0, 'last_error' => 0, 'last_msg' => '']);
    $state['last_worker'] = nexy_now();
    nexy_write_json_atomic(NEXY_STATE_FILE, $state);

    $result = [
        'ok' => false,
        'added' => 0,
        'kept' => 0,
        'pruned' => 0,
        'out_count' => 0,
        'msg' => '',
    ];

    try {
        nexy_log("Worker run_once begin", ['links' => count($LINKS)]);

        $fetched = nexy_curl_multi_fetch($LINKS, NEXY_FETCH_TIMEOUT, NEXY_FETCH_CONN_TIMEOUT, NEXY_FETCH_CONCURRENCY);

        $items = nexy_items_load();
        $before = count($items);
        nexy_items_prune($items);

        // ✅ فقط یک بار budget
        $newBudget = nexy_window_take(NEXY_MAX_NEW_PER_WINDOW);

        $added = 0;

        foreach ($fetched as $url => $raw) {
            if ($raw === null || $raw === '') continue;

            $outs = smart_extract_outbounds($raw);
            if (!$outs) continue;

            foreach ($outs as $ob) {
                $proto = strtolower($ob['protocol'] ?? '');
                if ($proto === '') continue;

                $ob = normalize_outbound($ob);
                $ob = add_sockopt_to_outbound($ob);

                $health = outbound_health($ob, $HEALTH_TIMEOUT);
                if (!$health['ok']) continue;

                $fp = nexy_fingerprint_outbound($ob);

                if (isset($items[$fp])) {
                    $items[$fp]['t'] = nexy_now();
                    continue;
                }

                if ($newBudget <= 0) {
                    continue;
                }

                $items[$fp] = [
                    't'  => nexy_now(),
                    'ob' => $ob,
                    'ep' => $health['endpoint'] ?? null,
                    'rtt' => $health['rtt_ms'] ?? null,
                ];

                $added++;
                $newBudget--;

                if (count($items) >= NEXY_MAX_ITEMS) break 2;
            }
        }

        nexy_items_prune($items);
        $after = count($items);

        $result['added'] = $added;
        $result['kept'] = $after;
        $result['pruned'] = max(0, $before - $after);

        $cfg = nexy_build_config_from_items($items);

        if (isset($cfg['outbounds']) && is_array($cfg['outbounds']) && count($cfg['outbounds']) > 0) {
            nexy_write_json_atomic(NEXY_CURRENT_FILE, $cfg);
            nexy_write_json_atomic(NEXY_LASTGOOD_FILE, $cfg);
            $state['last_ok'] = nexy_now();
            $state['last_msg'] = "ok; added=$added kept=$after";
            nexy_write_json_atomic(NEXY_STATE_FILE, $state);
        } else {
            $state['last_error'] = nexy_now();
            $state['last_msg'] = "warning: built empty outbounds; kept last_good";
            nexy_write_json_atomic(NEXY_STATE_FILE, $state);
        }

        $result['out_count'] = isset($cfg['outbounds']) && is_array($cfg['outbounds']) ? count($cfg['outbounds']) : 0;
        $result['ok'] = true;
        $result['msg'] = "worker ok";
        nexy_log("Worker done", $result);

        nexy_items_save($items);

    } catch (\Throwable $e) {
        $result['ok'] = false;
        $result['msg'] = "worker error: " . $e->getMessage();
        $state['last_error'] = nexy_now();
        $state['last_msg'] = $result['msg'];
        nexy_write_json_atomic(NEXY_STATE_FILE, $state);
        nexy_log("Worker exception", ['err' => $e->getMessage(), 'line' => $e->getLine()]);
    } finally {
        nexy_lock_release($fh);
    }

    return $result;
}

function nexy_daemon_loop(): void {
    while (true) {
        try {
            nexy_worker_run_once();
        } catch (\Throwable $e) {
            nexy_log("Daemon loop exception", ['err' => $e->getMessage()]);
        }
        sleep(NEXY_DAEMON_SLEEP);
    }
}

function nexy_curl_multi_fetch(array $urls, int $timeoutSec, int $connTimeoutSec, int $concurrency): array {
    $out = [];
    foreach ($urls as $u) $out[$u] = null;

    if (!function_exists('curl_multi_init')) {
        foreach ($urls as $u) {
            $ctx = stream_context_create([
                'http' => ['timeout' => $timeoutSec, 'ignore_errors' => true, 'header' => "Cache-Control: no-cache\r\nUser-Agent: Mozilla/5.0\r\nConnection: close\r\n"],
                'ssl'  => ['verify_peer' => false, 'verify_peer_name' => false],
            ]);
            $d = @file_get_contents($u, false, $ctx);
            $out[$u] = ($d === false || $d === '') ? null : $d;
        }
        return $out;
    }

    $mh = curl_multi_init();
    $handles = [];
    $queue = array_values($urls);
    $active = 0;

    $addOne = function(string $url) use (&$mh, &$handles, $timeoutSec, $connTimeoutSec) {
        $ch = curl_init($url);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_TIMEOUT, $timeoutSec);
        curl_setopt($ch, CURLOPT_CONNECTTIMEOUT, $connTimeoutSec);
        curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
        curl_setopt($ch, CURLOPT_MAXREDIRS, 3);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
        curl_setopt($ch, CURLOPT_SSL_VERIFYHOST, false);
        curl_setopt($ch, CURLOPT_HTTPHEADER, [
            'Cache-Control: no-cache, no-store, must-revalidate',
            'Pragma: no-cache',
            'Connection: close',
            'User-Agent: Mozilla/5.0'
        ]);
        curl_multi_add_handle($mh, $ch);
        $handles[(int)$ch] = ['ch' => $ch, 'url' => $url];
    };

    while (!empty($queue) && count($handles) < $concurrency) {
        $addOne(array_shift($queue));
    }

    do {
        do { $mrc = curl_multi_exec($mh, $active); } while ($mrc === CURLM_CALL_MULTI_PERFORM);

        while ($info = curl_multi_info_read($mh)) {
            $ch = $info['handle'];
            $key = (int)$ch;
            $url = $handles[$key]['url'] ?? '';
            $data = curl_multi_getcontent($ch);
            $code = (int)curl_getinfo($ch, CURLINFO_HTTP_CODE);
            $err  = curl_error($ch);

            if ($info['result'] === CURLE_OK && $code >= 200 && $code < 500 && $data !== '') {
                $out[$url] = $data;
            } else {
                $out[$url] = null;
                nexy_log("Fetch fail", ['url' => $url, 'code' => $code, 'err' => $err]);
            }

            curl_multi_remove_handle($mh, $ch);
            curl_close($ch);
            unset($handles[$key]);

            if (!empty($queue)) {
                $addOne(array_shift($queue));
            }
        }

        if ($active) curl_multi_select($mh, 0.5);

    } while ($active || !empty($handles));

    curl_multi_close($mh);
    return $out;
}

function nexy_build_config_from_items(array $items): array {
    $servers = [];
    $seenTags = ['direct' => true, 'block' => true, 'dns-out' => true];
    $mainTags = [];

    $list = [];
    foreach ($items as $fp => $it) {
        if (!isset($it['ob']) || !is_array($it['ob'])) continue;
        $ob = $it['ob'];
        $rtt = (int)($it['rtt'] ?? 999999);
        $list[] = ['fp' => $fp, 'rtt' => $rtt, 'ob' => $ob, 'ep' => ($it['ep'] ?? null)];
    }
    usort($list, function($a, $b) { return $a['rtt'] <=> $b['rtt']; });

    $take = 0;
    foreach ($list as $row) {
        if ($take >= NEXY_MAX_ITEMS) break;
        $ob = $row['ob'];

        $proto = strtolower($ob['protocol'] ?? 'server');
        $base  = $ob['tag'] ?? ($proto . '_');
        $ob['tag'] = unique_tag(preg_replace('~[^a-zA-Z0-9_\-]~', '_', $base . '_'), $seenTags);

        $ob['__health'] = [
            'endpoint' => $row['ep'],
            'rtt_ms'   => $row['rtt'],
        ];
        $ob['__ping'] = $row['rtt'];

        $servers[] = $ob;
        $mainTags[] = $ob['tag'];
        $take++;
    }

    $template = build_template();

    foreach ($servers as $ob) {
        $template['outbounds'][] = $ob;
    }

    forceAllowInsecure($template);

    if (!empty($mainTags)) {
        $template['routing']['balancers'][0]['selector'] = $mainTags;
    }

    $allTags = array_values(array_unique($mainTags));
    addObservatoryToConfig($template, $allTags);

    if (!isset($template['__meta']) || !is_array($template['__meta'])) $template['__meta'] = [];
    $template['__meta']['health'] = ['requireAll' => CHECK_REQUIRE_ALL];
    $template['__meta']['cache'] = [
        'ttl_sec' => NEXY_ITEM_TTL,
        'max_items' => NEXY_MAX_ITEMS,
        'max_new_per_5min' => NEXY_MAX_NEW_PER_WINDOW,
    ];
    $template['__meta']['worker'] = nexy_read_json(NEXY_STATE_FILE, []);

    return $template;
}

/* ===================================================================
   ======================= ORIGINAL CODE (as-is) =======================
   =================================================================== */

/** ---------- HTTP fetch (no cache) ---------- */
function http_get_nocache(string $url, int $timeout): ?string {
    $ctx = stream_context_create([
        'http' => [
            'timeout' => $timeout,
            'ignore_errors' => true,
            'header' =>
                "Cache-Control: no-cache, no-store, must-revalidate\r\n" .
                "Pragma: no-cache\r\n" .
                "User-Agent: Mozilla/5.0\r\n" .
                "Connection: close\r\n",
        ],
        'ssl'  => [
            'verify_peer' => false,
            'verify_peer_name' => false,
        ],
    ]);

    $data = @file_get_contents($url, false, $ctx);
    return ($data === false || $data === '') ? null : $data;
}

/** ---------- small helpers ---------- */
function is_vless(string $s): bool { return stripos($s, 'vless://') === 0; }
function is_vmess(string $s): bool { return stripos($s, 'vmess://') === 0; }

function remove_nulls(&$arr) {
    if (!is_array($arr)) return;
    foreach ($arr as $k => $v) {
        if (is_array($v)) {
            remove_nulls($arr[$k]);
        } elseif ($v === null) {
            unset($arr[$k]);
        }
    }
}

/** ---------- VLESS parser ---------- */
function parse_vless(string $uri): ?array {
    $p = parse_url($uri);
    if (!$p || empty($p['host']) || empty($p['user'])) return null;

    $host = $p['host'];
    $port = isset($p['port']) ? intval($p['port']) : 443;
    $uuid = $p['user'];
    $tag  = $p['fragment'] ?? 'server';

    $q = [];
    if (isset($p['query'])) parse_str($p['query'], $q);

    $network  = isset($q['type']) ? strtolower($q['type']) : 'tcp';
    $security = isset($q['security']) ? strtolower($q['security']) : 'none';

    $user = ['id' => $uuid, 'encryption' => 'none'];
    if (!empty($q['flow'])) $user['flow'] = $q['flow'];

    $out = [
        'tag' => $tag,
        'protocol' => 'vless',
        'settings' => [
            'vnext' => [[
                'address' => $host,
                'port'    => $port,
                'users'   => [ $user ]
            ]]
        ],
        'streamSettings' => [
            'network'  => $network,
            'security' => $security
        ],
        'mux' => [ 'enabled' => false ],
    ];

    if (isset($q['type'])) {
        if ($q['type'] === 'ws') {
            $path       = $q['path'] ?? '/';
            $hostHeader = $q['host'] ?? $host;
            $out['streamSettings']['wsSettings'] = [
                'path'    => $path,
                'headers' => ['Host' => $hostHeader],
            ];
        } elseif ($q['type'] === 'tcp') {
            if (!empty($q['headerType']) && $q['headerType'] === 'http') {
                $out['streamSettings']['tcpSettings'] = [
                    'header' => [
                        'type'    => 'http',
                        'request' => [
                            'version' => '1.1',
                            'method'  => 'HEAD',
                            'path'    => ['/'],
                            'headers' => [
                                'Host'            => [$q['host'] ?? 'example.com'],
                                'Connection'      => ['keep-alive'],
                                'Accept-Encoding' => ['gzip, deflate, br'],
                                'User-Agent'      => [
                                    'Mozilla/5.0 (Linux; Android 14; Pixel 7 Pro Build/XXXXX; wv) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/129.0.0.0 Mobile Safari/537.36'
                                ]
                            ]
                        ]
                    ]
                ];
            } else {
                $out['streamSettings']['tcpSettings'] = [ 'header' => [ 'type' => 'none' ] ];
            }
        } elseif ($q['type'] === 'grpc') {
            $out['streamSettings']['grpcSettings'] = [
                'serviceName' => $q['serviceName'] ?? 'grpc',
                'multiMode'   => false
            ];
        }
    }

    if (isset($q['security'])) {
        if ($q['security'] === 'reality') {
            $re = [
                'show'        => false,
                'fingerprint' => $q['fp'] ?? 'chrome',
                'serverName'  => $q['sni'] ?? $host,
                'publicKey'   => $q['pbk'] ?? '',
                'shortId'     => $q['sid'] ?? '',
                'spiderX'     => $q['spx'] ?? '/',
                'allowInsecure' => false
            ];
            if (!empty($q['pbk'])) $re['publicKey'] = $q['pbk'];
            if (!empty($q['sid'])) $re['shortId']   = $q['sid'];
            if (!empty($q['spx'])) $re['spiderX']   = $q['spx'];
            $out['streamSettings']['realitySettings'] = $re;
        } elseif ($security === 'tls') {
            $out['streamSettings']['tlsSettings'] = [
                'allowInsecure' => false,
                'fingerprint'   => $q['fp'] ?? 'chrome',
                'serverName'    => $q['sni'] ?? $host
            ];
        } else {
            $out['streamSettings']['security'] = 'none';
        }
    }

    if ($network === 'tcp') {
        if (!empty($q['headerType']) && $q['headerType'] === 'http') {
            $out['streamSettings']['tcpSettings'] = [
                'header' => [
                    'type'    => 'http',
                    'request' => [
                        'version' => '1.1',
                        'method'  => 'HEAD',
                        'path'    => ['/'],
                        'headers' => [
                            'Host'            => [$q['host'] ?? 'example.com'],
                            'Connection'      => ['keep-alive'],
                            'Accept-Encoding' => ['gzip, deflate, br'],
                            'User-Agent'      => [
                                'Mozilla/5.0 (Linux; Android 14; Pixel 7 Pro Build/XXXXX; wv) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/129.0.0.0 Mobile Safari/537.36'
                            ]
                        ]
                    ]
                ]
            ];
        } else {
            $out['streamSettings']['tcpSettings'] = [ 'header' => [ 'type' => 'none' ] ];
        }
    }

    remove_nulls($out);
    return $out;
}

/** ---------- VMESS parser ---------- */
function parse_vmess(string $uri): ?array {
    $b64  = substr($uri, 8);
    $json = base64_decode(strtr($b64, '-_', '+/'), true);
    if ($json === false) return null;
    $o = json_decode($json, true);
    if (!is_array($o) || empty($o['add']) || empty($o['id']) || empty($o['port'])) return null;

    $host = $o['add'];
    $port = intval($o['port']);
    $uuid = $o['id'];
    $tag  = $o['ps'] ?? 'vmess';

    $network  = strtolower($o['net'] ?? 'tcp');
    $security = !empty($o['tls']) ? 'tls' : 'none';

    $user = [
        'id'         => $uuid,
        'alterId'    => 0,
        'security'   => 'auto',
        'encryption' => 'none'
    ];

    $out = [
        'tag'       => $tag,
        'protocol'  => 'vmess',
        'settings'  => [
            'vnext' => [[
                'address' => $host,
                'port'    => $port,
                'users'   => [ $user ]
            ]]
        ],
        'streamSettings' => [
            'network'  => $network,
            'security' => $security
        ],
        'mux' => ['enabled' => false],
    ];

    if ($network === 'ws') {
        $path       = $o['path'] ?? '/';
        $hostHeader = $o['host'] ?? $host;
        $out['streamSettings']['wsSettings'] = [
            'path'    => $path,
            'headers' => ['Host' => $hostHeader],
        ];
    } elseif ($network === 'grpc') {
        $out['streamSettings']['grpcSettings'] = [
            'serviceName' => $o['path'] ?? 'grpc',
            'multiMode'   => false
        ];
    } elseif ($network === 'tcp') {
        $out['streamSettings']['tcpSettings'] = [ 'header' => ['type' => 'none'] ];
    }

    if ($security === 'tls') {
        $out['streamSettings']['tlsSettings'] = [
            'allowInsecure' => false,
            'serverName'    => $o['sni'] ?? ($o['host'] ?? $host),
            'fingerprint'   => 'chrome'
        ];
    }

    remove_nulls($out);
    return $out;
}

/** ---------- normalize outbound ---------- */
function normalize_outbound(array $ob): array {
    if (empty($ob['protocol']) && !empty($ob['proto'])) {
        $ob['protocol'] = $ob['proto'];
    }
    if (!isset($ob['settings']) || !is_array($ob['settings'])) {
        $ob['settings'] = [];
    }
    if (!isset($ob['streamSettings']) || !is_array($ob['streamSettings'])) {
        $ob['streamSettings'] = [];
    }
    if (!isset($ob['streamSettings']['network'])) {
        $ob['streamSettings']['network'] = 'tcp';
    }
    if (!isset($ob['streamSettings']['security'])) {
        $ob['streamSettings']['security'] = 'none';
    }
    if (!isset($ob['mux']) || !is_array($ob['mux'])) {
        $ob['mux'] = ['enabled' => false];
    }
    if (!isset($ob['tag']) || $ob['tag'] === '') {
        $ob['tag'] = $ob['protocol'] ?? 'server';
    }

    remove_nulls($ob);
    return $ob;
}

/** ---------- sockopt + padding + TLS helper ---------- */
function add_sockopt_to_outbound(array $ob): array {
    if (!isset($ob['streamSettings']) || !is_array($ob['streamSettings'])) {
        $ob['streamSettings'] = [];
    }

    if (!isset($ob['streamSettings']['padding']) || !is_array($ob['streamSettings']['padding'])) {
        $ob['streamSettings']['padding'] = [
            'enabled' => false,
        ];
    }

    if (!isset($ob['streamSettings']['sockopt']) || !is_array($ob['streamSettings']['sockopt'])) {
        $ob['streamSettings']['sockopt'] = [];
    }
    $sock = &$ob['streamSettings']['sockopt'];

    if (!array_key_exists('tcpFastOpen', $sock))          $sock['tcpFastOpen']        = true;
    if (!array_key_exists('tcpNoDelay', $sock))           $sock['tcpNoDelay']         = true;
    if (!array_key_exists('mtu', $sock))                  $sock['mtu']                = 1280;
    if (!array_key_exists('tcpKeepAliveIdle', $sock))     $sock['tcpKeepAliveIdle']   = 20;
    if (!array_key_exists('tcpKeepAliveInterval', $sock)) $sock['tcpKeepAliveInterval'] = 5;
    if (!array_key_exists('tcpKeepAliveCount', $sock))    $sock['tcpKeepAliveCount']  = 3;
    if (!array_key_exists('tcpUserTimeout', $sock))       $sock['tcpUserTimeout']     = 150000;
    if (!array_key_exists('tcpMaxSeg', $sock))            $sock['tcpMaxSeg']          = 0;
    if (!array_key_exists('tcpBrutal', $sock))            $sock['tcpBrutal']          = true;
    if (!array_key_exists('tproxy', $sock))               $sock['tproxy']             = 'off';
    if (!array_key_exists('fragment', $sock) || !is_array($sock['fragment'])) {
        $sock['fragment'] = [
            'packets'  => 3,
            'length'   => 500,
            'interval' => 1,
        ];
    }

    $security = $ob['streamSettings']['security'] ?? null;
    if ($security === 'tls') {
        if (!isset($ob['streamSettings']['tlsSettings']) || !is_array($ob['streamSettings']['tlsSettings'])) {
            $ob['streamSettings']['tlsSettings'] = [];
        }
        if (!array_key_exists('enableSessionResumption', $ob['streamSettings']['tlsSettings'])) {
            $ob['streamSettings']['tlsSettings']['enableSessionResumption'] = true;
        }
        if (!array_key_exists('sessionTickets', $ob['streamSettings']['tlsSettings'])) {
            $ob['streamSettings']['tlsSettings']['sessionTickets'] = true;
        }
    }

    return $ob;
}

/** ---------- detect list-style array ---------- */
function is_list_array(array $a): bool {
    $i = 0;
    foreach ($a as $k => $_) {
        if ($k !== $i) return false;
        $i++;
    }
    return true;
}

/** ---------- JSON extraction ---------- */
function extract_outbounds_from_json(string $json): array {
    $data = json_decode($json, true);
    if (!is_array($data)) return [];
    if (isset($data['outbounds']) && is_array($data['outbounds'])) {
        $outs = [];
        foreach ($data['outbounds'] as $ob) {
            if (!is_array($ob) || empty($ob['protocol'])) continue;
            $outs[] = normalize_outbound($ob);
        }
        return $outs;
    }
    if (isset($data['protocol'])) return [ normalize_outbound($data) ];
    return [];
}

/** ---------- smart extraction ---------- */
function smart_extract_outbounds(string $raw): array {
    $t = trim($raw);

    $outs = extract_outbounds_from_json($t);
    if ($outs) return $outs;

    $first = strpos($t, '{'); $last = strrpos($t, '}');
    if ($first !== false && $last !== false && $last > $first) {
        $chunk = substr($t, $first, $last - $first + 1);
        $outs = extract_outbounds_from_json($chunk);
        if ($outs) return $outs;
    }

    $multi = [];
    if (preg_match_all('/\{(?:[^{}]|(?R))*\}/s', $t, $m)) {
        foreach ($m[0] as $blk) {
            $o = extract_outbounds_from_json($blk);
            if ($o) $multi = array_merge($multi, $o);
        }
        if ($multi) return $multi;
    }

    $col = [];
    foreach (preg_split('/\r?\n+/', $t) as $line) {
        $line = trim($line);
        if ($line === '') continue;

        if (is_vless($line)) {
            $ob = parse_vless($line);
            if ($ob) $col[] = normalize_outbound($ob);
            continue;
        }
        if (is_vmess($line)) {
            $ob = parse_vmess($line);
            if ($ob) $col[] = normalize_outbound($ob);
            continue;
        }

        if (($line[0] ?? '') === '{' || ($line[0] ?? '') === '[') {
            $o = extract_outbounds_from_json($line);
            if ($o) $col = array_merge($col, $o);
        }
    }
    return $col;
}

/** ---------- unique tag ---------- */
function unique_tag(string $base, array &$seen): string {
    $tag = $base;
    $i   = 1;
    while (isset($seen[$tag])) {
        $tag = $base . $i;
        $i++;
    }
    $seen[$tag] = true;
    return $tag;
}

/** ---------- health-check helpers (جدید) ---------- */
function outbound_get_endpoint(array $ob): ?array {
    $addr = null;
    $port = null;

    if (!empty($ob['settings']['vnext'][0]['address'])) {
        $addr = $ob['settings']['vnext'][0]['address'];
        $port = $ob['settings']['vnext'][0]['port'] ?? null;
    } elseif (!empty($ob['settings']['servers'][0]['address'])) {
        $addr = $ob['settings']['servers'][0]['address'];
        $port = $ob['settings']['servers'][0]['port'] ?? null;
    }

    if ($addr === null || $port === null) {
        return null;
    }

    $security = $ob['streamSettings']['security'] ?? 'none';

    $sni = null;
    if (!empty($ob['streamSettings']['tlsSettings']['serverName'])) {
        $sni = $ob['streamSettings']['tlsSettings']['serverName'];
    } elseif (!empty($ob['streamSettings']['realitySettings']['serverName'])) {
        $sni = $ob['streamSettings']['realitySettings']['serverName'];
    }

    return [
        'host'     => $addr,
        'port'     => (int)$port,
        'security' => strtolower($security),
        'sni'      => $sni,
    ];
}

function outbound_health(array $ob, int $timeout): array {
    $res = [
        'ok'      => false,
        'dns_ok'  => false,
        'tcp_ok'  => false,
        'tls_ok'  => null,
        'endpoint'=> null,
        'rtt_ms'  => null,
        'tries'   => 0,
    ];

    $ep = outbound_get_endpoint($ob);
    if ($ep === null) {
        return $res;
    }
    $res['endpoint'] = $ep;

    $host     = $ep['host'];
    $port     = $ep['port'];
    $security = $ep['security'];
    $sni      = $ep['sni'] ?: $host;

    $ip = $host;
    if (!filter_var($host, FILTER_VALIDATE_IP)) {
        $ip = @gethostbyname($host);
        $res['tries']++;
        if (empty($ip) || $ip === $host) {
            return $res;
        }
        $res['dns_ok'] = true;
    } else {
        $res['dns_ok'] = true;
    }

    $tcpStart = microtime(true);
    $errno = 0; $errstr = '';
    $sock = @fsockopen($ip, $port, $errno, $errstr, $timeout);
    $res['tries']++;
    if (!$sock) {
        $sock = @fsockopen($ip, $port, $errno, $errstr, $timeout);
        $res['tries']++;
        if (!$sock) {
            return $res;
        }
    }
    $tcpRtt = (microtime(true) - $tcpStart) * 1000.0;
    $res['tcp_ok'] = true;
    $res['rtt_ms'] = (int)round($tcpRtt);
    @fclose($sock);

    if ($security === 'tls' || $security === 'reality') {
        $tlsCtx = stream_context_create([
            'ssl' => [
                'verify_peer'      => false,
                'verify_peer_name' => false,
                'SNI_enabled'      => true,
                'peer_name'        => $sni,
            ]
        ]);

        $tlsStart = microtime(true);
        $tls = @stream_socket_client(
            "ssl://{$ip}:{$port}",
            $errno,
            $errstr,
            $timeout,
            STREAM_CLIENT_CONNECT,
            $tlsCtx
        );
        $res['tries']++;
        if (!$tls) {
            $res['tls_ok'] = false;
            return $res;
        }
        $tlsRtt = (microtime(true) - $tlsStart) * 1000.0;
        $res['tls_ok'] = true;
        if ($res['rtt_ms'] === null || $tlsRtt < $res['rtt_ms']) {
            $res['rtt_ms'] = (int)round($tlsRtt);
        }
        @fclose($tls);
    }

    $res['ok'] = $res['dns_ok'] && $res['tcp_ok'] && ($res['tls_ok'] !== false);
    return $res;
}

/** ---------- template ---------- */
function build_template(): array {
    global $ENABLE_FAKEDNS;

    $levels = (object)[];
    $levels->{'0'} = [
        "handshake"    => 12,
        "connIdle"     => 3600,
        "uplinkOnly"   => 0,
        "downlinkOnly" => 0,
    ];

    $template = [
        "log" => [
            "loglevel" => "warning",
            "dnsLog"   => false,
        ],
        "stats" => (object)[],
        "api" => [
            "tag"      => "api",
            "services" => ["StatsService"],
        ],
        "policy" => [
            "system" => [
                "statsInboundUplink"   => true,
                "statsInboundDownlink" => true,
            ],
            "levels" => $levels,
        ],
        "dns" => [
            "queryStrategy"         => "UseIP",
            "disableCache"          => false,
            "disableFallback"       => false,
            "disableFallbackIfMatch"=> true,
            "hosts" => [
                "domain:googleapis.cn"         => "googleapis.com",
                "cloudflare-dns.com"           => "1.1.1.1",
                "security.cloudflare-dns.com"  => "1.1.1.1",
                "dns.google"                   => "8.8.8.8",
            ],
            "servers" => [
                ["address" => "https://1.1.1.1/dns-query", "outboundTag" => "dns-out"],
                ["address" => "https://8.8.8.8/dns-query", "outboundTag" => "dns-out"],
                ["address" => "https://9.9.9.9/dns-query", "outboundTag" => "dns-out"],
                ["address" => "1.1.1.1", "outboundTag" => "dns-out", "timeoutMs" => 800, "serveStale" => true, "serveExpiredTTL" => 30],
                ["address" => "8.8.8.8", "outboundTag" => "dns-out", "timeoutMs" => 800, "serveStale" => true],
            ],
        ],
        "observatory" => [
            "subjectSelector"   => ["auto-group"],
            "probeURL"          => "http://edge.microsoft.com/captiveportal/generate_204",
            "enableConcurrency" => true,
        ],
        "inbounds" => [
            [
                "tag"      => "socks",
                "port"     => 10808,
                "listen"   => "127.0.0.1",
                "protocol" => "socks",
                "sniffing" => [
                    "enabled"      => true,
                    "routeOnly"    => false,
                    "destOverride" => ["http","tls","quic"],
                ],
                "settings" => [
                    "auth"      => "noauth",
                    "udp"       => true,
                    "ip"        => "127.0.0.1",
                    "userLevel" => 0,
                ],
            ],
            [
                "tag"      => "dns-in",
                "port"     => 10853,
                "listen"   => "127.0.0.1",
                "protocol" => "dokodemo-door",
                "settings" => [
                    "network"        => "tcp,udp",
                    "followRedirect" => false,
                    "address"        => "1.1.1.1",
                    "port"           => 53,
                ],
            ],
        ],
        "outbounds" => [
            ["tag" => "direct", "protocol" => "freedom", "settings" => ["domainStrategy" => "UseIP"]],
            ["tag" => "block", "protocol" => "blackhole", "settings" => (object)[]],
            ["tag" => "dns-out", "protocol" => "dns", "settings" => (object)[]],
        ],
        "routing" => [
            "domainStrategy" => "IPIfNonMatch",
            "domainMatcher"  => "hybrid",
            "balancers" => [
                ["tag" => "auto-group", "selector" => [], "strategy" => ["type" => "leastPing"]],
            ],
            "rules" => [
                ["type" => "field", "inboundTag" => ["dns-in"], "outboundTag" => "dns-out"],
                ["type" => "field", "protocol" => ["bittorrent"], "outboundTag" => "block"],
                ["type" => "field", "port" => "6881-6999", "outboundTag" => "block"],
                ["type" => "field", "ip" => ["10.0.0.0/8","172.16.0.0/12","192.168.0.0/16","127.0.0.0/8"], "outboundTag" => "direct"],
                ["type" => "field", "ip" => ["1.1.1.1"], "port" => "53", "outboundTag" => "dns-out"],
                ["type" => "field", "domain" => ["dns.google","cloudflare-dns.com","security.cloudflare-dns.com"], "network" => "tcp", "balancerTag" => "auto-group"],
                ["type" => "field", "network" => "tcp,udp", "balancerTag" => "auto-group"],
            ],
        ],
    ];

    if ($ENABLE_FAKEDNS) {
        $template['fakedns'] = [
            ["ipPool" => "198.18.0.0/15", "poolSize" => 10000],
        ];

        if (isset($template['inbounds'][0]['sniffing']['destOverride']) && is_array($template['inbounds'][0]['sniffing']['destOverride'])) {
            if (!in_array('fakedns', $template['inbounds'][0]['sniffing']['destOverride'], true)) {
                $template['inbounds'][0]['sniffing']['destOverride'][] = 'fakedns';
            }
        }
    }

    return $template;
}

function addObservatoryToConfig(array &$jsonConfig, array $preferredSelectors = []) {
    if (!isset($jsonConfig['outbounds']) || !is_array($jsonConfig['outbounds'])) return;

    $selectors = [];
    foreach ($jsonConfig['outbounds'] as $ob) {
        if (!isset($ob['tag'])) continue;
        $tag = $ob['tag'];
        if (in_array($tag, ['direct', 'dns-out', 'block'], true)) continue;
        $selectors[] = $tag;
    }

    $selectors = array_values(array_unique($selectors));

    if (!empty($preferredSelectors)) {
        $preferredSelectors = array_values(array_unique($preferredSelectors));
        $intersection       = array_values(array_intersect($preferredSelectors, $selectors));
        $selectors          = !empty($intersection) ? $intersection : $preferredSelectors;
    }

    if (empty($selectors)) return;

    $jsonConfig['observatory'] = [
        'subjectSelector'   => $selectors,
        'probeURL'          => 'http://detectportal.firefox.com/success.txt',
        'enableConcurrency' => true,
    ];
}

function forceAllowInsecure(&$template) {
    if (!isset($template['outbounds']) || !is_array($template['outbounds'])) return;

    foreach ($template['outbounds'] as &$ob) {
        if (!isset($ob['streamSettings']) || !is_array($ob['streamSettings'])) continue;

        $ss =& $ob['streamSettings'];
        $security = $ss['security'] ?? '';

        if ($security === 'tls') {
            if (!isset($ss['tlsSettings']) || !is_array($ss['tlsSettings'])) {
                $ss['tlsSettings'] = [];
            }
            $ss['tlsSettings']['allowInsecure'] = true;
        }

        if ($security === 'reality') {
            if (!isset($ss['realitySettings']) || !is_array($ss['realitySettings'])) {
                $ss['realitySettings'] = [];
            }
            $ss['realitySettings']['allowInsecure'] = true;
        }
    }
}