<?php
/**
 * Config Cache (JSON/RAW) – One-Time Print per Config
 *
 * - از لینک‌ها کانفیگ می‌گیرد (هر چی بدن – JSON یا متن خام)
 * - عمر هر کانفیگ: 5 دقیقه (300 ثانیه)
 * - هر لینک حداکثر هر 30 ثانیه یک بار درخواست می‌خورد (در حالت عادی)
 * - در هر درخواست، فقط یک کانفیگ چاپ می‌شود
 * - بعد از چاپ، همان کانفیگ از کش حذف می‌شود (تکراری چاپ نمی‌شود)
 * - اگر کش خالی شد، فوراً از لینک‌ها می‌گیرد؛ اگر باز هم نشد، مستقیم از لینک اول چاپ می‌کند
 */

header('Content-Type: application/json; charset=utf-8');

/* ====== USER SETTINGS ====== */
$LINKS = [
    'https://fitn1.ir/Api/AAA/1.php',
    'https://fitn1.ir/Api/AAA/2.php',
    'https://fitn1.ir/Api/AAA/3.php',
];

$FETCH_INTERVAL = 30;   // فاصله بین دو درخواست به هر لینک (ثانیه)
$CONFIG_TTL     = 300;  // عمر هر کانفیگ در کش (ثانیه) = ۵ دقیقه

$DATA_DIR   = __DIR__ . '/data_cfg_cache';
$STATE_FILE = $DATA_DIR . '/state.json';
$LOCK_FILE  = $DATA_DIR . '/lock.pid';

/* ====== آماده‌سازی پوشه ====== */
if (!is_dir($DATA_DIR)) {
    @mkdir($DATA_DIR, 0775, true);
}

/* ====== قفل برای جلوگیری از اجرای همزمان ====== */
$lockFp = @fopen($LOCK_FILE, 'c+');
if ($lockFp === false || !flock($lockFp, LOCK_EX)) {
    echo json_encode(['error' => 'lock_failed']);
    exit;
}

/* ====== توابع کمکی ====== */

function load_state($file)
{
    if (!file_exists($file)) {
        return [
            'items' => [],   // هر آیتم: ['config' => STRING, 'added' => timestamp]
            'links' => [],   // url => ['last_fetch' => timestamp]
        ];
    }
    $json = @file_get_contents($file);
    if ($json === false || $json === '') {
        return [
            'items' => [],
            'links' => [],
        ];
    }
    $data = json_decode($json, true);
    if (!is_array($data)) {
        return [
            'items' => [],
            'links' => [],
        ];
    }
    if (!isset($data['items']) || !is_array($data['items'])) {
        $data['items'] = [];
    }
    if (!isset($data['links']) || !is_array($data['links'])) {
        $data['links'] = [];
    }
    return $data;
}

function save_state($file, $state)
{
    $tmp  = $file . '.tmp';
    $json = json_encode($state, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
    @file_put_contents($tmp, $json);
    @rename($tmp, $file);
}

/**
 * گرفتن بدنه‌ی پاسخ از URL
 */
function fetch_body($url)
{
    if (!function_exists('curl_init')) {
        $ctx  = stream_context_create([
            'http' => [
                'timeout' => 15,
                'header'  => "User-Agent: SimpleCfgFetcher/1.0\r\n",
            ],
        ]);
        $body = @file_get_contents($url, false, $ctx);
        return ($body === false || $body === '') ? null : $body;
    }

    $ch = curl_init();
    curl_setopt_array($ch, [
        CURLOPT_URL            => $url,
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_CONNECTTIMEOUT => 8,
        CURLOPT_TIMEOUT        => 15,
        CURLOPT_FOLLOWLOCATION => true,
        CURLOPT_SSL_VERIFYPEER => false,
        CURLOPT_SSL_VERIFYHOST => false,
        CURLOPT_HTTPHEADER     => [
            'User-Agent: SimpleCfgFetcher/1.0',
        ],
    ]);
    $body = curl_exec($ch);
    $code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);

    if ($body === false || $body === '' || $code < 200 || $code >= 300) {
        return null;
    }
    return $body;
}

/**
 * حذف کانفیگ‌های منقضی شده
 */
function purge_expired(&$state, $ttl)
{
    $now = time();
    $out = [];
    foreach ($state['items'] as $it) {
        if (!isset($it['config'], $it['added'])) {
            continue;
        }
        if ($now - (int)$it['added'] <= $ttl) {
            $out[] = $it;
        }
    }
    $state['items'] = $out;
}

/**
 * یک دور گرفتن کانفیگ از لینک‌ها
 * اگر $force = true باشد، محدودیت ۳۰ ثانیه را نادیده می‌گیرد.
 */
function refill_from_links(&$state, $LINKS, $FETCH_INTERVAL, $ttl, $force = false)
{
    $now = time();

    foreach ($LINKS as $url) {
        if (!isset($state['links'][$url])) {
            $state['links'][$url] = ['last_fetch' => 0];
        }
        $last = (int)$state['links'][$url]['last_fetch'];

        if (!$force && ($now - $last) < $FETCH_INTERVAL) {
            continue;
        }

        $body = fetch_body($url);
        if ($body === null) {
            continue;
        }

        // کل بدنه را به عنوان یک کانفیگ جدید ذخیره می‌کنیم (بدون دِدیوپ سراسری)
        $state['items'][] = [
            'config' => $body,
            'added'  => $now,
        ];

        $state['links'][$url]['last_fetch'] = $now;
    }

    purge_expired($state, $ttl);
}

/* ====== منطق اصلی ====== */

$state = load_state($STATE_FILE);

// ۱) پاک کردن آیتم‌های منقضی شده
purge_expired($state, $CONFIG_TTL);

// ۲) اگر کش خالی است، یک بار به‌زور از لینک‌ها بگیر
if (count($state['items']) === 0) {
    refill_from_links($state, $LINKS, $FETCH_INTERVAL, $CONFIG_TTL, true);
}

// ۳) اگر هنوز خالی است، آخرین تلاش: مستقیم از لینک اول چاپ کن
if (count($state['items']) === 0) {
    $fallback = fetch_body($LINKS[0]);

    flock($lockFp, LOCK_UN);
    fclose($lockFp);

    if ($fallback === null) {
        echo json_encode(['error' => 'NO_CONFIG_AVAILABLE']);
    } else {
        echo $fallback;
    }
    exit;
}

// ۴) انتخاب یک کانفیگ و حذف آن از صف (بدون تکرار)
$item = array_shift($state['items']);   // اولین آیتم را برمی‌داریم و از آرایه حذف می‌کنیم

// ۵) ذخیره وضعیت جدید و آزاد کردن قفل
save_state($STATE_FILE, $state);
flock($lockFp, LOCK_UN);
fclose($lockFp);

// ۶) خروجی: فقط همان کانفیگی که این نوبت برایش بود
echo $item['config'];