<?php
/**
 * JSON Config Fetcher with Cache & Round-Robin
 * - لینک‌ها JSON کامل برمی‌گردانند (کل یک کانفیگ Xray/v2ray)
 * - هر 30 ثانیه حداکثر یک درخواست به هر لینک
 * - عمر هر کانفیگ: 5 دقیقه (300 ثانیه)
 * - بعد از هر درخواست موفق: 2 ثانیه تأخیر
 * - در هر کال: فقط یک JSON کامل چاپ می‌شود (round-robin)
 */

header('Content-Type: application/json; charset=utf-8');

/* ====== USER SETTINGS ====== */
$LINKS = [
    'https://fitn1.ir/Api/AAA/1.php',
    'https://fitn1.ir/Api/AAA/2.php',
    'https://fitn1.ir/Api/AAA/3.php',
];

// فاصله زمانی بین دو درخواست به هر لینک (ثانیه)
$FETCH_INTERVAL = 130;

// عمر هر کانفیگ در کش (ثانیه) = 5 دقیقه
$CONFIG_TTL = 600;

// فاصله بین دو درخواست موفق به لینک‌ها (ثانیه)
$DELAY_AFTER_SUCCESS = 2;

// مسیر ذخیره‌سازی کش (نسبت به همین فایل)
$DATA_DIR   = __DIR__ . '/data_json_cache';
$STATE_FILE = $DATA_DIR . '/state.json';
$LOCK_FILE  = $DATA_DIR . '/lock.pid';

/* ====== آماده‌سازی پوشه‌ها ====== */
if (!is_dir($DATA_DIR)) {
    @mkdir($DATA_DIR, 0775, true);
}

/* ====== قفل ساده برای جلوگیری از اجرای همزمان ====== */
$lockFp = @fopen($LOCK_FILE, 'c+');
if ($lockFp === false) {
    // اگر قفل باز نشد، بی‌صدا خارج شویم
    echo json_encode(['error' => 'lock_failed']);
    exit;
}
if (!flock($lockFp, LOCK_EX)) {
    fclose($lockFp);
    echo json_encode(['error' => 'lock_unavailable']);
    exit;
}

/* ====== توابع کمکی ====== */

function load_state($STATE_FILE)
{
    if (!file_exists($STATE_FILE)) {
        return [
            'items'  => [],  // هر آیتم: ['id'=>..., 'config'=>RAW_JSON_STRING, 'added'=>timestamp]
            'links'  => [],  // هر لینک: url => ['last_fetch'=>timestamp]
            'cursor' => 0,   // برای round-robin
        ];
    }

    $json = @file_get_contents($STATE_FILE);
    if ($json === false || $json === '') {
        return [
            'items'  => [],
            'links'  => [],
            'cursor' => 0,
        ];
    }

    $data = json_decode($json, true);
    if (!is_array($data)) {
        return [
            'items'  => [],
            'links'  => [],
            'cursor' => 0,
        ];
    }

    if (!isset($data['items']) || !is_array($data['items'])) {
        $data['items'] = [];
    }
    if (!isset($data['links']) || !is_array($data['links'])) {
        $data['links'] = [];
    }
    if (!isset($data['cursor']) || !is_int($data['cursor'])) {
        $data['cursor'] = 0;
    }

    return $data;
}

function save_state($STATE_FILE, $state)
{
    $tmpFile = $STATE_FILE . '.tmp';
    $json = json_encode($state, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
    @file_put_contents($tmpFile, $json);
    @rename($tmpFile, $STATE_FILE);
}

/**
 * ارسال درخواست به لینک و برگرداندن پاسخ خام
 * هیچ trim یا دستکاری روی بدنه انجام نمی‌شود
 */
function fetch_url_body($url)
{
    $ch = curl_init();
    curl_setopt_array($ch, [
        CURLOPT_URL            => $url,
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_CONNECTTIMEOUT => 8,
        CURLOPT_TIMEOUT        => 15,
        CURLOPT_FOLLOWLOCATION => true,
        CURLOPT_SSL_VERIFYPEER => false,
        CURLOPT_SSL_VERIFYHOST => false,
        CURLOPT_HTTPHEADER     => [
            'User-Agent: JSONFetcher/1.0',
        ],
    ]);

    $body = curl_exec($ch);
    $err  = curl_error($ch);
    $code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);

    if ($body === false || $body === '' || $code < 200 || $code >= 300) {
        // برای دیباگ:
        // error_log("Fetch failed for $url. HTTP=$code ERR=$err");
        return null;
    }

    return $body;
}

/**
 * چون هر پاسخ یک کانفیگ JSON کامل است،
 * همین بدنه را به عنوان یک کانفیگ واحد برمی‌گردانیم.
 * فقط چک می‌کنیم که شبیه JSON باشد (با { یا [ شروع شود).
 */
function extract_json_configs_from_body($body)
{
    // بدون trim شدید؛ فقط کپی خام
    $trimLeft = ltrim($body);
    if ($trimLeft === '') {
        return [];
    }
    $firstChar = $trimLeft[0];
    if ($firstChar !== '{' && $firstChar !== '[') {
        // اگر دوست داشتی می‌تونی اینجا رد کنی، یا کل بدنه رو همونطور ذخیره کنی
        // فعلاً فقط JSONهای واضح رو قبول می‌کنیم.
        return [];
    }

    // همین یک JSON کامل را به عنوان یک کانفیگ برمی‌گردانیم
    return [$body];
}

/* ====== منطق اصلی ====== */

$now   = time();
$state = load_state($STATE_FILE);

// پاک کردن کانفیگ‌های منقضی شده (قدیمی‌تر از TTL)
$filteredItems = [];
foreach ($state['items'] as $item) {
    if (!isset($item['added']) || !isset($item['config']) || !isset($item['id'])) {
        continue;
    }
    if ($now - (int)$item['added'] <= $CONFIG_TTL) {
        $filteredItems[] = $item;
    }
}
$state['items'] = $filteredItems;

// جلوگیری از ذخیره تکراری: map از idها (id = sha256 روی کل JSON خام)
$seenIds = [];
foreach ($state['items'] as $it) {
    $seenIds[$it['id']] = true;
}

// اگر کش خالی است، اجازه بده فوری از لینک‌ها بگیرد
$needForceFetch = (count($state['items']) === 0);

// حلقه روی لینک‌ها با رعایت فاصله زمانی و تأخیر بین درخواست‌ها
foreach ($LINKS as $url) {
    if (!isset($state['links'][$url])) {
        $state['links'][$url] = [
            'last_fetch' => 0,
        ];
    }

    $lastFetch = (int)$state['links'][$url]['last_fetch'];
    $dueByInterval = ($now - $lastFetch >= $FETCH_INTERVAL);

    if ($needForceFetch || $dueByInterval) {
        $body = fetch_url_body($url);
        if ($body !== null) {
            $configs = extract_json_configs_from_body($body);
            if (!empty($configs)) {
                foreach ($configs as $cfg) {
                    // id بر اساس کل JSON خام
                    $id = hash('sha256', $cfg);
                    if (isset($seenIds[$id])) {
                        continue;
                    }
                    $seenIds[$id] = true;
                    $state['items'][] = [
                        'id'     => $id,
                        'config' => $cfg,
                        'added'  => $now,
                    ];
                }
            }

            // بروزرسانی زمان آخرین fetch
            $state['links'][$url]['last_fetch'] = $now;

            // جلوگیری از درخواست پشت‌سرهم به لینک بعدی
            if ($DELAY_AFTER_SUCCESS > 0) {
                sleep($DELAY_AFTER_SUCCESS);
            }
        }
    }
}

// انتخاب یک کانفیگ برای چاپ (round-robin)
$count = count($state['items']);
if ($count === 0) {
    // هیچ کانفیگی در کش نیست
    save_state($STATE_FILE, $state);
    if (isset($lockFp)) {
        flock($lockFp, LOCK_UN);
        fclose($lockFp);
    }
    echo json_encode(['error' => 'NO_CONFIG_AVAILABLE']);
    exit;
}

// cursor
$cursor = isset($state['cursor']) ? (int)$state['cursor'] : 0;
if ($cursor < 0 || $cursor >= $count) {
    $cursor = 0;
}

// آیتم انتخابی این نوبت
$item = $state['items'][$cursor];

// بروزرسانی cursor برای نوبت بعد
$state['cursor'] = $cursor + 1;
if ($state['cursor'] >= $count) {
    $state['cursor'] = 0;
}

// ذخیره state
save_state($STATE_FILE, $state);

// آزاد کردن قفل
if (isset($lockFp)) {
    flock($lockFp, LOCK_UN);
    fclose($lockFp);
}

/* ====== خروجی نهایی: فقط یک JSON خام ====== */
echo $item['config'];