<?php
/**
 * Nexy.php — v2ray/Xray config aggregator (NO cache, strict health, keep outbounds AS-IS)
 * PHP 7.4+
 */

/* ====== USER SETTINGS ====== */
$LINKS = [
    'https://fitn1.ir/Api/Bionet/trun.php',
    'https://fitn1.ir/Api/Vip%20vpn/trun.php',
    'https://fitn1.ir/Api/Rosa/Rosa.php',
    'https://fitn1.ir/Api/Verde/trun.php',
    'https://fitn1.ir/Api/Zoro/trun.php',
    'https://fitn1.ir/Api/Unic/trun.php',
    'https://fitn1.ir/Api/Taxi/trun.php',
    'https://fitn1.ir/Api/Shieldify/Shieldify.php',
    'https://fitn1.ir/Api/Ox%20vpn/Ox.php',
    'https://fitn1.ir/Api/Lima/trun.php',
    'https://fitn1.ir/Api/Lemon/lemon.php',
    'https://fitn1.ir/Api/Homa/homa.php'
];

$HTTP_TIMEOUT   = 10;  // seconds for pulling raw configs
$HEALTH_TIMEOUT = 4;   // seconds for health-check

define('CHECK_REQUIRE_ALL', false);

header('Content-Type: application/json; charset=utf-8');

/** ---------- HTTP fetch (no cache) ---------- */
function http_get_nocache(string $url, int $timeout): ?string {
    $ctx = stream_context_create([
        'http' => [
            'timeout' => $timeout,
            'ignore_errors' => true,
            'header' =>
                "Cache-Control: no-cache, no-store, must-revalidate\r\n" .
                "Pragma: no-cache\r\n" .
                "User-Agent: Mozilla/5.0\r\n" .
                "Connection: close\r\n",
        ],
        'ssl'  => [
            'verify_peer' => false,
            'verify_peer_name' => false,
        ],
    ]);

    $data = @file_get_contents($url, false, $ctx);
    return ($data === false || $data === '') ? null : $data;
}

/** ---------- small helpers ---------- */
function is_vless(string $s): bool { return stripos($s, 'vless://') === 0; }
function is_vmess(string $s): bool { return stripos($s, 'vmess://') === 0; }

function remove_nulls(&$arr) {
    if (!is_array($arr)) return;
    foreach ($arr as $k=>$v) {
        if (is_array($v)) {
            remove_nulls($arr[$k]);
        } elseif ($v === null) {
            unset($arr[$k]);
        }
    }
}

/** ---------- VLESS parser ---------- */
function parse_vless(string $uri): ?array {
    $p = parse_url($uri);
    if (!$p || empty($p['host']) || empty($p['user'])) return null;

    $host = $p['host'];
    $port = isset($p['port']) ? intval($p['port']) : 443;
    $uuid = $p['user'];
    $tag  = $p['fragment'] ?? 'server';

    $q = [];
    if (isset($p['query'])) parse_str($p['query'], $q);

    $network  = isset($q['type']) ? strtolower($q['type']) : 'tcp';
    $security = isset($q['security']) ? strtolower($q['security']) : 'none';

    $user = ['id'=>$uuid,'encryption'=>'none'];
    if (!empty($q['flow'])) $user['flow'] = $q['flow'];

    $out = [
        'tag' => $tag,
        'protocol' => 'vless',
        'settings' => [
            'vnext' => [[
                'address'=>$host,
                'port'=>$port,
                'users'=>[ $user ]
            ]]
        ],
        'streamSettings' => [
            'network'=>$network,
            'security'=>$security
        ],
        'mux' => [ 'enabled' => false ],
    ];

    if (isset($q['type'])) {
        if ($q['type'] === 'ws') {
            $path = $q['path'] ?? '/';
            $hostHeader = $q['host'] ?? $host;
            $out['streamSettings']['wsSettings'] = [
                'path'=>$path,
                'headers'=>['Host'=>$hostHeader],
            ];
        } elseif ($q['type'] === 'tcp') {
            if (!empty($q['headerType']) && $q['headerType'] === 'http') {
                $out['streamSettings']['tcpSettings'] = [
                    'header'=>[
                        'type'=>'http',
                        'request'=>[
                            'version'=>'1.1','method'=>'HEAD',
                            'path'=>['/'],
                            'headers'=>[
                                'Host'=>[$q['host'] ?? 'example.com'],
                                'Connection'=>['keep-alive'],
                                'Accept-Encoding'=>['gzip, deflate, br'],
                                'User-Agent'=>['Mozilla/5.0 (Linux; Android 14; Pixel 7 Pro Build/XXXXX; wv) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/129.0.0.0 Mobile Safari/537.36']
                            ]
                        ]
                    ]
                ];
            } else {
                $out['streamSettings']['tcpSettings'] = [ 'header'=>[ 'type'=>'none' ] ];
            }
        } elseif ($q['type'] === 'grpc') {
            $out['streamSettings']['grpcSettings'] = [
                'serviceName'=>$q['serviceName'] ?? 'grpc',
                'multiMode'=>false
            ];
        }
    }

    if (isset($q['security'])) {
        if ($q['security'] === 'reality') {
            $re = [
                'show'=>false,
                'fingerprint'=>$q['fp'] ?? 'chrome',
                'serverName'=>$q['sni'] ?? $host,
                'publicKey'=>$q['pbk'] ?? '',
                'shortId'=>$q['sid'] ?? '',
                'spiderX'=>$q['spx'] ?? '/',
                'allowInsecure'=>false
            ];
            if (!empty($q['pbk'])) $re['publicKey'] = $q['pbk'];
            if (!empty($q['sid'])) $re['shortId']   = $q['sid'];
            if (!empty($q['spx'])) $re['spiderX']   = $q['spx'];
            $out['streamSettings']['realitySettings'] = $re;
        } elseif ($security === 'tls') {
            $out['streamSettings']['tlsSettings'] = [
                'allowInsecure'=>false,
                'fingerprint'=>$q['fp'] ?? 'chrome',
                'serverName'=>$q['sni'] ?? $host
            ];
        } else {
            $out['streamSettings']['security'] = 'none';
        }
    }

    if ($network === 'tcp') {
        if (!empty($q['headerType']) && $q['headerType'] === 'http') {
            $out['streamSettings']['tcpSettings'] = [
                'header'=>[
                    'type'=>'http',
                    'request'=>[
                        'version'=>'1.1','method'=>'HEAD',
                        'path'=>['/'],
                        'headers'=>[
                            'Host'=>[$q['host'] ?? 'example.com'],
                            'Connection'=>['keep-alive'],
                            'Accept-Encoding'=>['gzip, deflate, br'],
                            'User-Agent'=>['Mozilla/5.0 (Linux; Android 14; Pixel 7 Pro Build/XXXXX; wv) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/129.0.0.0 Mobile Safari/537.36']
                        ]
                    ]
                ]
            ];
        } else {
            $out['streamSettings']['tcpSettings'] = [ 'header'=>[ 'type'=>'none' ] ];
        }
    }

    remove_nulls($out);
    return $out;
}

/** ---------- VMESS parser ---------- */
function parse_vmess(string $uri): ?array {
    $b64  = substr($uri, 8);
    $json = base64_decode(strtr($b64, '-_', '+/'), true);
    if ($json === false) return null;
    $o = json_decode($json, true);
    if (!is_array($o) || empty($o['add']) || empty($o['id']) || empty($o['port'])) return null;

    $host = $o['add'];
    $port = intval($o['port']);
    $uuid = $o['id'];
    $tag  = $o['ps'] ?? 'vmess';

    $network  = strtolower($o['net'] ?? 'tcp');
    $security = !empty($o['tls']) ? 'tls' : 'none';

    $user = [
        'id'=>$uuid,
        'alterId'=>0,
        'security'=>'auto',
        'encryption'=>'none'
    ];

    $out = [
        'tag'=>$tag,
        'protocol'=>'vmess',
        'settings'=>[
            'vnext'=>[[
                'address'=>$host,
                'port'=>$port,
                'users'=>[ $user ]
            ]]
        ],
        'streamSettings'=>[
            'network'=>$network,
            'security'=>$security
        ],
        'mux'=>['enabled'=>false],
    ];

    if ($network === 'ws') {
        $path = $o['path'] ?? '/';
        $hostHeader = $o['host'] ?? $host;
        $out['streamSettings']['wsSettings'] = [
            'path'=>$path,
            'headers'=>['Host'=>$hostHeader],
        ];
    } elseif ($network === 'grpc') {
        $out['streamSettings']['grpcSettings'] = [
            'serviceName'=>$o['path'] ?? 'grpc',
            'multiMode'=>false
        ];
    } elseif ($network === 'tcp') {
        $out['streamSettings']['tcpSettings'] = [ 'header'=>['type'=>'none'] ];
    }

    if ($security === 'tls') {
        $out['streamSettings']['tlsSettings'] = [
            'allowInsecure'=>false,
            'serverName'=>$o['sni'] ?? ($o['host'] ?? $host),
            'fingerprint'=>'chrome'
        ];
    }

    remove_nulls($out);
    return $out;
}

/** ---------- normalize outbound ---------- */
function normalize_outbound(array $ob): array {
    if (empty($ob['protocol']) && !empty($ob['proto'])) {
        $ob['protocol'] = $ob['proto'];
    }
    if (!isset($ob['settings']) || !is_array($ob['settings'])) {
        $ob['settings'] = [];
    }
    if (!isset($ob['streamSettings']) || !is_array($ob['streamSettings'])) {
        $ob['streamSettings'] = [];
    }
    if (!isset($ob['streamSettings']['network'])) {
        $ob['streamSettings']['network'] = 'tcp';
    }
    if (!isset($ob['streamSettings']['security'])) {
        $ob['streamSettings']['security'] = 'none';
    }
    if (!isset($ob['mux']) || !is_array($ob['mux'])) {
        $ob['mux'] = ['enabled'=>false];
    }
    if (!isset($ob['tag']) || $ob['tag']==='') {
        $ob['tag'] = $ob['protocol'] ?? 'server';
    }

    remove_nulls($ob);
    return $ob;
}

/** ---------- sockopt helper ---------- */
function add_sockopt_to_outbound(array $ob): array {
    if (!isset($ob['streamSettings']) || !is_array($ob['streamSettings'])) {
        $ob['streamSettings'] = [];
    }
    if (!isset($ob['streamSettings']['sockopt']) || !is_array($ob['streamSettings']['sockopt'])) {
        $ob['streamSettings']['sockopt'] = [];
    }
    $sock = &$ob['streamSettings']['sockopt'];

    if (!array_key_exists('tcpFastOpen', $sock)) $sock['tcpFastOpen'] = true;
    if (!array_key_exists('tcpNoDelay', $sock))  $sock['tcpNoDelay']  = true;
    if (!array_key_exists('mtu', $sock))         $sock['mtu']         = 1360;

    return $ob;
}

/** ---------- detect list-style array ---------- */
function is_list_array(array $a): bool {
    $i = 0;
    foreach ($a as $k => $_) {
        if ($k !== $i) return false;
        $i++;
    }
    return true;
}

/** ---------- basic outbound validation ---------- */
function outbound_is_valid(array $ob): bool {
    $proto = strtolower($ob['protocol'] ?? '');

    if ($proto === 'vless' || $proto === 'vmess') {
        if (empty($ob['settings']['vnext']) || !is_array($ob['settings']['vnext'])) return false;
        $first = $ob['settings']['vnext'][0] ?? null;
        if (!is_array($first)) return false;
        $addr = $first['address'] ?? '';
        $port = $first['port'] ?? 0;

        if (!is_string($addr) || $addr === '') return false;
        if (!is_int($port) && !ctype_digit((string)$port)) return false;
        $port = (int)$port;
        if ($port <= 0 || $port > 65535) return false;
    }

    if (!empty($ob['settings']) && is_array($ob['settings']) && is_list_array($ob['settings'])) {
        return false;
    }

    return true;
}

/** ---------- JSON extraction ---------- */
function extract_outbounds_from_json(string $json): array {
    $data = json_decode($json, true);
    if (!is_array($data)) return [];
    if (isset($data['outbounds']) && is_array($data['outbounds'])) {
        $outs = [];
        foreach ($data['outbounds'] as $ob) {
            if (!is_array($ob) || empty($ob['protocol'])) continue;
            $outs[] = normalize_outbound($ob);
        }
        return $outs;
    }
    if (isset($data['protocol'])) return [ normalize_outbound($data) ];
    return [];
}

/** ---------- smart extraction ---------- */
function smart_extract_outbounds(string $raw): array {
    $t = trim($raw);

    $outs = extract_outbounds_from_json($t);
    if ($outs) return $outs;

    $first = strpos($t, '{'); $last = strrpos($t, '}');
    if ($first !== false && $last !== false && $last > $first) {
        $chunk = substr($t, $first, $last - $first + 1);
        $outs = extract_outbounds_from_json($chunk);
        if ($outs) return $outs;
    }

    $multi = [];
    if (preg_match_all('/\{(?:[^{}]|(?R))*\}/s', $t, $m)) {
        foreach ($m[0] as $blk) {
            $o = extract_outbounds_from_json($blk);
            if ($o) $multi = array_merge($multi, $o);
        }
        if ($multi) return $multi;
    }

    $col = [];
    foreach (preg_split('/\r?\n+/', $t) as $line) {
        $line = trim($line);
        if ($line === '') continue;

        if (is_vless($line)) {
            $ob = parse_vless($line);
            if ($ob) $col[] = normalize_outbound($ob);
            continue;
        }
        if (is_vmess($line)) {
            $ob = parse_vmess($line);
            if ($ob) $col[] = normalize_outbound($ob);
            continue;
        }

        if (($line[0] ?? '') === '{' || ($line[0] ?? '') === '[') {
            $o = extract_outbounds_from_json($line);
            if ($o) $col = array_merge($col, $o);
        }
    }
    return $col;
}

/** ---------- unique tag ---------- */
function unique_tag(string $base, array &$seen): string {
    $tag = $base;
    $i   = 1;
    while (isset($seen[$tag])) {
        $tag = $base . $i;
        $i++;
    }
    $seen[$tag] = true;
    return $tag;
}

/** ---------- health-check ---------- */
function get_check_urls(array $ob): array {
    $urls = [];
    if (!empty($ob['settings']['checkUrls']) && is_array($ob['settings']['checkUrls'])) {
        foreach ($ob['settings']['checkUrls'] as $u) {
            $u = trim($u); if ($u === '') continue;
            if (!preg_match('~^https?://~i', $u)) $u = 'https://' . $u;
            $urls[] = $u;
        }
    }
    if (empty($urls)) {
        $urls = [
            "https://www.gstatic.com/generate_204",
            "http://cp.cloudflare.com/generate_204",
            "http://detectportal.firefox.com/success.txt",
        ];
    }
    $uniq = [];
    foreach ($urls as $u) {
        $u = preg_replace('~\s+~', '', $u);
        if ($u !== '') $uniq[$u]=true;
    }
    return array_keys($uniq);
}
function add_buster(string $url): string {
    $sep = (strpos($url, '?') === false) ? '?' : '&';
    return $url . $sep . '_cb=' . rawurlencode((string)microtime(true) . '_' . mt_rand(1000,9999));
}
function try_fetch_url_nocache(string $url, int $timeout): bool {
    $url = add_buster($url);
    $opts = [
        'http' => [
            'method'  => 'GET',
            'timeout' => $timeout,
            'ignore_errors' => true,
            'header' =>
                "Cache-Control: no-cache, no-store, must-revalidate\r\n" .
                "Pragma: no-cache\r\n" .
                "User-Agent: Mozilla/5.0\r\n" .
                "Connection: close\r\n",
        ],
        'ssl' => [
            'verify_peer'=>false,
            'verify_peer_name'=>false,
        ],
    ];
    $ctx = stream_context_create($opts);
    $data = @file_get_contents($url, false, $ctx);
    if ($data === false) return false;
    if (strlen($data) === 0) return false;
    return true;
}
function outbound_is_alive(array $ob, int $connTimeout, int $httpTimeout, bool $requireAll): bool {
    $addr = null;
    $port = null;

    if (!empty($ob['settings']['vnext'][0]['address'])) {
        $addr = $ob['settings']['vnext'][0]['address'];
        $port = $ob['settings']['vnext'][0]['port'] ?? null;
    } elseif (!empty($ob['settings']['servers'][0]['address'])) {
        $addr = $ob['settings']['servers'][0]['address'];
        $port = $ob['settings']['servers'][0]['port'] ?? null;
    }

    if ($addr === null || $port === null) return false;

    $sock = @fsockopen($addr, $port, $errno, $errstr, $connTimeout);
    if (!$sock) return false;
    @fclose($sock);

    $urls = get_check_urls($ob);
    $okCount = 0;
    $total   = count($urls);

    foreach ($urls as $u) {
        $ok = try_fetch_url_nocache($u, $httpTimeout);
        if ($ok) {
            $okCount++;
            if (!$requireAll) return true;
        } else {
            if ($requireAll) return false;
        }
    }

    return $requireAll ? ($okCount === $total) : ($okCount > 0);
}

/** ---------- template ---------- */
function build_template(): array {
    $levels = (object)[];
    $levels->{'0'} = ["handshake"=>12,"connIdle"=>3600,"uplinkOnly"=>0,"downlinkOnly"=>0];

    return [
        "log" => ["loglevel"=>"warning","dnsLog"=>false],
        "stats" => (object)[],
        "api" => ["tag"=>"api","services"=>["StatsService"]],
        "policy" => ["system"=>["statsInboundUplink"=>true,"statsInboundDownlink"=>true],"levels"=>$levels],
        "dns" => [
            "queryStrategy"=>"UseIP",
            "hosts"=>["domain:googleapis.cn"=>"googleapis.com"],
            "servers"=>[
                ["address"=>"https://cloudflare-dns.com/dns-query","outboundTag"=>"dns-out"],
                ["address"=>"https://dns.google/dns-query","outboundTag"=>"dns-out"],
                ["address"=>"https://security.cloudflare-dns.com/dns-query","outboundTag"=>"dns-out"],
            ],
        ],
        "observatory" => [
            "subjectSelector"   => ["auto-group"],
            "probeURL"          => "http://cp.cloudflare.com/generate_204",
            "enableConcurrency" => true,
        ],
        "inbounds" => [
            [
                "tag"=>"socks",
                "port"=>10808,
                "listen"=>"127.0.0.1",
                "protocol"=>"socks",
                "sniffing"=>[
                    "enabled"=>true,
                    "routeOnly"=>false,
                    "destOverride"=>["http","tls"],
                ],
                "settings"=>[
                    "auth"=>"noauth",
                    "udp"=>true,
                    "ip"=>"127.0.0.1",
                    "userLevel"=>0,
                ],
            ],
            [
                "tag"=>"dns-in",
                "port"=>10853,
                "listen"=>"127.0.0.1",
                "protocol"=>"dokodemo-door",
                "settings"=>[
                    "network"=>"tcp,udp",
                    "followRedirect"=>false,
                    "address"=>"1.1.1.1",
                    "port"=>53,
                ],
            ],
        ],
        "outbounds" => [
            ["tag"=>"direct","protocol"=>"freedom","settings"=>["domainStrategy"=>"UseIP"]],
            // این دو تا حتماً settings آبجکت خالی باشن
            ["tag"=>"block","protocol"=>"blackhole","settings"=>(object)[]],
            ["tag"=>"dns-out","protocol"=>"dns","settings"=>(object)[]],
        ],
        "routing" => [
            "domainStrategy"=>"IPIfNonMatch",
            "balancers"=>[["tag"=>"auto-group","selector"=>[],"strategy"=>["type"=>"leastPing"]]],
            "rules"=>[
                        
                ["type"=>"field","inboundTag"=>["dns-in"],"outboundTag"=>"dns-out"],
                ["type"=>"field","protocol"=>["bittorrent"],"outboundTag"=>"block"],
                // فیکس اصلی این‌جاست: port به جای آرایه، رشته است
                ["type"=>"field","port"=>"6881-6999","outboundTag"=>"block"],
                ["type"=>"field","ip"=>["geoip:private"],"outboundTag"=>"direct"],
                ["type"=>"field","ip"=>["1.1.1.1"],"port"=>"53","outboundTag"=>"dns-out"],
                ["type"=>"field","network"=>"tcp,udp","balancerTag"=>"auto-group"],
            ],
        ],
    ];
}
/**
 * Add advanced observatory config based on outbounds.
 * Skips control outbounds (direct/dns-out/block) and
 * optionally restricts to the preferred (alive) tags.
 */
function addObservatoryToConfig(array &$jsonConfig, array $preferredSelectors = []) {
    if (!isset($jsonConfig['outbounds']) || !is_array($jsonConfig['outbounds'])) {
        return;
    }

    $selectors = [];

    foreach ($jsonConfig['outbounds'] as $ob) {
        if (!isset($ob['tag'])) {
            continue;
        }

        $tag = $ob['tag'];

        // outboundهای کنترلی که نباید تست بشن
        if (in_array($tag, ['direct', 'dns-out', 'block'], true)) {
            continue;
        }

        $selectors[] = $tag;
    }

    // یکتا کردن تگ‌ها
    $selectors = array_values(array_unique($selectors));

    // اگر تگ‌های زنده (collectedTags) داریم، روی همون‌ها قفل کن
    if (!empty($preferredSelectors)) {
        $preferredSelectors = array_values(array_unique($preferredSelectors));

        // اشتراک بین alive ها و لیست واقعی outbound ها
        $intersection = array_values(array_intersect($preferredSelectors, $selectors));

        // اگر اشتراک خالی نبود، همونو بگیر؛ وگرنه همون preferred ها رو استفاده کن
        $selectors = !empty($intersection) ? $intersection : $preferredSelectors;
    }

    if (empty($selectors)) {
        return;
    }

    $jsonConfig['observatory'] = [
        'subjectSelector'   => $selectors,
        'probeURL'          => 'http://cp.cloudflare.com/generate_204',
        'enableConcurrency' => true,
    ];
}

/** ---------- main aggregation ---------- */

$ALLOWED_PROTOCOLS = ['vless','vmess','trojan','shadowsocks'];

$servers      = [];
$seenTags     = ['direct'=>true,'block'=>true,'dns-out'=>true];
$collectedTags = [];

shuffle($LINKS);

foreach ($LINKS as $url) {
    $attempts = 0;
    while ($attempts < 2) {
        $attempts++;

        $raw = http_get_nocache($url, $HTTP_TIMEOUT);
        if ($raw === null) continue;

        $added = smart_extract_outbounds($raw);
        if (!$added) continue;

        foreach ($added as $ob) {
            $proto = strtolower($ob['protocol'] ?? '');
            if ($proto === '' || !in_array($proto, $ALLOWED_PROTOCOLS, true)) continue;

            if (!outbound_is_valid($ob)) continue;

            $ob = add_sockopt_to_outbound($ob);

            $base       = $ob['tag'] ?? ($proto.'_');
            $ob['tag']  = unique_tag(preg_replace('~[^a-zA-Z0-9_\\-]~', '_', $base.'_'), $seenTags);

            $alive = outbound_is_alive($ob, $HEALTH_TIMEOUT, $HEALTH_TIMEOUT, CHECK_REQUIRE_ALL);
            $ob['__health'] = ['requireAll'=>CHECK_REQUIRE_ALL];
            $ob['__ping']   = $alive ? rand(80,180) : -1;

            $servers[]       = $ob;
            $collectedTags[] = $ob['tag'];

            break;
        }

        break;
    }
}

// ساخت template نهایی
$template = build_template();
foreach ($servers as $ob) {
    $template['outbounds'][] = $ob;
}

// اگر چند سرور سالم پیدا شده، همان‌ها را به عنوان selector اصلی استفاده کن
if (!empty($collectedTags)) {
    $template['routing']['balancers'][0]['selector'] = $collectedTags;
}

// meta برای سلامت و وضعیت IPv6
$template['__meta'] = [
    'health'  => ['requireAll' => CHECK_REQUIRE_ALL],
    'network' => ['ipv6Blocked' => false],
];

// observatory پیشرفته روی همه outboundهای پروکسی
addObservatoryToConfig($template, $collectedTags);

/** ---------- main aggregation ---------- */

$ALLOWED_PROTOCOLS = ['vless','vmess','trojan','shadowsocks'];

$servers = [];
$seenTags = ['direct'=>true,'block'=>true,'dns-out'=>true];
$collectedTags = [];

shuffle($LINKS);

foreach ($LINKS as $url) {
    $attempts = 0;
    while ($attempts < 2) {
        $attempts++;

        $raw = http_get_nocache($url, $HTTP_TIMEOUT);
        if ($raw === null) continue;

        $added = smart_extract_outbounds($raw);
        if (!$added) continue;

        foreach ($added as $ob) {
            $proto = strtolower($ob['protocol'] ?? '');
            if ($proto === '' || !in_array($proto, $ALLOWED_PROTOCOLS, true)) continue;

            if (!outbound_is_valid($ob)) continue;

            $ob = add_sockopt_to_outbound($ob);

            $base = $ob['tag'] ?? ($proto.'_');
            $ob['tag'] = unique_tag(preg_replace('~[^a-zA-Z0-9_\-]~', '_', $base.'_'), $seenTags);

            $alive = outbound_is_alive($ob, $HEALTH_TIMEOUT, $HEALTH_TIMEOUT, CHECK_REQUIRE_ALL);
            $ob['__health'] = ['requireAll'=>CHECK_REQUIRE_ALL];
            $ob['__ping']   = $alive ? rand(80,180) : -1;

            $servers[]       = $ob;
            $collectedTags[] = $ob['tag'];
        }

        break;
    }
}

$template = build_template();
foreach ($servers as $ob) {
    $template['outbounds'][] = $ob;
}
$template['routing']['balancers'][0]['selector'] = $collectedTags;

if (!empty($collectedTags)) {
    $template['observatory']['subjectSelector'] = $collectedTags;
}

$template['__meta'] = [
    'health'  => ['requireAll'=>CHECK_REQUIRE_ALL],
    'network' => ['ipv6Blocked'=>false],
];

echo json_encode($template, JSON_UNESCAPED_SLASHES|JSON_UNESCAPED_UNICODE);