<?php
/**
 * Nexy.php — v2ray/Xray config aggregator (NO cache, strict health, keep outbounds AS-IS)
 * PHP 7.4+
 */

/* ====== USER SETTINGS ====== */
$LINKS = [
    'https://fitn1.ir/Api/Bionet/trun.php',
    'https://fitn1.ir/Api/Vip%20vpn/trun.php',
    'https://fitn1.ir/Api/Verde/trun.php',
    'https://fitn1.ir/Api/Zoro/trun.php',
    'https://fitn1.ir/Api/Unic/trun.php',
    'https://fitn1.ir/Api/Taxi/trun.php',
    'https://fitn1.ir/Api/Shieldify/Shieldify.php',
    'https://fitn1.ir/Api/Ox%20vpn/Ox.php',
    'https://fitn1.ir/Api/Lima/trun.php',
    'https://fitn1.ir/Api/Lemon/lemon.php',
    'https://fitn1.ir/Api/Rosa/Rosahome.php',
    'https://fitn1.ir/Api/Ash/trun.php',
    'https://fitn1.ir/Api/Bull/trun.php',
    'https://fitn1.ir/Api/Homa/homa.php',
    'https://fitn1.ir/Api/Ahu/trun.php',
    'https://fitn1.ir/Api/Exir/trun.php',
    'https://fitn1.ir/Api/Yrvpn/Yr.php',
    'https://fitn1.ir/Api/ConfigTelegram/Configtelegram.php',
    'https://fitn1.ir/Api/Oxygen%20/trun.php',
    'https://fitn1.ir/Api/Avox/trun.php',
    'https://fitn1.ir/Api/Caren/trun.php',
    'https://fitn1.ir/Api/dragon/dragon.php',
    'https://fitn1.ir/Api/Like%20vpn/trun.php',
    'https://fitn1.ir/Api/Mona/trun.php',
    'https://fitn1.ir/Api/Monkey/trun.php',
    'https://fitn1.ir/Api/node/nodevpn.php',
    'https://fitn1.ir/Api/Shiro/Shiro.php',
    'https://fitn1.ir/Api/Thor/trun.php',
    'https://fitn1.ir/Api/Fon/trun.php',
    'https://fitn1.ir/Api/Vpn-lar/trun.php'
];

$HTTP_TIMEOUT   = 10;  // seconds for pulling raw configs
$HEALTH_TIMEOUT = 4;   // seconds for health-check

$REQUEST_TIMEOUT = 70;

$RATE_LIMIT_SECONDS = 3;
$LOCK_TTL_SECONDS   = 70;

$SAVE_OUTPUT_HISTORY = true;
$HISTORY_DIR         = __DIR__ . '/history';
$HISTORY_TTL_SECONDS = 5 * 60;

$PROBE_URLS = [
    "http://edge.microsoft.com/captiveportal/generate_204",
    "http://detectportal.firefox.com/success.txt",
    "https://www.gstatic.com/generate_204",
    "http://cp.cloudflare.com/generate_204",
    "http://connectivitycheck.gstatic.com/generate_204"
];

$ENABLE_FAKEDNS = true;

define('CHECK_REQUIRE_ALL', false);

header('Content-Type: application/json; charset=utf-8');

/* ===================== FEATURE FLAGS =====================
 * قانون کلی:
 * - false => اصلاح/نرمال‌سازی انجام شود (بدون ساخت مشکل)
 * - true  => مثل قبل باشد (رفتار قبلی حتی اگر مشکل‌دار)
 */
$FEATURES = [
    // قبلی‌ها
    'enable_mux'           => false,
    'enable_xhttp'         => false,
    'enable_sockopt'       => false,
    'enable_padding'       => false,
    'enable_fragment'      => false,
    'enable_metadata'      => false,
    'allow_structural_mix' => false, // true => مشکل قبل / false => جلوگیری از میکس ساختاری

    // جدیدها (طبق درخواست شما)
    'keep_bad_tags_selectors'      => false, // false => تگ‌ها/سلکتورها اصلاح شوند | true => مثل قبل
    'allow_duplicate_tags'         => false, // false => تگ تکراری نداشته باشد | true => مثل قبل
    'keep_bad_dns_doh'             => false, // false => DoH/DNS اصلاح شود | true => مثل قبل
    'keep_disableFallbackIfMatch'  => false, // false => مقدار درست/بولین شود | true => مثل قبل
    'keep_observatory_leastping'   => false, // false => تضمین leastPing/selector درست | true => مثل قبل
    'keep_invalid_ranges'          => false, // false => رنج‌ها و مقادیر نامعتبر اصلاح شوند | true => مثل قبل
    'keep_fakedns_logic'           => false, // false => FakeDNS و destOverride درست/هماهنگ | true => مثل قبل
];

/* ===================== 0) request control ===================== */

@set_time_limit((int)$REQUEST_TIMEOUT);
@ini_set('max_execution_time', (string)((int)$REQUEST_TIMEOUT));

function ensure_dir(string $dir) { if (!is_dir($dir)) @mkdir($dir, 0777, true); }

function cleanup_history_files(string $dir, int $ttlSeconds) {
    if (!is_dir($dir)) return;
    $now = time();
    foreach (glob($dir . '/*.txt') ?: [] as $f) {
        $mt = @filemtime($f);
        if ($mt !== false && ($now - $mt) > $ttlSeconds) @unlink($f);
    }
}

function save_output_line(string $dir, string $line) {
    ensure_dir($dir);
    $name = 'outputs_' . date('Ymd_Hi') . '.txt';
    $path = rtrim($dir, '/\\') . '/' . $name;
    $fp = @fopen($path, 'ab');
    if ($fp) { @fwrite($fp, $line . "\n"); @fclose($fp); }
}

function client_ip(): string {
    $ip = $_SERVER['HTTP_CF_CONNECTING_IP'] ?? $_SERVER['HTTP_X_FORWARDED_FOR'] ?? $_SERVER['REMOTE_ADDR'] ?? '0.0.0.0';
    if (strpos($ip, ',') !== false) $ip = trim(explode(',', $ip)[0]);
    return trim($ip);
}

function serve_one_output(string $dir): ?string {
    if (!is_dir($dir)) return null;
    $files = glob(rtrim($dir, '/\\') . '/outputs_*.txt');
    if (!$files) return null;
    sort($files);

    $ptrFile = rtrim($dir, '/\\') . '/pointer.idx';
    $ptrRaw  = is_file($ptrFile) ? trim((string)@file_get_contents($ptrFile)) : '';
    $fileIdx = 0; $lineIdx = 0;
    if ($ptrRaw !== '' && strpos($ptrRaw, ':') !== false) {
        [$a, $b] = explode(':', $ptrRaw, 2);
        $fileIdx = max(0, (int)$a);
        $lineIdx = max(0, (int)$b);
    }
    if ($fileIdx >= count($files)) { $fileIdx = 0; $lineIdx = 0; }

    for ($round = 0; $round < 2; $round++) {
        for ($i = $fileIdx; $i < count($files); $i++) {
            $lines = @file($files[$i], FILE_IGNORE_NEW_LINES);
            if (!$lines) { $lineIdx = 0; continue; }
            $n = count($lines);
            while ($lineIdx < $n && trim((string)$lines[$lineIdx]) === '') $lineIdx++;

            if ($lineIdx < $n) {
                $out = (string)$lines[$lineIdx];
                $nextLine = $lineIdx + 1;
                $nextFile = $i;
                if ($nextLine >= $n) { $nextLine = 0; $nextFile = $i + 1; if ($nextFile >= count($files)) $nextFile = 0; }
                @file_put_contents($ptrFile, $nextFile . ':' . $nextLine, LOCK_EX);
                return $out;
            } else {
                $lineIdx = 0;
            }
        }
        $fileIdx = 0; $lineIdx = 0;
    }
    return null;
}

function serve_from_history_and_exit(string $dir): void {
    $out = serve_one_output($dir);
    if ($out !== null && $out !== '') { header('Content-Type: application/json; charset=utf-8'); echo $out; exit; }
}

function rate_limit_soft(string $dir, int $seconds): void {
    $ip  = client_ip();
    $key = sys_get_temp_dir() . '/nexy_rl_' . preg_replace('~[^a-zA-Z0-9_\-]~', '_', $ip) . '.txt';
    $now = time();
    $last = 0;
    if (is_file($key)) $last = (int)@file_get_contents($key);
    if ($last > 0 && ($now - $last) < $seconds) serve_from_history_and_exit($dir);
    @file_put_contents($key, (string)$now, LOCK_EX);
}

function try_acquire_lock(string $lockFile, int $ttlSeconds): bool {
    $now = time();
    if (!is_file($lockFile)) return (@file_put_contents($lockFile, (string)$now . "\n" . getmypid(), LOCK_EX) !== false);
    $mtime = @filemtime($lockFile);
    if ($mtime !== false && ($now - $mtime) > $ttlSeconds) { @unlink($lockFile); return (@file_put_contents($lockFile, (string)$now . "\n" . getmypid(), LOCK_EX) !== false); }
    return false;
}

function release_lock(string $lockFile): void { @unlink($lockFile); }

cleanup_history_files($HISTORY_DIR, $HISTORY_TTL_SECONDS);
rate_limit_soft($HISTORY_DIR, $RATE_LIMIT_SECONDS);

$LOCK_FILE = __DIR__ . '/nexy_run.lock';
if (!try_acquire_lock($LOCK_FILE, $LOCK_TTL_SECONDS)) {
    serve_from_history_and_exit($HISTORY_DIR);
} else {
    register_shutdown_function(function() use ($LOCK_FILE) { release_lock($LOCK_FILE); });
}

/** ---------- HTTP fetch (no cache) ---------- */
function http_get_nocache(string $url, int $timeout): ?string {
    $ctx = stream_context_create([
        'http' => [
            'timeout' => $timeout,
            'ignore_errors' => true,
            'header' =>
                "Cache-Control: no-cache, no-store, must-revalidate\r\n" .
                "Pragma: no-cache\r\n" .
                "User-Agent: Mozilla/5.0\r\n" .
                "Connection: close\r\n",
        ],
        'ssl'  => [
            'verify_peer' => false,
            'verify_peer_name' => false,
        ],
    ]);
    $data = @file_get_contents($url, false, $ctx);
    return ($data === false || $data === '') ? null : $data;
}

/** ---------- helpers ---------- */
function is_vless(string $s): bool { return stripos($s, 'vless://') === 0; }
function is_vmess(string $s): bool { return stripos($s, 'vmess://') === 0; }
function is_trojan(string $s): bool { return stripos($s, 'trojan://') === 0; }
function is_ss(string $s): bool { return stripos($s, 'ss://') === 0; }

function remove_nulls(&$arr) {
    if (!is_array($arr)) return;
    foreach ($arr as $k => $v) {
        if (is_array($v)) remove_nulls($arr[$k]);
        elseif ($v === null) unset($arr[$k]);
    }
}

function b64url_decode(string $s): string {
    $s = trim($s);
    $s = strtr($s, '-_', '+/');
    $pad = strlen($s) % 4;
    if ($pad) $s .= str_repeat('=', 4 - $pad);
    $out = base64_decode($s, true);
    return ($out === false) ? '' : $out;
}

function clamp_int($v, int $min, int $max, int $def): int {
    if (!is_int($v) && !ctype_digit((string)$v)) return $def;
    $v = (int)$v;
    if ($v < $min) return $def;
    if ($v > $max) return $def;
    return $v;
}

function is_valid_tag_name(string $t): bool {
    // اجازه: a-zA-Z0-9_- و حداقل 1
    return $t !== '' && preg_match('~^[a-zA-Z0-9_\-]+$~', $t);
}

/** ---------- VLESS parser ---------- */
function parse_vless(string $uri): ?array {
    $p = parse_url($uri);
    if (!$p || empty($p['host']) || empty($p['user'])) return null;

    $host = $p['host'];
    $port = isset($p['port']) ? intval($p['port']) : 443;
    $uuid = $p['user'];
    $tag  = $p['fragment'] ?? 'server';

    $q = [];
    if (isset($p['query'])) parse_str($p['query'], $q);

    $network  = isset($q['type']) ? strtolower($q['type']) : 'tcp';
    $security = isset($q['security']) ? strtolower($q['security']) : 'none';

    $user = ['id' => $uuid, 'encryption' => 'none'];
    if (!empty($q['flow'])) $user['flow'] = $q['flow'];

    $out = [
        'tag' => $tag,
        'protocol' => 'vless',
        'settings' => [
            'vnext' => [[
                'address' => $host,
                'port'    => $port,
                'users'   => [ $user ]
            ]]
        ],
        'streamSettings' => [
            'network'  => $network,
            'security' => $security
        ],
        'mux' => [ 'enabled' => false ],
    ];

    if (isset($q['type'])) {
        if ($q['type'] === 'ws') {
            $path       = $q['path'] ?? '/';
            $hostHeader = $q['host'] ?? $host;
            $out['streamSettings']['wsSettings'] = [
                'path'    => $path,
                'headers' => ['Host' => $hostHeader],
            ];
        } elseif ($q['type'] === 'tcp') {
            if (!empty($q['headerType']) && $q['headerType'] === 'http') {
                $out['streamSettings']['tcpSettings'] = [
                    'header' => [
                        'type'    => 'http',
                        'request' => [
                            'version' => '1.1',
                            'method'  => 'HEAD',
                            'path'    => ['/'],
                            'headers' => [
                                'Host'            => [$q['host'] ?? 'example.com'],
                                'Connection'      => ['keep-alive'],
                                'Accept-Encoding' => ['gzip, deflate, br'],
                                'User-Agent'      => ['Mozilla/5.0 (Linux; Android 14) AppleWebKit/537.36 (KHTML, like Gecko) Chrome Mobile Safari/537.36']
                            ]
                        ]
                    ]
                ];
            } else {
                $out['streamSettings']['tcpSettings'] = [ 'header' => [ 'type' => 'none' ] ];
            }
        } elseif ($q['type'] === 'grpc') {
            $out['streamSettings']['grpcSettings'] = [
                'serviceName' => $q['serviceName'] ?? 'grpc',
                'multiMode'   => false
            ];
        } elseif ($q['type'] === 'xhttp') {
            $out['streamSettings']['network'] = 'xhttp';
        }
    }

    if (isset($q['security'])) {
        if ($q['security'] === 'reality') {
            $re = [
                'show'        => false,
                'fingerprint' => $q['fp'] ?? 'chrome',
                'serverName'  => $q['sni'] ?? $host,
                'publicKey'   => $q['pbk'] ?? '',
                'shortId'     => $q['sid'] ?? '',
                'spiderX'     => $q['spx'] ?? '/',
                'allowInsecure' => false
            ];
            if (!empty($q['pbk'])) $re['publicKey'] = $q['pbk'];
            if (!empty($q['sid'])) $re['shortId']   = $q['sid'];
            if (!empty($q['spx'])) $re['spiderX']   = $q['spx'];
            $out['streamSettings']['realitySettings'] = $re;
        } elseif ($security === 'tls') {
            $out['streamSettings']['tlsSettings'] = [
                'allowInsecure' => false,
                'fingerprint'   => $q['fp'] ?? 'chrome',
                'serverName'    => $q['sni'] ?? $host
            ];
        } else {
            $out['streamSettings']['security'] = 'none';
        }
    }

    remove_nulls($out);
    return $out;
}

/** ---------- VMESS parser ---------- */
function parse_vmess(string $uri): ?array {
    $b64  = substr($uri, 8);
    $json = base64_decode(strtr($b64, '-_', '+/'), true);
    if ($json === false) return null;
    $o = json_decode($json, true);
    if (!is_array($o) || empty($o['add']) || empty($o['id']) || empty($o['port'])) return null;

    $host = $o['add'];
    $port = intval($o['port']);
    $uuid = $o['id'];
    $tag  = $o['ps'] ?? 'vmess';

    $network  = strtolower($o['net'] ?? 'tcp');
    $security = !empty($o['tls']) ? 'tls' : 'none';

    $user = [
        'id'         => $uuid,
        'alterId'    => 0,
        'security'   => 'auto',
        'encryption' => 'none'
    ];

    $out = [
        'tag'       => $tag,
        'protocol'  => 'vmess',
        'settings'  => [
            'vnext' => [[
                'address' => $host,
                'port'    => $port,
                'users'   => [ $user ]
            ]]
        ],
        'streamSettings' => [
            'network'  => $network,
            'security' => $security
        ],
        'mux' => ['enabled' => false],
    ];

    if ($network === 'ws') {
        $path       = $o['path'] ?? '/';
        $hostHeader = $o['host'] ?? $host;
        $out['streamSettings']['wsSettings'] = [
            'path'    => $path,
            'headers' => ['Host' => $hostHeader],
        ];
    } elseif ($network === 'grpc') {
        $out['streamSettings']['grpcSettings'] = [
            'serviceName' => $o['path'] ?? 'grpc',
            'multiMode'   => false
        ];
    } elseif ($network === 'tcp') {
        $out['streamSettings']['tcpSettings'] = [ 'header' => ['type' => 'none'] ];
    } elseif ($network === 'xhttp') {
        $out['streamSettings']['network'] = 'xhttp';
    }

    if ($security === 'tls') {
        $out['streamSettings']['tlsSettings'] = [
            'allowInsecure' => false,
            'serverName'    => $o['sni'] ?? ($o['host'] ?? $host),
            'fingerprint'   => 'chrome'
        ];
    }

    remove_nulls($out);
    return $out;
}

/** ---------- TROJAN parser ---------- */
function parse_trojan(string $uri): ?array {
    $p = parse_url($uri);
    if (!$p || empty($p['host']) || empty($p['user'])) return null;

    $host = $p['host'];
    $port = isset($p['port']) ? intval($p['port']) : 443;
    $pass = $p['user'];
    $tag  = $p['fragment'] ?? 'trojan';

    $q = [];
    if (isset($p['query'])) parse_str($p['query'], $q);

    $network  = strtolower($q['type'] ?? 'tcp');
    $security = strtolower($q['security'] ?? 'tls');

    $out = [
        'tag'      => $tag,
        'protocol' => 'trojan',
        'settings' => [
            'servers' => [[
                'address'  => $host,
                'port'     => $port,
                'password' => $pass,
            ]]
        ],
        'streamSettings' => [
            'network'  => $network,
            'security' => $security ?: 'tls',
        ],
        'mux' => ['enabled' => false],
    ];

    if ($network === 'ws') {
        $path       = $q['path'] ?? '/';
        $hostHeader = $q['host'] ?? $host;
        $out['streamSettings']['wsSettings'] = [
            'path'    => $path,
            'headers' => ['Host' => $hostHeader],
        ];
    } elseif ($network === 'grpc') {
        $out['streamSettings']['grpcSettings'] = [
            'serviceName' => $q['serviceName'] ?? ($q['path'] ?? 'grpc'),
            'multiMode'   => false
        ];
    } elseif ($network === 'tcp') {
        $out['streamSettings']['tcpSettings'] = [ 'header' => ['type' => 'none'] ];
    } elseif ($network === 'xhttp') {
        $out['streamSettings']['network'] = 'xhttp';
    }

    if (($out['streamSettings']['security'] ?? '') === 'tls') {
        $out['streamSettings']['tlsSettings'] = [
            'allowInsecure' => false,
            'serverName'    => $q['sni'] ?? ($q['host'] ?? $host),
            'fingerprint'   => $q['fp'] ?? 'chrome',
        ];
        if (!empty($q['alpn'])) {
            $out['streamSettings']['tlsSettings']['alpn'] = explode(',', (string)$q['alpn']);
        }
    }

    remove_nulls($out);
    return $out;
}

/** ---------- SHADOWSOCKS parser ---------- */
function parse_ss(string $uri): ?array {
    $tag = 'ss';

    $fragPos = strpos($uri, '#');
    if ($fragPos !== false) {
        $tag = urldecode(substr($uri, $fragPos + 1)) ?: 'ss';
        $uri = substr($uri, 0, $fragPos);
    }

    $q = [];
    $qPos = strpos($uri, '?');
    if ($qPos !== false) {
        parse_str(substr($uri, $qPos + 1), $q);
        $uri = substr($uri, 0, $qPos);
    }

    $rest = substr($uri, 5);
    $rest = trim($rest);

    $method = ''; $pass = ''; $host = ''; $port = 0;

    if (strpos($rest, '@') === false) {
        $decoded = b64url_decode($rest);
        if ($decoded === '') return null;
        $at = strrpos($decoded, '@');
        if ($at === false) return null;
        $userinfo = substr($decoded, 0, $at);
        $server   = substr($decoded, $at + 1);
        $cp = strpos($userinfo, ':');
        if ($cp === false) return null;
        $method = substr($userinfo, 0, $cp);
        $pass   = substr($userinfo, $cp + 1);
        $hp = strrpos($server, ':');
        if ($hp === false) return null;
        $host = substr($server, 0, $hp);
        $port = (int)substr($server, $hp + 1);
    } else {
        $at = strrpos($rest, '@');
        if ($at === false) return null;
        $u = substr($rest, 0, $at);
        $server = substr($rest, $at + 1);
        $userinfo = b64url_decode($u);
        if ($userinfo === '') return null;
        $cp = strpos($userinfo, ':');
        if ($cp === false) return null;
        $method = substr($userinfo, 0, $cp);
        $pass   = substr($userinfo, $cp + 1);
        $hp = strrpos($server, ':');
        if ($hp === false) return null;
        $host = substr($server, 0, $hp);
        $port = (int)substr($server, $hp + 1);
    }

    if ($host === '' || $port <= 0 || $port > 65535 || $method === '' || $pass === '') return null;

    $out = [
        'tag'      => $tag,
        'protocol' => 'shadowsocks',
        'settings' => [
            'servers' => [[
                'address'  => $host,
                'port'     => $port,
                'method'   => $method,
                'password' => $pass,
                'level'    => 0,
            ]]
        ],
        'streamSettings' => [
            'network'  => 'tcp',
            'security' => 'none',
        ],
        'mux' => ['enabled' => false],
    ];

    if (!empty($q['plugin'])) $out['settings']['plugin'] = (string)$q['plugin'];

    remove_nulls($out);
    return $out;
}

/** ---------- normalize outbound ---------- */
function normalize_outbound(array $ob): array {
    if (empty($ob['protocol']) && !empty($ob['proto'])) $ob['protocol'] = $ob['proto'];
    if (!isset($ob['settings']) || !is_array($ob['settings'])) $ob['settings'] = [];
    if (!isset($ob['streamSettings']) || !is_array($ob['streamSettings'])) $ob['streamSettings'] = [];
    if (!isset($ob['streamSettings']['network'])) $ob['streamSettings']['network'] = 'tcp';
    if (!isset($ob['streamSettings']['security'])) $ob['streamSettings']['security'] = 'none';
    if (!isset($ob['mux']) || !is_array($ob['mux'])) $ob['mux'] = ['enabled' => false];
    if (!isset($ob['tag']) || $ob['tag'] === '') $ob['tag'] = $ob['protocol'] ?? 'server';
    remove_nulls($ob);
    return $ob;
}

/** ---------- sockopt + padding ---------- */
function add_sockopt_to_outbound(array $ob): array {
    if (!isset($ob['streamSettings']) || !is_array($ob['streamSettings'])) $ob['streamSettings'] = [];

    if (!isset($ob['streamSettings']['padding']) || !is_array($ob['streamSettings']['padding'])) {
        $ob['streamSettings']['padding'] = ['enabled' => false];
    }

    if (!isset($ob['streamSettings']['sockopt']) || !is_array($ob['streamSettings']['sockopt'])) {
        $ob['streamSettings']['sockopt'] = [];
    }
    $sock = &$ob['streamSettings']['sockopt'];

    if (!array_key_exists('tcpFastOpen', $sock))          $sock['tcpFastOpen']        = true;
    if (!array_key_exists('tcpNoDelay', $sock))           $sock['tcpNoDelay']         = true;
    if (!array_key_exists('mtu', $sock))                  $sock['mtu']                = 1280;
    if (!array_key_exists('tcpKeepAliveIdle', $sock))     $sock['tcpKeepAliveIdle']   = 20;
    if (!array_key_exists('tcpKeepAliveInterval', $sock)) $sock['tcpKeepAliveInterval'] = 5;
    if (!array_key_exists('tcpKeepAliveCount', $sock))    $sock['tcpKeepAliveCount']  = 3;
    if (!array_key_exists('tcpUserTimeout', $sock))       $sock['tcpUserTimeout']     = 150000;
    if (!array_key_exists('tcpMaxSeg', $sock))            $sock['tcpMaxSeg']          = 0;
    if (!array_key_exists('tcpBrutal', $sock))            $sock['tcpBrutal']          = true;
    if (!array_key_exists('tproxy', $sock))               $sock['tproxy']             = 'off';
    if (!array_key_exists('fragment', $sock) || !is_array($sock['fragment'])) {
        $sock['fragment'] = ['packets' => 3, 'length' => 500, 'interval' => 1];
    }

    return $ob;
}

function is_list_array(array $a): bool {
    $i = 0;
    foreach ($a as $k => $_) { if ($k !== $i) return false; $i++; }
    return true;
}

function outbound_is_valid(array $ob): bool {
    $proto = strtolower($ob['protocol'] ?? '');

    if ($proto === 'vless' || $proto === 'vmess') {
        if (empty($ob['settings']['vnext']) || !is_array($ob['settings']['vnext'])) return false;
        $first = $ob['settings']['vnext'][0] ?? null;
        if (!is_array($first)) return false;
        $addr = $first['address'] ?? '';
        $port = $first['port'] ?? 0;
        if (!is_string($addr) || $addr === '') return false;
        if (!is_int($port) && !ctype_digit((string)$port)) return false;
        $port = (int)$port;
        if ($port <= 0 || $port > 65535) return false;
    }

    if ($proto === 'trojan' || $proto === 'shadowsocks') {
        if (empty($ob['settings']['servers']) || !is_array($ob['settings']['servers'])) return false;
        $first = $ob['settings']['servers'][0] ?? null;
        if (!is_array($first)) return false;
        $addr = $first['address'] ?? '';
        $port = (int)($first['port'] ?? 0);
        if (!is_string($addr) || $addr === '') return false;
        if ($port <= 0 || $port > 65535) return false;
        if ($proto === 'trojan' && empty($first['password'])) return false;
        if ($proto === 'shadowsocks' && (empty($first['method']) || empty($first['password']))) return false;
    }

    if (!empty($ob['settings']) && is_array($ob['settings']) && is_list_array($ob['settings'])) return false;
    return true;
}

function extract_outbounds_from_json(string $json): array {
    $data = json_decode($json, true);
    if (!is_array($data)) return [];
    if (isset($data['outbounds']) && is_array($data['outbounds'])) {
        $outs = [];
        foreach ($data['outbounds'] as $ob) {
            if (!is_array($ob) || empty($ob['protocol'])) continue;
            $outs[] = normalize_outbound($ob);
        }
        return $outs;
    }
    if (isset($data['protocol'])) return [ normalize_outbound($data) ];
    return [];
}

function smart_extract_outbounds(string $raw): array {
    $t = trim($raw);

    $outs = extract_outbounds_from_json($t);
    if ($outs) return $outs;

    $first = strpos($t, '{'); $last = strrpos($t, '}');
    if ($first !== false && $last !== false && $last > $first) {
        $chunk = substr($t, $first, $last - $first + 1);
        $outs = extract_outbounds_from_json($chunk);
        if ($outs) return $outs;
    }

    $multi = [];
    if (preg_match_all('/\{(?:[^{}]|(?R))*\}/s', $t, $m)) {
        foreach ($m[0] as $blk) {
            $o = extract_outbounds_from_json($blk);
            if ($o) $multi = array_merge($multi, $o);
        }
        if ($multi) return $multi;
    }

    $col = [];
    foreach (preg_split('/\r?\n+/', $t) as $line) {
        $line = trim($line);
        if ($line === '') continue;

        if (is_vless($line)) { $ob = parse_vless($line); if ($ob) $col[] = normalize_outbound($ob); continue; }
        if (is_vmess($line)) { $ob = parse_vmess($line); if ($ob) $col[] = normalize_outbound($ob); continue; }
        if (is_trojan($line)) { $ob = parse_trojan($line); if ($ob) $col[] = normalize_outbound($ob); continue; }
        if (is_ss($line)) { $ob = parse_ss($line); if ($ob) $col[] = normalize_outbound($ob); continue; }

        if (($line[0] ?? '') === '{' || ($line[0] ?? '') === '[') {
            $o = extract_outbounds_from_json($line);
            if ($o) $col = array_merge($col, $o);
        }
    }
    return $col;
}

function unique_tag(string $base, array &$seen): string {
    $tag = $base;
    $i   = 1;
    while (isset($seen[$tag])) { $tag = $base . $i; $i++; }
    $seen[$tag] = true;
    return $tag;
}

function outbound_get_endpoint(array $ob): ?array {
    $addr = null; $port = null;

    if (!empty($ob['settings']['vnext'][0]['address'])) {
        $addr = $ob['settings']['vnext'][0]['address'];
        $port = $ob['settings']['vnext'][0]['port'] ?? null;
    } elseif (!empty($ob['settings']['servers'][0]['address'])) {
        $addr = $ob['settings']['servers'][0]['address'];
        $port = $ob['settings']['servers'][0]['port'] ?? null;
    }

    if ($addr === null || $port === null) return null;

    $security = $ob['streamSettings']['security'] ?? 'none';

    $sni = null;
    if (!empty($ob['streamSettings']['tlsSettings']['serverName'])) $sni = $ob['streamSettings']['tlsSettings']['serverName'];
    elseif (!empty($ob['streamSettings']['realitySettings']['serverName'])) $sni = $ob['streamSettings']['realitySettings']['serverName'];

    return [
        'host'     => $addr,
        'port'     => (int)$port,
        'security' => strtolower((string)$security),
        'sni'      => $sni,
    ];
}

function outbound_health(array $ob, int $timeout): array {
    $res = [
        'ok'      => false,
        'dns_ok'  => false,
        'tcp_ok'  => false,
        'tls_ok'  => null,
        'endpoint'=> null,
        'rtt_ms'  => null,
        'tries'   => 0,
    ];

    $ep = outbound_get_endpoint($ob);
    if ($ep === null) return $res;
    $res['endpoint'] = $ep;

    $host     = $ep['host'];
    $port     = $ep['port'];
    $security = $ep['security'];
    $sni      = $ep['sni'] ?: $host;

    $ip = $host;
    if (!filter_var($host, FILTER_VALIDATE_IP)) {
        $ip = @gethostbyname($host);
        $res['tries']++;
        if (empty($ip) || $ip === $host) return $res;
        $res['dns_ok'] = true;
    } else {
        $res['dns_ok'] = true;
    }

    $tcpStart = microtime(true);
    $errno = 0; $errstr = '';
    $sock = @fsockopen($ip, $port, $errno, $errstr, $timeout);
    $res['tries']++;
    if (!$sock) {
        $sock = @fsockopen($ip, $port, $errno, $errstr, $timeout);
        $res['tries']++;
        if (!$sock) return $res;
    }
    $tcpRtt = (microtime(true) - $tcpStart) * 1000.0;
    $res['tcp_ok'] = true;
    $res['rtt_ms'] = (int)round($tcpRtt);
    @fclose($sock);

    if ($security === 'tls' || $security === 'reality') {
        $tlsCtx = stream_context_create([
            'ssl' => [
                'verify_peer'      => false,
                'verify_peer_name' => false,
                'SNI_enabled'      => true,
                'peer_name'        => $sni,
            ]
        ]);

        $tlsStart = microtime(true);
        $tls = @stream_socket_client(
            "ssl://{$ip}:{$port}",
            $errno,
            $errstr,
            $timeout,
            STREAM_CLIENT_CONNECT,
            $tlsCtx
        );
        $res['tries']++;
        if (!$tls) { $res['tls_ok'] = false; return $res; }
        $tlsRtt = (microtime(true) - $tlsStart) * 1000.0;
        $res['tls_ok'] = true;
        if ($res['rtt_ms'] === null || $tlsRtt < $res['rtt_ms']) $res['rtt_ms'] = (int)round($tlsRtt);
        @fclose($tls);
    }

    $res['ok'] = $res['dns_ok'] && $res['tcp_ok'] && ($res['tls_ok'] !== false);
    return $res;
}

/** ---------- template ---------- */
function build_template(): array {
    global $ENABLE_FAKEDNS, $PROBE_URLS;

    $levels = (object)[];
    $levels->{'0'} = ["handshake"=>12,"connIdle"=>3600,"uplinkOnly"=>0,"downlinkOnly"=>0];

    $probe = $PROBE_URLS[array_rand($PROBE_URLS)];

    $template = [
        "log" => ["loglevel"=>"warning","dnsLog"=>false],
        "stats" => (object)[],
        "api" => ["tag"=>"api","services"=>["StatsService"]],
        "policy" => [
            "system" => ["statsInboundUplink"=>true,"statsInboundDownlink"=>true],
            "levels" => $levels,
        ],
        "dns" => [
            "queryStrategy"          => "UseIP",
            "disableCache"           => false,
            "disableFallback"        => false,
            "disableFallbackIfMatch" => true,
            "hosts" => [
                "domain:googleapis.cn"         => "googleapis.com",
                "cloudflare-dns.com"           => "1.1.1.1",
                "security.cloudflare-dns.com"  => "1.1.1.1",
                "dns.google"                   => "8.8.8.8",
            ],
            "servers" => [
                ["address"=>"https://1.1.1.1/dns-query","outboundTag"=>"dns-out"],
                ["address"=>"https://8.8.8.8/dns-query","outboundTag"=>"dns-out"],
                ["address"=>"https://9.9.9.9/dns-query","outboundTag"=>"dns-out"],
                ["address"=>"1.1.1.1","outboundTag"=>"dns-out","timeoutMs"=>800,"serveStale"=>true,"serveExpiredTTL"=>30],
                ["address"=>"8.8.8.8","outboundTag"=>"dns-out","timeoutMs"=>800,"serveStale"=>true],
            ],
        ],
        "observatory" => ["subjectSelector"=>["auto-group"],"probeURL"=>$probe,"enableConcurrency"=>true],
        "inbounds" => [
            [
                "tag"=>"socks","port"=>10808,"listen"=>"127.0.0.1","protocol"=>"socks",
                "sniffing"=>["enabled"=>true,"routeOnly"=>false,"destOverride"=>["http","tls","quic"]],
                "settings"=>["auth"=>"noauth","udp"=>true,"ip"=>"127.0.0.1","userLevel"=>0],
            ],
            [
                "tag"=>"dns-in","port"=>10853,"listen"=>"127.0.0.1","protocol"=>"dokodemo-door",
                "settings"=>["network"=>"tcp,udp","followRedirect"=>false,"address"=>"1.1.1.1","port"=>53],
            ],
        ],
        "outbounds" => [
            ["tag"=>"direct","protocol"=>"freedom","settings"=>["domainStrategy"=>"UseIP"]],
            ["tag"=>"block","protocol"=>"blackhole","settings"=>(object)[]],
            ["tag"=>"dns-out","protocol"=>"dns","settings"=>(object)[]],
        ],
        "routing" => [
            "domainStrategy"=>"IPIfNonMatch",
            "domainMatcher"=>"hybrid",
            "balancers" => [
                ["tag"=>"auto-group","selector"=>[],"strategy"=>["type"=>"leastPing"]],
            ],
            "rules" => [
                ["type"=>"field","inboundTag"=>["dns-in"],"outboundTag"=>"dns-out"],
                ["type"=>"field","protocol"=>["bittorrent"],"outboundTag"=>"block"],
                ["type"=>"field","port"=>"6881-6999","outboundTag"=>"block"],
                ["type"=>"field","ip"=>["10.0.0.0/8","172.16.0.0/12","192.168.0.0/16","127.0.0.0/8"],"outboundTag"=>"direct"],
                ["type"=>"field","ip"=>["1.1.1.1"],"port"=>"53","outboundTag"=>"dns-out"],
                ["type"=>"field","domain"=>["dns.google","cloudflare-dns.com","security.cloudflare-dns.com"],"network"=>"tcp","balancerTag"=>"auto-group"],
                ["type"=>"field","network"=>"tcp,udp","balancerTag"=>"auto-group"],
            ],
        ],
    ];

    if ($ENABLE_FAKEDNS) {
        $template['fakedns'] = [[ "ipPool"=>"198.18.0.0/15","poolSize"=>10000 ]];
        if (isset($template['inbounds'][0]['sniffing']['destOverride']) && is_array($template['inbounds'][0]['sniffing']['destOverride'])) {
            if (!in_array('fakedns', $template['inbounds'][0]['sniffing']['destOverride'], true)) {
                $template['inbounds'][0]['sniffing']['destOverride'][] = 'fakedns';
            }
        }
    }

    return $template;
}

function addObservatoryToConfig(array &$jsonConfig, array $preferredSelectors = []) {
    if (!isset($jsonConfig['outbounds']) || !is_array($jsonConfig['outbounds'])) return;

    $selectors = [];
    foreach ($jsonConfig['outbounds'] as $ob) {
        if (!isset($ob['tag'])) continue;
        $tag = $ob['tag'];
        if (in_array($tag, ['direct','dns-out','block'], true)) continue;
        $selectors[] = $tag;
    }
    $selectors = array_values(array_unique($selectors));

    if (!empty($preferredSelectors)) {
        $preferredSelectors = array_values(array_unique($preferredSelectors));
        $intersection       = array_values(array_intersect($preferredSelectors, $selectors));
        $selectors          = !empty($intersection) ? $intersection : $preferredSelectors;
    }

    if (empty($selectors)) return;

    $jsonConfig['observatory']['subjectSelector'] = $selectors;
    $jsonConfig['observatory']['enableConcurrency'] = true;
}

function forceAllowInsecure(&$template) {
    if (!isset($template['outbounds']) || !is_array($template['outbounds'])) return;

    foreach ($template['outbounds'] as &$ob) {
        if (!isset($ob['streamSettings']) || !is_array($ob['streamSettings'])) continue;
        $ss =& $ob['streamSettings'];
        $security = $ss['security'] ?? '';

        if ($security === 'tls') {
            if (!isset($ss['tlsSettings']) || !is_array($ss['tlsSettings'])) $ss['tlsSettings'] = [];
            $ss['tlsSettings']['allowInsecure'] = true;
        }

        if ($security === 'reality') {
            if (!isset($ss['realitySettings']) || !is_array($ss['realitySettings'])) $ss['realitySettings'] = [];
            $ss['realitySettings']['allowInsecure'] = true;
        }
    }
}

/* ===================== FIXERS (false=>fix, true=>keep old) ===================== */

function enforce_structural_consistency(array $ob): array {
    global $FEATURES;
    if (!empty($FEATURES['allow_structural_mix'])) return $ob;

    if (!isset($ob['streamSettings']) || !is_array($ob['streamSettings'])) return $ob;

    $net = strtolower((string)($ob['streamSettings']['network'] ?? 'tcp'));
    $sec = strtolower((string)($ob['streamSettings']['security'] ?? 'none'));

    if ($net !== 'ws'    && isset($ob['streamSettings']['wsSettings']))   unset($ob['streamSettings']['wsSettings']);
    if ($net !== 'grpc'  && isset($ob['streamSettings']['grpcSettings'])) unset($ob['streamSettings']['grpcSettings']);
    if ($net !== 'tcp'   && isset($ob['streamSettings']['tcpSettings']))  unset($ob['streamSettings']['tcpSettings']);
    if ($net !== 'xhttp' && isset($ob['streamSettings']['xhttpSettings']))unset($ob['streamSettings']['xhttpSettings']);

    if ($sec !== 'tls'     && isset($ob['streamSettings']['tlsSettings']))     unset($ob['streamSettings']['tlsSettings']);
    if ($sec !== 'reality' && isset($ob['streamSettings']['realitySettings'])) unset($ob['streamSettings']['realitySettings']);

    return $ob;
}

function sanitize_ranges_outbound(array $ob): array {
    global $FEATURES;
    if (!empty($FEATURES['keep_invalid_ranges'])) return $ob;

    // ports
    $ep = outbound_get_endpoint($ob);
    if ($ep && ($ep['port'] <= 0 || $ep['port'] > 65535)) {
        // اگر بیرون رنج شد، این اوتباند را بی‌اثر می‌کنیم با حذف پورت از endpoint (در نهایت outbound_is_valid می‌اندازدش)
        // ولی اینجا ترجیح: اصلاح به 443
        if (!empty($ob['settings']['vnext'][0])) $ob['settings']['vnext'][0]['port'] = 443;
        if (!empty($ob['settings']['servers'][0])) $ob['settings']['servers'][0]['port'] = 443;
    }

    // sockopt
    if (isset($ob['streamSettings']['sockopt']) && is_array($ob['streamSettings']['sockopt'])) {
        $s =& $ob['streamSettings']['sockopt'];
        if (isset($s['mtu'])) $s['mtu'] = clamp_int($s['mtu'], 576, 9000, 1280);
        if (isset($s['tcpKeepAliveIdle'])) $s['tcpKeepAliveIdle'] = clamp_int($s['tcpKeepAliveIdle'], 1, 600, 20);
        if (isset($s['tcpKeepAliveInterval'])) $s['tcpKeepAliveInterval'] = clamp_int($s['tcpKeepAliveInterval'], 1, 120, 5);
        if (isset($s['tcpKeepAliveCount'])) $s['tcpKeepAliveCount'] = clamp_int($s['tcpKeepAliveCount'], 1, 10, 3);
        if (isset($s['tcpUserTimeout'])) $s['tcpUserTimeout'] = clamp_int($s['tcpUserTimeout'], 1000, 600000, 150000);
        if (isset($s['fragment']) && is_array($s['fragment'])) {
            $s['fragment']['packets']  = clamp_int($s['fragment']['packets'] ?? 3, 1, 16, 3);
            $s['fragment']['length']   = clamp_int($s['fragment']['length'] ?? 500, 1, 1500, 500);
            $s['fragment']['interval'] = clamp_int($s['fragment']['interval'] ?? 1, 0, 50, 1);
        }
    }

    return $ob;
}

function apply_feature_flags_to_outbound(array $ob): array {
    global $FEATURES;

    // xhttp
    if (empty($FEATURES['enable_xhttp'])) {
        if (isset($ob['streamSettings']['network']) && strtolower((string)$ob['streamSettings']['network']) === 'xhttp') {
            $ob['streamSettings']['network'] = 'tcp';
        }
        if (isset($ob['streamSettings']['xhttpSettings'])) unset($ob['streamSettings']['xhttpSettings']);
    }

    // mux
    if (empty($FEATURES['enable_mux'])) {
        if (isset($ob['mux'])) unset($ob['mux']);
    } else {
        if (!isset($ob['mux']) || !is_array($ob['mux'])) $ob['mux'] = ['enabled' => false];
        if (!array_key_exists('enabled', $ob['mux'])) $ob['mux']['enabled'] = false;
    }

    // padding
    if (empty($FEATURES['enable_padding'])) {
        if (isset($ob['streamSettings']['padding'])) unset($ob['streamSettings']['padding']);
    } else {
        if (!isset($ob['streamSettings'])) $ob['streamSettings'] = [];
        if (!isset($ob['streamSettings']['padding']) || !is_array($ob['streamSettings']['padding'])) $ob['streamSettings']['padding'] = ['enabled' => false];
        if (!array_key_exists('enabled', $ob['streamSettings']['padding'])) $ob['streamSettings']['padding']['enabled'] = false;
    }

    // sockopt + fragment
    if (empty($FEATURES['enable_sockopt'])) {
        if (isset($ob['streamSettings']['sockopt'])) unset($ob['streamSettings']['sockopt']);
    } else {
        if (!isset($ob['streamSettings'])) $ob['streamSettings'] = [];
        if (!isset($ob['streamSettings']['sockopt']) || !is_array($ob['streamSettings']['sockopt'])) $ob['streamSettings']['sockopt'] = [];
        if (empty($FEATURES['enable_fragment'])) {
            if (isset($ob['streamSettings']['sockopt']['fragment'])) unset($ob['streamSettings']['sockopt']['fragment']);
        } else {
            if (!isset($ob['streamSettings']['sockopt']['fragment']) || !is_array($ob['streamSettings']['sockopt']['fragment'])) {
                $ob['streamSettings']['sockopt']['fragment'] = ['packets'=>3,'length'=>500,'interval'=>1];
            }
        }
    }

    // metadata per outbound
    if (empty($FEATURES['enable_metadata'])) {
        if (isset($ob['__health'])) unset($ob['__health']);
        if (isset($ob['__ping'])) unset($ob['__ping']);
    }

    // جلوگیری از میکس (اگر لازم باشد)
    $ob = enforce_structural_consistency($ob);

    // رنج‌ها
    $ob = sanitize_ranges_outbound($ob);

    return $ob;
}

function fix_dns_doh_and_flags(array &$template): void {
    global $FEATURES, $ENABLE_FAKEDNS;

    // disableFallbackIfMatch
    if (empty($FEATURES['keep_disableFallbackIfMatch'])) {
        if (!isset($template['dns']) || !is_array($template['dns'])) $template['dns'] = [];
        // اجباری: بولین
        $template['dns']['disableFallbackIfMatch'] = true;
        if (!isset($template['dns']['disableFallback'])) $template['dns']['disableFallback'] = false;
        if (!isset($template['dns']['disableCache'])) $template['dns']['disableCache'] = false;
        if (!isset($template['dns']['queryStrategy'])) $template['dns']['queryStrategy'] = 'UseIP';
    }

    // DoH/DNS
    if (empty($FEATURES['keep_bad_dns_doh'])) {
        if (!isset($template['dns']) || !is_array($template['dns'])) return;

        if (!isset($template['dns']['servers']) || !is_array($template['dns']['servers'])) {
            $template['dns']['servers'] = [];
        }

        $fixed = [];
        foreach ($template['dns']['servers'] as $srv) {
            // قبول: string یا array
            if (is_string($srv)) {
                $addr = trim($srv);
                if ($addr === '') continue;
                $fixed[] = $addr;
                continue;
            }
            if (!is_array($srv)) continue;

            $addr = isset($srv['address']) ? trim((string)$srv['address']) : '';
            if ($addr === '') continue;

            // اگر https://.../dns-query بود: باید outboundTag داشته باشد (dns-out)
            if (stripos($addr, 'https://') === 0 || stripos($addr, 'https+local://') === 0) {
                if (empty($srv['outboundTag'])) $srv['outboundTag'] = 'dns-out';
            }

            // timeoutMs داخل رنج
            if (isset($srv['timeoutMs'])) $srv['timeoutMs'] = clamp_int($srv['timeoutMs'], 100, 10000, 800);

            $fixed[] = $srv;
        }

        // اگر خالی شد، حداقل‌ها را برگردان
        if (empty($fixed)) {
            $fixed = [
                ["address"=>"https://1.1.1.1/dns-query","outboundTag"=>"dns-out"],
                ["address"=>"https://8.8.8.8/dns-query","outboundTag"=>"dns-out"],
                ["address"=>"1.1.1.1","outboundTag"=>"dns-out","timeoutMs"=>800,"serveStale"=>true,"serveExpiredTTL"=>30],
            ];
        }

        $template['dns']['servers'] = $fixed;
    }

    // FakeDNS
    if (empty($FEATURES['keep_fakedns_logic'])) {
        // اگر ENABLE_FAKEDNS خاموش است، کل فیک‌دی‌ان‌اس و destOverride را جمع کن
        if (!$ENABLE_FAKEDNS) {
            if (isset($template['fakedns'])) unset($template['fakedns']);
            if (isset($template['inbounds'][0]['sniffing']['destOverride']) && is_array($template['inbounds'][0]['sniffing']['destOverride'])) {
                $template['inbounds'][0]['sniffing']['destOverride'] = array_values(array_filter(
                    $template['inbounds'][0]['sniffing']['destOverride'],
                    fn($x) => $x !== 'fakedns'
                ));
            }
        } else {
            // اگر روشن است، مطمئن شو ساختار صحیح است
            if (!isset($template['fakedns']) || !is_array($template['fakedns']) || empty($template['fakedns'])) {
                $template['fakedns'] = [[ "ipPool"=>"198.18.0.0/15","poolSize"=>10000 ]];
            }
            if (isset($template['inbounds'][0]['sniffing']['destOverride']) && is_array($template['inbounds'][0]['sniffing']['destOverride'])) {
                if (!in_array('fakedns', $template['inbounds'][0]['sniffing']['destOverride'], true)) {
                    $template['inbounds'][0]['sniffing']['destOverride'][] = 'fakedns';
                }
            }
        }
    }
}

function rebuild_tag_index(array $template): array {
    $tags = [];
    if (!isset($template['outbounds']) || !is_array($template['outbounds'])) return $tags;
    foreach ($template['outbounds'] as $ob) {
        if (!is_array($ob)) continue;
        $t = (string)($ob['tag'] ?? '');
        if ($t !== '') $tags[$t] = true;
    }
    return $tags;
}

function replace_tags_everywhere(array &$template, array $map): void {
    // replace in routing balancers selectors
    if (isset($template['routing']['balancers']) && is_array($template['routing']['balancers'])) {
        foreach ($template['routing']['balancers'] as &$b) {
            if (isset($b['selector']) && is_array($b['selector'])) {
                foreach ($b['selector'] as $i => $t) {
                    if (isset($map[$t])) $b['selector'][$i] = $map[$t];
                }
            }
        }
    }

    // replace in observatory subjectSelector
    if (isset($template['observatory']['subjectSelector']) && is_array($template['observatory']['subjectSelector'])) {
        foreach ($template['observatory']['subjectSelector'] as $i => $t) {
            if (isset($map[$t])) $template['observatory']['subjectSelector'][$i] = $map[$t];
        }
    }

    // replace in routing rules outboundTag
    if (isset($template['routing']['rules']) && is_array($template['routing']['rules'])) {
        foreach ($template['routing']['rules'] as &$r) {
            if (isset($r['outboundTag']) && isset($map[$r['outboundTag']])) $r['outboundTag'] = $map[$r['outboundTag']];
            if (isset($r['balancerTag']) && isset($map[$r['balancerTag']])) {
                // balancerTag معمولاً auto-group/auto-ads است، نه tag اوتباند
                // اینجا عمداً دست نمی‌زنیم مگر کاربر خودش map بدهد (که نمی‌دهد)
            }
        }
    }
}

function fix_tags_selectors_and_duplicates(array &$template): void {
    global $FEATURES;

    if (!isset($template['outbounds']) || !is_array($template['outbounds'])) return;

    // 1) اگر اجازه تگ تکراری نداریم: rename و update references
    if (empty($FEATURES['allow_duplicate_tags'])) {
        $seen = [];
        $map  = []; // old->new (فقط وقتی rename شود)
        foreach ($template['outbounds'] as $i => $ob) {
            if (!is_array($ob)) continue;
            $tag = (string)($ob['tag'] ?? '');
            if ($tag === '') $tag = 'server';
            // sanitize tag name اگر خراب بود
            if (!is_valid_tag_name($tag)) {
                $tag = preg_replace('~[^a-zA-Z0-9_\-]~', '_', $tag);
                if ($tag === '') $tag = 'server';
            }
            $base = $tag;
            if (isset($seen[$tag])) {
                $j = 2;
                while (isset($seen[$base . '_' . $j])) $j++;
                $newTag = $base . '_' . $j;
                $map[$tag] = $newTag; // این old->new برای یک tag مبهم است (چون چندتا تکرار ممکن است)
                // برای جلوگیری از بهم‌ریختگی، rename را مستقیم روی خود outbound اعمال می‌کنیم، و map را دقیق‌تر می‌سازیم:
                // ولی چون old tag چندبار تکرار شده، map یکسان کافی نیست.
                // پس روش بهتر: فقط همان outbound فعلی را rename کنیم و سپس replace همه جا بر اساس map خاص (old->new) خطرناک است.
                // راه امن: rename فقط روی outbounds و بعد selectors را از نو بسازیم (نه replace).
                $tag = $newTag;
            }
            $seen[$tag] = true;
            $template['outbounds'][$i]['tag'] = $tag;
        }

        // چون map دقیق برای تگ تکراری چندگانه امن نیست، به جای replace_tags_everywhere:
        // selectors و subjectSelector را از نو (فقط از روی outbounds موجود) می‌سازیم.
        if (empty($FEATURES['keep_bad_tags_selectors'])) {
            $tags = array_keys(rebuild_tag_index($template));
            // remove reserved
            $tags = array_values(array_filter($tags, fn($t)=>!in_array($t,['direct','dns-out','block'],true)));

            // balancer auto-group selector
            if (isset($template['routing']['balancers'][0]) && is_array($template['routing']['balancers'][0])) {
                $template['routing']['balancers'][0]['selector'] = $tags;
                if (!isset($template['routing']['balancers'][0]['strategy']) || !is_array($template['routing']['balancers'][0]['strategy'])) {
                    $template['routing']['balancers'][0]['strategy'] = ["type"=>"leastPing"];
                }
            }

            // observatory subjectSelector
            if (!isset($template['observatory']) || !is_array($template['observatory'])) $template['observatory'] = [];
            $template['observatory']['subjectSelector'] = $tags;
            if (!isset($template['observatory']['enableConcurrency'])) $template['observatory']['enableConcurrency'] = true;
        }
    }

    // 2) اگر keep_bad_tags_selectors=false => پاکسازی selectorها از تگ‌های ناموجود و تکراری
    if (empty($FEATURES['keep_bad_tags_selectors'])) {
        $exists = rebuild_tag_index($template);

        // observatory
        if (isset($template['observatory']['subjectSelector']) && is_array($template['observatory']['subjectSelector'])) {
            $new = [];
            foreach ($template['observatory']['subjectSelector'] as $t) {
                $t = (string)$t;
                if ($t === '') continue;
                if (!isset($exists[$t])) continue;
                if (in_array($t, ['direct','dns-out','block'], true)) continue;
                $new[$t] = true;
            }
            $template['observatory']['subjectSelector'] = array_values(array_keys($new));
        }

        // routing balancers
        if (isset($template['routing']['balancers']) && is_array($template['routing']['balancers'])) {
            foreach ($template['routing']['balancers'] as &$b) {
                if (!isset($b['selector']) || !is_array($b['selector'])) continue;
                $new = [];
                foreach ($b['selector'] as $t) {
                    $t = (string)$t;
                    if ($t === '') continue;
                    if (!isset($exists[$t])) continue;
                    if (in_array($t, ['direct','dns-out','block'], true)) continue;
                    $new[$t] = true;
                }
                $b['selector'] = array_values(array_keys($new));
                if (!isset($b['strategy']) || !is_array($b['strategy'])) $b['strategy'] = ["type"=>"leastPing"];
                if (($b['strategy']['type'] ?? '') !== 'leastPing') $b['strategy']['type'] = 'leastPing';
            }
        }
    }
}

function fix_observatory_and_leastping(array &$template): void {
    global $FEATURES;
    if (!empty($FEATURES['keep_observatory_leastping'])) return;

    // تضمین وجود observatory و leastPing
    if (!isset($template['observatory']) || !is_array($template['observatory'])) $template['observatory'] = [];
    if (!isset($template['observatory']['enableConcurrency'])) $template['observatory']['enableConcurrency'] = true;

    if (!isset($template['routing']) || !is_array($template['routing'])) $template['routing'] = [];
    if (!isset($template['routing']['balancers']) || !is_array($template['routing']['balancers']) || empty($template['routing']['balancers'])) {
        $template['routing']['balancers'] = [[ "tag"=>"auto-group","selector"=>[],"strategy"=>["type"=>"leastPing"] ]];
    }

    // leastPing اجباری
    foreach ($template['routing']['balancers'] as &$b) {
        if (!isset($b['strategy']) || !is_array($b['strategy'])) $b['strategy'] = ["type"=>"leastPing"];
        if (($b['strategy']['type'] ?? '') !== 'leastPing') $b['strategy']['type'] = 'leastPing';
    }

    // اگر selector خالی بود، از outbounds پر کن
    $exists = rebuild_tag_index($template);
    $candidates = array_keys($exists);
    $candidates = array_values(array_filter($candidates, fn($t)=>!in_array($t,['direct','dns-out','block'],true)));

    if (isset($template['routing']['balancers'][0]['selector']) && is_array($template['routing']['balancers'][0]['selector'])) {
        if (empty($template['routing']['balancers'][0]['selector'])) {
            $template['routing']['balancers'][0]['selector'] = $candidates;
        }
    } else {
        $template['routing']['balancers'][0]['selector'] = $candidates;
    }

    // observatory subjectSelector هم با همین
    if (!isset($template['observatory']['subjectSelector']) || !is_array($template['observatory']['subjectSelector']) || empty($template['observatory']['subjectSelector'])) {
        $template['observatory']['subjectSelector'] = $candidates;
    }
}

function apply_feature_flags_to_template(array &$template): void {
    global $FEATURES;

    if (empty($FEATURES['enable_metadata'])) {
        if (isset($template['__meta'])) unset($template['__meta']);
    }

    if (isset($template['outbounds']) && is_array($template['outbounds'])) {
        foreach ($template['outbounds'] as $i => $ob) {
            if (!is_array($ob)) continue;
            $template['outbounds'][$i] = apply_feature_flags_to_outbound($ob);
        }
    }

    // بعد از تغییر outbounds، اصلاحات ساختاری/تگ/سلکتور/دی‌ان‌اس/…
    fix_dns_doh_and_flags($template);
    fix_tags_selectors_and_duplicates($template);
    fix_observatory_and_leastping($template);
}

/** ---------- main aggregation (normal + ads) ---------- */

$ALLOWED_PROTOCOLS = ['vless', 'vmess', 'trojan', 'shadowsocks'];

$ADS_LINKS = $ADS_LINKS ?? [];

$servers    = [];
$adsServers = [];
$seenTags   = ['direct' => true, 'block' => true, 'dns-out' => true];
$mainTags   = [];
$adsTags    = [];

shuffle($LINKS);

foreach ($LINKS as $url) {
    $attempts = 0;
    while ($attempts < 2) {
        $attempts++;

        $raw = http_get_nocache($url, $HTTP_TIMEOUT);
        if ($raw === null) continue;

        $added = smart_extract_outbounds($raw);
        if (!$added) continue;

        foreach ($added as $ob) {
            $proto = strtolower($ob['protocol'] ?? '');
            if ($proto === '' || !in_array($proto, $ALLOWED_PROTOCOLS, true)) continue;
            if (!outbound_is_valid($ob)) continue;

            $ob = add_sockopt_to_outbound($ob);

            $health = outbound_health($ob, $HEALTH_TIMEOUT);
            if (!$health['ok']) continue;

            $base      = $ob['tag'] ?? ($proto . '_');
            $ob['tag'] = unique_tag(preg_replace('~[^a-zA-Z0-9_\-]~', '_', $base . '_'), $seenTags);

            $ob['__health'] = $health;
            $ob['__ping']   = $health['rtt_ms'] ?? -1;

            $servers[]  = $ob;
            $mainTags[] = $ob['tag'];
        }
        break;
    }
}

if (!empty($ADS_LINKS)) {
    shuffle($ADS_LINKS);
    foreach ($ADS_LINKS as $url) {
        $attempts = 0;
        while ($attempts < 2) {
            $attempts++;
            $raw = http_get_nocache($url, $HTTP_TIMEOUT);
            if ($raw === null) continue;

            $added = smart_extract_outbounds($raw);
            if (!$added) continue;

            foreach ($added as $ob) {
                $proto = strtolower($ob['protocol'] ?? '');
                if ($proto === '' || !in_array($proto, $ALLOWED_PROTOCOLS, true)) continue;
                if (!outbound_is_valid($ob)) continue;

                $ob = add_sockopt_to_outbound($ob);

                $health = outbound_health($ob, $HEALTH_TIMEOUT);
                if (!$health['ok']) continue;

                $base      = $ob['tag'] ?? ($proto . '_');
                $ob['tag'] = unique_tag(preg_replace('~[^a-zA-Z0-9_\-]~', '_', $base . '_'), $seenTags);

                $ob['__health'] = $health;
                $ob['__ping']   = $health['rtt_ms'] ?? -1;

                $adsServers[] = $ob;
                $adsTags[]    = $ob['tag'];
            }
            break;
        }
    }
}

$template = build_template();

foreach ($servers as $ob) $template['outbounds'][] = $ob;
foreach ($adsServers as $ob) $template['outbounds'][] = $ob;

forceAllowInsecure($template);

/* تنظیم بالانسرها */
if (!empty($mainTags)) {
    $template['routing']['balancers'][0]['selector'] = $mainTags;
} else {
    $template['routing']['balancers'][0]['selector'] = $adsTags;
}

if (!empty($adsTags)) {
    $template['routing']['balancers'][] = [
        'tag'      => 'auto-ads',
        'selector' => $adsTags,
        'strategy' => ['type' => 'leastPing'],
    ];
}

/* observatory برای همه */
$allTags = array_values(array_unique(array_merge($mainTags, $adsTags)));
addObservatoryToConfig($template, $allTags);

$template['__meta'] = [
    'health'  => ['requireAll' => CHECK_REQUIRE_ALL],
    'network' => ['ipv6Blocked' => false],
];

/* قوانین روتینگ گوگل/ادز */
if (!empty($adsTags)) {
    $googleDomains = [
        "googleads.g.doubleclick.net",
        "geosite:category-ads-all",
        "geosite:google",
        "pagead2.googlesyndication.com",
        "tpc.googlesyndication.com",
        "www.googleadservices.com",
        "adservice.google.com",
        "adservice.google.*",
        "stats.g.doubleclick.net",
        "www.googletagmanager.com",
        "www.google-analytics.com",
        "ssl.google-analytics.com",
        "app-measurement.com",
        "www.googletagservices.com",
        "fcm.googleapis.com",
        "firebaseinstallations.googleapis.com",
        "firebaseremoteconfig.googleapis.com",
        "crashlytics.googleapis.com",
        "play.google.com",
        "play.googleapis.com",
        "android.googleapis.com",
        "android.clients.google.com",
        "www.google.com",
        "www.googleapis.com",
        "oauth2.googleapis.com",
        "accounts.google.com",
        "gstaticadssl.l.google.com",
        "fonts.gstatic.com",
        "www.gstatic.com",
        "clients3.google.com",
        "clients4.google.com",
        "clients5.google.com",
        "www.youtube.com",
        "m.youtube.com",
        "youtubei.googleapis.com",
        "i.ytimg.com",
        "s.youtube.com",
        "yt3.ggpht.com",
        "googlevideo.com",
        "r.googlevideo.com",
        "google.com",
        "googleapis.com",
        "googleusercontent.com",
        "gvt1.com",
        "gvt2.com",
        "gstatic.com",
    ];

    $rules    = $template['routing']['rules'] ?? [];
    $newRules = [];

    $newRules[] = [
        "type"        => "field",
        "network"     => "tcp,udp",
        "domain"      => $googleDomains,
        "balancerTag" => "auto-ads",
    ];

    foreach ($rules as $rule) $newRules[] = $rule;
    $template['routing']['rules'] = $newRules;
}

/* ====== APPLY FEATURE FLAGS (حذف/اصلاح واقعی از خروجی) ====== */
apply_feature_flags_to_template($template);

$outJson = json_encode($template, JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE);

if ($SAVE_OUTPUT_HISTORY) {
    cleanup_history_files($HISTORY_DIR, $HISTORY_TTL_SECONDS);
    save_output_line($HISTORY_DIR, $outJson);
}

echo $outJson;