<?php
/**
 * Nexy.php — v2ray/Xray config aggregator (NO cache, strict health, keep outbounds AS-IS)
 * - No caching (remote & health)
 * - Keep input outbounds as-is; only add base sockopt (mark/tcpNoDelay/tcpKeepAliveIdle)
 * - IPv6 fully blocked in JSON routing; DNS DoH over IPv4
 * - Health: sanity + TCP connect (IPv4) + URL probes (require ALL)
 * - Failing outbounds stay in output with "__ping": -1
 * PHP 7.4+
 */

/* ====== USER SETTINGS ====== */
$LINKS = [
  'https://fitn1.ir/Api/ConfigTelegram/Configtelegram.php',
];
$HTTP_TIMEOUT    = 10;  // remote fetch timeout (s)
$HEALTH_TIMEOUT  = 4;   // per-URL health-check timeout (s)
define('CHECK_REQUIRE_ALL', true); // همهٔ URLها باید موفق شوند
/* =========================== */

header('Content-Type: application/json; charset=utf-8');

/** ---------- http (NO CACHE) ---------- */
function http_get_nocache(string $url, int $timeout): ?string {
  $ctx = stream_context_create([
    'http' => [
      'timeout' => $timeout,
      'ignore_errors' => true,
      'header' =>
        "Cache-Control: no-cache, no-store, must-revalidate\r\n" .
        "Pragma: no-cache\r\n" .
        "User-Agent: Mozilla/5.0\r\n" .
        "Connection: close\r\n",
    ],
    'ssl'  => [
      'verify_peer' => false,
      'verify_peer_name' => false,
    ],
  ]);
  $data = @file_get_contents($url, false, $ctx);
  return ($data === false || $data === '') ? null : $data;
}

/** ---------- proto detect/parse ---------- */
function is_vless(string $s): bool { return stripos($s, 'vless://') === 0; }
function is_vmess(string $s): bool { return stripos($s, 'vmess://') === 0; }

function parse_vless(string $uri): ?array {
  $p = parse_url($uri);
  if (!$p || empty($p['host']) || empty($p['user'])) return null;
  $host = $p['host'];
  $port = isset($p['port']) ? intval($p['port']) : 443;
  $uuid = $p['user'];
  $tag  = $p['fragment'] ?? 'server';
  $q = [];
  if (isset($p['query'])) parse_str($p['query'], $q);

  $network  = isset($q['type']) ? strtolower($q['type']) : 'tcp';
  $security = isset($q['security']) ? strtolower($q['security']) : 'none';

  $user = ['id'=>$uuid,'encryption'=>'none'];
  if (!empty($q['flow'])) $user['flow'] = $q['flow'];

  $out = [
    'tag' => $tag,
    'protocol' => 'vless',
    'settings' => [ 'vnext' => [[ 'address'=>$host, 'port'=>$port, 'users'=>[ $user ] ]] ],
    'streamSettings' => [ 'network'=>$network, 'security'=>$security ],
    'mux' => [ 'enabled' => false ],
  ];

  if ($network === 'ws') {
    $ws = ['path' => $q['path'] ?? '/', 'headers' => []];
    if (!empty($q['host'])) $ws['headers']['Host'] = $q['host'];
    $out['streamSettings']['wsSettings'] = $ws;
    if (!empty($q['sni'])) $out['streamSettings']['tlsSettings']['serverName'] = $q['sni'];
    if (!empty($q['alpn'])) $out['streamSettings']['tlsSettings']['alpn'] = explode(',', $q['alpn']);
    if (!empty($q['fp']))   $out['streamSettings']['tlsSettings']['fingerprint'] = $q['fp'];
  }

  if ($network === 'grpc') {
    $svc = $q['serviceName'] ?? ($q['service'] ?? '');
    $out['streamSettings']['grpcSettings'] = ['serviceName'=>$svc,'multiMode'=>true,'idle_timeout'=>300];
    if ($security === 'reality') {
      $re = ['show'=>false];
      if (!empty($q['sni'])) $re['serverName'] = $q['sni'];
      if (!empty($q['pbk'])) $re['publicKey']  = $q['pbk'];
      if (!empty($q['sid'])) $re['shortId']    = $q['sid'];
      if (!empty($q['fp']))  $re['fingerprint']= $q['fp'];
      if (!empty($q['spx'])) $re['spiderX']    = $q['spx'];
      $out['streamSettings']['realitySettings'] = $re;
    } elseif ($security === 'tls') {
      $out['streamSettings']['tlsSettings'] = [
        'allowInsecure'=>false, 'fingerprint'=>$q['fp'] ?? 'chrome', 'serverName'=>$q['sni'] ?? $host
      ];
    } else {
      $out['streamSettings']['security'] = 'none';
    }
  }

  if ($network === 'tcp') {
    if (!empty($q['headerType']) && $q['headerType'] === 'http') {
      $out['streamSettings']['tcpSettings'] = [
        'header'=>[
          'type'=>'http',
          'request'=>[
            'version'=>'1.1','method'=>'HEAD',
            'path'=>['/'],
            'headers'=>[
              'Host'=>[$q['host'] ?? 'example.com'],
              'Connection'=>['keep-alive'],
              'Accept-Encoding'=>['gzip, deflate, br'],
              'User-Agent'=>['Mozilla/5.0 (Linux; Android 14; Pixel 8 Pro) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/129.0.0.0 Mobile Safari/537.36']
            ]
          ]
        ]
      ];
    } else {
      $out['streamSettings']['tcpSettings'] = [ 'header'=>[ 'type'=>'none' ] ];
    }
    if ($security === 'reality') {
      $re = ['show'=>false];
      if (!empty($q['sni'])) $re['serverName'] = $q['sni'];
      if (!empty($q['pbk'])) $re['publicKey']  = $q['pbk'];
      if (!empty($q['sid'])) $re['shortId']    = $q['sid'];
      if (!empty($q['fp']))  $re['fingerprint']= $q['fp'];
      if (!empty($q['spx'])) $re['spiderX']    = $q['spx'];
      $out['streamSettings']['realitySettings'] = $re;
    } elseif ($security === 'tls') {
      $out['streamSettings']['tlsSettings'] = [
        'allowInsecure'=>false, 'fingerprint'=>$q['fp'] ?? 'chrome', 'serverName'=>$q['sni'] ?? $host
      ];
    } else {
      $out['streamSettings']['security'] = 'none';
    }
  }

  return $out;
}

function parse_vmess(string $uri): ?array {
  $b64 = substr($uri, 8);
  $json = base64_decode(strtr($b64, '-_', '+/'), true);
  if ($json === false) return null;
  $o = json_decode($json, true);
  if (!is_array($o) || empty($o['add']) || empty($o['id']) || empty($o['port'])) return null;

  $tag = 'vmess_' . substr($o['id'], 0, 8) . '_' . $o['add'];
  $out = [
    'tag'=>$tag,'protocol'=>'vmess',
    'settings'=>['vnext'=>[[ 'address'=>$o['add'], 'port'=>intval($o['port']),
      'users'=>[[ 'id'=>$o['id'], 'alterId'=>intval($o['aid'] ?? 0), 'security'=>'auto' ]]
    ]]],
    'streamSettings'=>['network'=>strtolower($o['net'] ?? 'tcp'),'security'=>strtolower($o['tls'] ?? 'none')],
    'mux'=>['enabled'=>false],
  ];
  if ($out['streamSettings']['network'] === 'ws') {
    $out['streamSettings']['wsSettings'] = [
      'path' => $o['path'] ?? '/',
      'headers' => !empty($o['host']) ? ['Host'=>$o['host']] : new stdClass()
    ];
  } elseif ($out['streamSettings']['network'] === 'grpc') {
    $out['streamSettings']['grpcSettings'] = ['serviceName'=>$o['path'] ?? '','multiMode'=>true,'idle_timeout'=>300];
  }
  return $out;
}

/** ---------- sanitize/normalize (ONLY BASE SOCKOPT) ---------- */
function remove_nulls(&$arr) {
  if (!is_array($arr)) return;
  foreach ($arr as $k => &$v) {
    if ($v === null) { unset($arr[$k]); continue; }
    if (is_array($v)) remove_nulls($v);
  }
}
function normalize_outbound(array $ob): array {
  if (!isset($ob['settings']) || !is_array($ob['settings'])) $ob['settings'] = new stdClass();
  if (!isset($ob['streamSettings']) || !is_array($ob['streamSettings'])) $ob['streamSettings'] = [];

  // فقط پایه‌ها
  if (isset($ob['streamSettings']['security']) && $ob['streamSettings']['security'] === '') {
    $ob['streamSettings']['security'] = 'none';
  }
  if (!isset($ob['streamSettings']['sockopt']) || !is_array($ob['streamSettings']['sockopt'])) {
    $ob['streamSettings']['sockopt'] = [];
  }
  $ob['streamSettings']['sockopt'] = array_merge([
    'tcpNoDelay' => true,
    'tcpKeepAliveIdle' => 60,
    'mark' => 255,
  ], $ob['streamSettings']['sockopt']);

  if (is_array($ob['settings']) && array_keys($ob['settings']) === range(0, count($ob['settings'])-1)) {
    $ob['settings'] = new stdClass();
  }

  remove_nulls($ob);
  return $ob;
}

/** ---------- JSON extraction ---------- */
function extract_outbounds_from_json(string $json): array {
  $data = json_decode($json, true);
  if (!is_array($data)) return [];
  if (isset($data['outbounds']) && is_array($data['outbounds'])) {
    $outs = [];
    foreach ($data['outbounds'] as $ob) {
      if (!is_array($ob) || empty($ob['protocol'])) continue;
      $outs[] = normalize_outbound($ob);
    }
    return $outs;
  }
  if (isset($data['protocol'])) return [ normalize_outbound($data) ];
  return [];
}

/** smart JSON/URI extraction */
function smart_extract_outbounds(string $raw): array {
  $t = trim($raw);
  $outs = extract_outbounds_from_json($t);
  if ($outs) return $outs;

  $first = strpos($t, '{'); $last = strrpos($t, '}');
  if ($first !== false && $last !== false && $last > $first) {
    $chunk = substr($t, $first, $last - $first + 1);
    $outs = extract_outbounds_from_json($chunk);
    if ($outs) return $outs;
  }
  $multi = [];
  if (preg_match_all('/\{(?:[^{}]|(?R))*\}/s', $t, $m)) {
    foreach ($m[0] as $blk) {
      $o = extract_outbounds_from_json($blk);
      if ($o) $multi = array_merge($multi, $o);
    }
    if ($multi) return $multi;
  }
  $col = [];
  foreach (preg_split('/\r?\n+/', $t) as $line) {
    $line = trim($line);
    if ($line === '') continue;
    if (is_vless($line)) { $ob = parse_vless($line); if ($ob) $col[] = normalize_outbound($ob); continue; }
    if (is_vmess($line)) { $ob = parse_vmess($line); if ($ob) $col[] = normalize_outbound($ob); continue; }
    if (($line[0] ?? '') === '{' || ($line[0] ?? '') === '[') {
      $o = extract_outbounds_from_json($line);
      if ($o) $col = array_merge($col, $o);
    }
  }
  return $col;
}

function unique_tag(string $base, array &$seen): string {
  $i = 1; do { $tag = $base.$i++; } while (isset($seen[$tag]));
  $seen[$tag] = true; return $tag;
}

/** ---------- domain helpers ---------- */
function to_domain_pattern(string $host): string {
  $host = trim($host);
  if ($host === '') return '';
  $host = preg_replace('~^https?://~i', '', $host);
  $host = explode('/', $host, 2)[0];
  if (strpos($host, '*.') === 0) $host = substr($host, 2);
  return 'domain:' . $host;
}
function expand_domain_list(array $list): array {
  $o = [];
  foreach ($list as $d) { $p = to_domain_pattern($d); if ($p !== '') $o[] = $p; }
  return array_values(array_unique($o));
}

/** ---------- allowlists ---------- */
$ALLOW_SOCIAL = expand_domain_list([
  'instagram.com','cdninstagram.com','fbcdn.net','fna.fbcdn.net','facebook.com',
  'whatsapp.com','whatsapp.net','telegram.org','t.me',
  'youtube.com','youtu.be','googlevideo.com','ytimg.com','i.ytimg.com',
]);
$ALLOW_BROWSERS = expand_domain_list([
  'google.com','gstatic.com','fonts.googleapis.com','fonts.gstatic.com',
  'mozilla.org','firefox.com','addons.mozilla.org','cloudflare.com','akamaihd.net','akamai.net'
]);
$ALLOW_GAMES = expand_domain_list([
  'pubgmobile.com','igamecj.com','tencentgames.com','gcloud.qq.com',
  'supercell.com','clashofclans.com','clashroyale.com','brawlstars.com',
  'callofduty.com','activision.com'
]);
$ALLOW_GOOGLE = expand_domain_list([
  'googleapis.com','www.googleapis.com','accounts.google.com','oauth2.googleapis.com',
  'play.googleapis.com','gstatic.com','googlesyndication.com','googleadservices.com',
  'doubleclick.net','google-analytics.com','googletagmanager.com','googletagservices.com',
  'admob.com','firebaseinstallations.googleapis.com','app-measurement.com',
  'firebaselogging-pa.googleapis.com','firebasecrashlytics.googleapis.com',
  'consent.google.com','adsettings.google.com','pagead2.googlesyndication.com',
  'tpc.googlesyndication.com','securepubads.g.doubleclick.net','ad.doubleclick.net',
  'bid.g.doubleclick.net','googleads.g.doubleclick.net'
]);
$ALLOW_CUSTOM = expand_domain_list([
  'remoteconfig.org','panalapps.com','fordpanal.com','ipaddress.my','*.ipaddress.my'
]);
$TELEGRAM_IPS = [ '91.108.0.0/16','149.154.160.0/20' ];

/** ---------- template (NO fragment/noises outbounds) ---------- */
function build_template(): array {
  $levels = new stdClass();
  $levels->{'0'} = ["handshake"=>12,"connIdle"=>3600,"uplinkOnly"=>0,"downlinkOnly"=>0];

  return [
    "log" => ["loglevel"=>"warning","dnsLog"=>false],
    "stats" => new stdClass(),
    "api" => ["tag"=>"api","services"=>["StatsService"]],
    "policy" => ["system"=>["statsInboundUplink"=>true,"statsInboundDownlink"=>true],"levels"=>$levels],
    "dns" => [
      "queryStrategy"=>"UseIPv4",
      "hosts"=>["domain:googleapis.cn"=>"googleapis.com"],
      "servers"=>[
        ["address"=>"https://cloudflare-dns.com/dns-query","outboundTag"=>"dns-out"],
        ["address"=>"https://dns.google/dns-query","outboundTag"=>"dns-out"],
        ["address"=>"https://security.cloudflare-dns.com/dns-query","outboundTag"=>"dns-out"]
      ]
    ],
    "observatory" => [
      "subjectSelector"=>["auto-group"],
      "probeURL"=>"http://cp.cloudflare.com/generate_204",
      "probeInterval"=>"5s","probeTimeout"=>"3s","maxRetries"=>1
    ],
    "inbounds" => [
      ["tag"=>"socks","listen"=>"127.0.0.1","port"=>10808,"protocol"=>"socks",
       "settings"=>["auth"=>"noauth","udp"=>true],
       "sniffing"=>["enabled"=>true,"destOverride"=>["http","tls","quic"],"routeOnly"=>false]],
      ["tag"=>"http","listen"=>"127.0.0.1","port"=>10809,"protocol"=>"http",
       "settings"=>["allowTransparent"=>false],
       "sniffing"=>["enabled"=>true,"destOverride"=>["http","tls"],"routeOnly"=>false]],
      ["tag"=>"dns-in","listen"=>"127.0.0.1","port"=>10853,"protocol"=>"dokodemo-door",
       "settings"=>["address"=>"1.1.1.1","port"=>53,"network"=>"tcp,udp"]]
    ],
    "outbounds" => [
      ["protocol"=>"freedom","settings"=>["domainStrategy"=>"AsIs"],"tag"=>"direct"],
      ["protocol"=>"blackhole","settings"=>["response"=>["type"=>"http"]],"tag"=>"block"],
      ["tag"=>"dns-out","protocol"=>"dns","settings"=>new stdClass()]
    ],
    "routing" => [
      "domainStrategy"=>"IPIfNonMatch",
      "balancers"=>[["tag"=>"auto-group","selector"=>[]]],
      "rules"=>[
        // *** BLOCK ALL IPv6 ***
        ["type"=>"field","ip"=>["::/0"],"outboundTag"=>"block"],

        // DNS inside tunnel
        ["type"=>"field","inboundTag"=>["dns-in"],"outboundTag"=>"dns-out"],

        // STUN/RTC via tunnel
        ["type"=>"field","network"=>"udp","port"=>"3478-3481","balancerTag"=>"auto-group"],

        // UDP/QUIC domain exceptions BEFORE udp:443 block
        ["type"=>"field","network"=>"udp","domain"=>array_merge(
          $GLOBALS['ALLOW_SOCIAL'],$GLOBALS['ALLOW_GOOGLE'],$GLOBALS['ALLOW_CUSTOM'],
          $GLOBALS['ALLOW_BROWSERS'],$GLOBALS['ALLOW_GAMES']
        ),"balancerTag"=>"auto-group"],
        ["type"=>"field","network"=>"udp","ip"=>$GLOBALS['TELEGRAM_IPS'],"balancerTag"=>"auto-group"],

        // prevent QUIC outside tunnel
        ["type"=>"field","network"=>"udp","port"=>"443","outboundTag"=>"blackhole"],

        // private LAN direct (IPv4)
        ["type"=>"field","ip"=>["geoip:private"],"outboundTag"=>"direct"],

        // explicit popular domains via tunnel
        ["type"=>"field","domain"=>array_merge(
          $GLOBALS['ALLOW_SOCIAL'],$GLOBALS['ALLOW_GOOGLE'],$GLOBALS['ALLOW_CUSTOM'],
          $GLOBALS['ALLOW_BROWSERS'],$GLOBALS['ALLOW_GAMES']
        ),"balancerTag"=>"auto-group"],

        // default: everything via tunnel
        ["type"=>"field","network"=>"tcp","balancerTag"=>"auto-group"],
        ["type"=>"field","network"=>"udp","balancerTag"=>"auto-group"]
      ]
    ],
    "telemetry"=>["enable"=>false]
  ];
}

/** ---------- health-check (strict) ---------- */
function get_check_urls(array $ob): array {
  $urls = [];
  if (!empty($ob['settings']['checkUrls']) && is_array($ob['settings']['checkUrls'])) {
    foreach ($ob['settings']['checkUrls'] as $u) {
      $u = trim($u); if ($u === '') continue;
      if (!preg_match('~^https?://~i', $u)) $u = 'https://' . $u;
      $urls[] = $u;
    }
  }
  if (empty($urls)) {
    $urls = [
      "https://www.gstatic.com/generate_204",
      "http://cp.cloudflare.com/generate_204",
      "http://detectportal.firefox.com/success.txt"
    ];
  }
  $uniq = [];
  foreach ($urls as $u) { $u = preg_replace('~\s+~', '', $u); if ($u !== '') $uniq[$u]=true; }
  return array_keys($uniq);
}
function add_buster(string $url): string {
  $sep = (strpos($url, '?') === false) ? '?' : '&';
  return $url . $sep . '_cb=' . rawurlencode((string)microtime(true) . '_' . mt_rand(1000,9999));
}
function try_fetch_url_nocache(string $url, int $timeout): bool {
  $url = add_buster($url);
  $opts = [
    'http' => [
      'method'=>'GET','timeout'=>$timeout,'ignore_errors'=>true,
      'header'=>"Cache-Control: no-cache, no-store, must-revalidate\r\nPragma: no-cache\r\nUser-Agent: Mozilla/5.0\r\nConnection: close\r\n"
    ],
    'ssl' => ['verify_peer'=>false,'verify_peer_name'=>false]
  ];
  $ctx = stream_context_create($opts);
  $body = @file_get_contents($url, false, $ctx);
  $ok = false;
  if (!empty($http_response_header) && is_array($http_response_header)) {
    foreach ($http_response_header as $hdr) {
      if (preg_match('~HTTP/\d\.\d\s+([0-9]{3})~i', $hdr, $m)) {
        $code = intval($m[1]); if ($code===200 || $code===204) $ok = true; break;
      }
    }
  } else { if ($body !== false) $ok = true; }
  return $ok;
}

/* ---- IPv4 resolver + TCP connect probe ---- */
function resolve_ipv4(string $host): ?string {
  if (filter_var($host, FILTER_VALIDATE_IP, FILTER_FLAG_IPV4)) return $host;
  $ip = gethostbyname($host);
  return (filter_var($ip, FILTER_VALIDATE_IP, FILTER_FLAG_IPV4)) ? $ip : null;
}
function tcp_connect_ok(string $host, int $port, int $timeout): bool {
  $ip = resolve_ipv4($host);
  if ($ip === null) return false;
  $errno = 0; $err = '';
  $fp = @fsockopen($ip, $port, $errno, $err, $timeout);
  if (!$fp) return false;
  stream_set_blocking($fp, true);
  fclose($fp);
  return true;
}

/* ---- sanity of outbound ---- */
function sane_outbound(array $ob): bool {
  if (empty($ob['protocol'])) return false;
  $vn = $ob['settings']['vnext'][0] ?? null;
  if (!$vn || empty($vn['address']) || empty($vn['port'])) return false;

  $proto = strtolower($ob['protocol']);
  if ($proto === 'vless' || $proto === 'vmess') {
    $uid = $vn['users'][0]['id'] ?? '';
    if (!preg_match('/^[0-9a-f]{8}-[0-9a-f]{4}-[1-5][0-9a-f]{3}-[89ab][0-9a-f]{3}-[0-9a-f]{12}$/i', $uid)) {
      return false;
    }
  }
  $sec = strtolower($ob['streamSettings']['security'] ?? 'none');
  if ($sec === 'reality') {
    $re = $ob['streamSettings']['realitySettings'] ?? [];
    if (empty($re['serverName']) || empty($re['publicKey'])) return false;
  }
  $port = intval($vn['port']);
  return ($port >= 1 && $port <= 65535);
}

/* ---- full health ---- */
function outbound_is_alive(array $ob, int $tcpTimeout, int $urlTimeout, bool $require_all): bool {
  if (!sane_outbound($ob)) return false;
  foreach ($ob['settings']['vnext'] as $vn) {
    $addr = $vn['address'] ?? ''; $port = intval($vn['port'] ?? 0);
    if ($addr === '' || $port === 0) return false;
    if (!tcp_connect_ok($addr, $port, $tcpTimeout)) return false;
  }
  $urls = get_check_urls($ob);
  $total = count($urls); $succ = 0;
  foreach ($urls as $u) {
    if (try_fetch_url_nocache($u, $urlTimeout)) $succ++;
    if (!$require_all && $succ > intval($total/2)) return true;
  }
  return $require_all ? ($succ === $total) : ($succ >= ceil($total/2));
}

/** ---------- build & merge ---------- */
$ALLOWED_PROTOCOLS = ['vless','vmess','trojan','shadowsocks','hysteria2'];
$servers = [];
$seenTags = ['direct'=>true,'block'=>true,'dns-out'=>true];
$collectedTags = [];

shuffle($LINKS);

foreach ($LINKS as $url) {
  $attempts = 0; $ok = false;
  while ($attempts < 2 && !$ok) {
    $attempts++;
    $raw = http_get_nocache($url, $HTTP_TIMEOUT);
    if ($raw === null) continue;

    $added = smart_extract_outbounds($raw);

    if (!$added) {
      $cands = preg_split('/\r?\n+/', trim($raw));
      if (!$cands || count($cands) === 0) $cands = [trim($raw)];
      foreach ($cands as $line) {
        $line = trim($line); if ($line === '') continue;
        if (is_vless($line)) { $ob = parse_vless($line); if ($ob) $added[] = normalize_outbound($ob); }
        elseif (is_vmess($line)) { $ob = parse_vmess($line); if ($ob) $added[] = normalize_outbound($ob); }
        elseif (($line[0] ?? '') === '{' || ($line[0] ?? '') === '[') {
          $outs = extract_outbounds_from_json($line);
          if ($outs) $added = array_merge($added, $outs);
        }
      }
    }

    foreach ($added as $ob) {
      if (!isset($ob['protocol'])) continue;
      $proto = strtolower($ob['protocol']);
      if (!in_array($proto, $ALLOWED_PROTOCOLS, true)) continue;

      $ob = normalize_outbound($ob);

      // Health: sanity + TCP + URL (require all)
      $alive = outbound_is_alive($ob, 3, $HEALTH_TIMEOUT, CHECK_REQUIRE_ALL);
      $ob['__health'] = ['requireAll'=>true];
      if (!$alive) { $ob['__ping'] = -1; } // در خروجی می‌ماند ولی منفی

      $tag = unique_tag('server', $seenTags);
      $ob['tag'] = $tag;
      $servers[] = $ob;
      $collectedTags[] = $tag;
      $ok = true;
    }
  }
}

if (count($servers) === 0) { http_response_code(204); exit; }

$tpl = build_template();
$tpl['outbounds'] = array_merge($tpl['outbounds'], $servers);
$tpl['routing']['balancers'][0]['selector'] = $collectedTags;

// متادیتای سراسری در JSON خروجی
$tpl['__meta'] = [
  'health'  => [ 'requireAll' => true ],
  'network' => [ 'ipv6Blocked' => true ]
];

// policy.levels must be object
if (isset($tpl['policy']['levels']) && is_array($tpl['policy']['levels'])) {
  $obj = new stdClass();
  $obj->{'0'} = $tpl['policy']['levels'][0] ?? ["handshake"=>12,"connIdle"=>3600,"uplinkOnly"=>0,"downlinkOnly"=>0];
  $tpl['policy']['levels'] = $obj;
}

// تضمین sockopt پایه روی همه outbounds
foreach ($tpl['outbounds'] as &$ob) { $ob = normalize_outbound($ob); }
unset($ob);

// Output
echo json_encode($tpl, JSON_UNESCAPED_SLASHES|JSON_UNESCAPED_UNICODE);