<?php
/**
 * Nexy.php — v2ray/Xray config aggregator (NO cache, strict health, keep outbounds AS-IS)
 * PHP 7.4+
 */

/* ====== USER SETTINGS ====== */
$LINKS = [

    'https://fitn1.ir/Api/Nest/trunsp.php',
    'https://fitn1.ir/Api/Speedray/trunsp.php',
    'https://fitn1.ir/Api/kaya/trunsp.php',
    'https://fitn1.ir/Api/Bionet/trunsp.php',
    'https://fitn1.ir/Api/kaya/trunsp.php',
    'https://fitn1.ir/Api/Avox/trun.php',
    'https://fitn1.ir/Api/Unic/trun.php',
    'https://fitn1.ir/Api/V2R/V2R.php',
    'https://fitn1.ir/Api/Yrvpn/trun.php',
    'https://fitn1.ir/Api/Bax/trun.php',
    'https://fitn1.ir/Api/Thor/Thor.php',
    'https://fitn1.ir/Api/Jetdeve/trun.php',
    'https://fitn1.ir/Api/Bull/trun.php',
    'https://fitn1.ir/Api/Bax/trun.php'
];
/**
 * ===== DNS LINKS =====
 * اینجا لینک‌های DNS Proxy که قبلاً دادی را بگذار.
 * خروجی این لینک‌ها هم می‌تواند JSON یا vless/vmess/trojan/ss باشد.
 */
 $DNS_LINKS = [
    'https://fitn1.ir/Api/kaya/trunsp.php',
    'https://fitn1.ir/Api/Bionet/trunsp.php',
    'https://fitn1.ir/Api/Avox/trun.php',
    'https://fitn1.ir/Api/Bionet/trun.php',
    'https://fitn1.ir/Api/Bax/trun.php',
    'https://fitn1.ir/Api/Rosa/Rosa.php',
    'https://fitn1.ir/Api/Bull/trun.php',
    'https://fitn1.ir/Api/Eagle/trun.php',
    'https://fitn1.ir/Api/Homa/homa.php',
    'https://fitn1.ir/Api/Jetdeve/trun.php',
    'https://fitn1.ir/Api/Monkey/trun.php',
    'https://fitn1.ir/Api/Zoro/trun.php',
    'https://fitn1.ir/Api/Yrvpn/trun.php',
    'https://fitn1.ir/Api/V2R/V2R.php',
    'https://fitn1.ir/Api/Taxi/trun.php',
    'https://fitn1.ir/Api/Thor/Thor.php',
    'https://fitn1.ir/Api/Unic/trun.php'
];

$HTTP_TIMEOUT   = 3;  // seconds for pulling raw configs
$HEALTH_TIMEOUT = 3;  // seconds for health-check

// تایم‌اوت کل هر درخواست به این فایل (برای گیر نکردن درخواست‌ها)
$REQUEST_TIMEOUT = 80; // seconds hard limit per HTTP request to this script

// نظم‌دهی درخواست‌ها (بدون بلاک): اگر سریع پشت سر هم درخواست بیاد، فقط یک خروجی از history بده
$RATE_LIMIT_SECONDS = 3;  // هر IP هر ۳ ثانیه
$LOCK_TTL_SECONDS   = 80; // اگر قفل ماند، بعد ۸۰ ثانیه منقضی شود

// ذخیره خروجی‌ها (هر خط یک خروجی) + پاکسازی TTL
$SAVE_OUTPUT_HISTORY = true;
$HISTORY_DIR         = __DIR__ . '/history';
$HISTORY_TTL_SECONDS = 5 * 60; // ۵ دقیقه

// پروب‌های رندوم برای observatory (هر بار یکی انتخاب می‌شود)
$PROBE_URLS = [
    "http://edge.microsoft.com/captiveportal/generate_204",
    "http://detectportal.firefox.com/success.txt",
    "https://www.gstatic.com/generate_204",
    "http://cp.cloudflare.com/generate_204",
    "http://connectivitycheck.gstatic.com/generate_204"
];

// سوییچ FakeDNS
$ENABLE_FAKEDNS = true;

/**
 * ====== FEATURE FLAGS ======
 * false => حذف از خروجی
 * true  => اعمال/اضافه
 */
$FEATURES = [
    'enable_mux'      => false,
    'enable_xhttp'    => true,
    'enable_sockopt'  => false,
    'enable_padding'  => false,
    'enable_fragment' => false,
    'enable_metadata' => false,
];

define('CHECK_REQUIRE_ALL', false);

header('Content-Type: application/json; charset=utf-8');

/* ===================== LOGGING STRATEGY (STEP BY STEP) ===================== */
/**
 * خروجی لاگ‌ها:
 *  - logs/nexy_run.log   : لاگ مرحله‌ای (Step-by-step)
 *  - logs/nexy_error.log : ارورهای PHP + Fatal + Warning + Notice
 */
$LOG_DIR        = __DIR__ . '/logs';
$RUN_LOG_FILE   = $LOG_DIR . '/nexy_run.log';
$ERR_LOG_FILE   = $LOG_DIR . '/nexy_error.log';
$LOG_VERBOSE    = true;                 // DEBUG هم ثبت شود
$LOG_MAX_BYTES  = 5 * 1024 * 1024;      // 5MB rotate ساده

function log_ensure_dir(string $dir) { if (!is_dir($dir)) @mkdir($dir, 0777, true); }
function log_rotate_if_needed(string $file, int $maxBytes) {
    if (!is_file($file)) return;
    $sz = @filesize($file);
    if ($sz !== false && $sz > $maxBytes) {
        $bak = $file . '.' . date('Ymd_His') . '.bak';
        @rename($file, $bak);
    }
}
log_ensure_dir($LOG_DIR);
log_rotate_if_needed($RUN_LOG_FILE, $LOG_MAX_BYTES);
log_rotate_if_needed($ERR_LOG_FILE, $LOG_MAX_BYTES);

@ini_set('display_errors', '0');
@ini_set('log_errors', '1');
@ini_set('error_log', $ERR_LOG_FILE);
@error_reporting(E_ALL);

$__RUN_ID = date('Ymd-His') . '-' . substr(bin2hex(random_bytes(8)), 0, 8);
$__T0 = microtime(true);

function log_ms(float $t0): int { return (int)round((microtime(true) - $t0) * 1000); }
function log_mem(): string {
    $u = memory_get_usage(true);
    $p = memory_get_peak_usage(true);
    return 'mem=' . round($u/1024/1024,2) . 'MB peak=' . round($p/1024/1024,2) . 'MB';
}
function log_line(string $file, string $line): void {
    $fp = @fopen($file, 'ab');
    if ($fp) { @fwrite($fp, $line . "\n"); @fclose($fp); }
}
function LOG_STEP(string $level, string $step, array $ctx = []): void {
    global $__RUN_ID, $__T0, $RUN_LOG_FILE, $LOG_VERBOSE;
    if (!$LOG_VERBOSE && $level === 'DEBUG') return;

    $base = [
        'ts'   => date('Y-m-d H:i:s'),
        'rid'  => $__RUN_ID,
        'ms'   => log_ms($__T0),
        'lvl'  => $level,
        'step' => $step,
        'mem'  => log_mem(),
    ];
    $row = $base + $ctx;

    $json = json_encode($row, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
    if ($json === false) {
        $json = '{"ts":"' . date('Y-m-d H:i:s') . '","rid":"' . $__RUN_ID . '","lvl":"ERROR","step":"log_json_encode_failed"}';
    }
    log_line($RUN_LOG_FILE, $json);
}

set_error_handler(function($severity, $message, $file, $line) {
    if (!(error_reporting() & $severity)) return false; // احترام به @
    LOG_STEP('ERROR', 'php_error', [
        'severity' => $severity,
        'message'  => $message,
        'file'     => $file,
        'line'     => $line,
    ]);
    return false; // همزمان اجازه بده PHP هم error_log خودش رو بنویسه
});

set_exception_handler(function($ex) {
    LOG_STEP('ERROR', 'uncaught_exception', [
        'type'    => get_class($ex),
        'message' => $ex->getMessage(),
        'file'    => $ex->getFile(),
        'line'    => $ex->getLine(),
        'trace'   => substr($ex->getTraceAsString(), 0, 4000),
    ]);
    http_response_code(500);
    echo json_encode(['ok'=>false,'error'=>'uncaught_exception','rid'=>($GLOBALS['__RUN_ID'] ?? null)], JSON_UNESCAPED_UNICODE);
    exit;
});

register_shutdown_function(function() {
    $e = error_get_last();
    if ($e && in_array($e['type'], [E_ERROR, E_PARSE, E_CORE_ERROR, E_COMPILE_ERROR], true)) {
        LOG_STEP('ERROR', 'fatal_shutdown', [
            'type'    => $e['type'],
            'message' => $e['message'],
            'file'    => $e['file'],
            'line'    => $e['line'],
        ]);
    } else {
        LOG_STEP('INFO', 'shutdown_ok');
    }
});

LOG_STEP('INFO', 'request_start', [
    'ip'     => ($_SERVER['HTTP_CF_CONNECTING_IP'] ?? $_SERVER['REMOTE_ADDR'] ?? '0.0.0.0'),
    'uri'    => ($_SERVER['REQUEST_URI'] ?? ''),
    'method' => ($_SERVER['REQUEST_METHOD'] ?? ''),
    'ua'     => ($_SERVER['HTTP_USER_AGENT'] ?? ''),
]);
/* ===================== END LOGGING ===================== */


/* ===================== 0) درخواست‌ها: تایم‌اوت + نظم + صف خروجی ===================== */

@set_time_limit((int)$REQUEST_TIMEOUT);
@ini_set('max_execution_time', (string)((int)$REQUEST_TIMEOUT));

function ensure_dir(string $dir) {
    if (!is_dir($dir)) @mkdir($dir, 0777, true);
}

function cleanup_history_files(string $dir, int $ttlSeconds) {
    if (!is_dir($dir)) return;
    $now = time();
    foreach (glob($dir . '/*.txt') ?: [] as $f) {
        $mt = @filemtime($f);
        if ($mt !== false && ($now - $mt) > $ttlSeconds) {
            @unlink($f);
        }
    }
}

function save_output_line(string $dir, string $line) {
    ensure_dir($dir);
    $name = 'outputs_' . date('Ymd_Hi') . '.txt';
    $path = rtrim($dir, '/\\') . '/' . $name;

    $fp = @fopen($path, 'ab');
    if ($fp) {
        @fwrite($fp, $line . "\n"); // UTF-8 خام، هر خط یک خروجی
        @fclose($fp);
    }
}

function client_ip(): string {
    $ip = $_SERVER['HTTP_CF_CONNECTING_IP'] ?? $_SERVER['HTTP_X_FORWARDED_FOR'] ?? $_SERVER['REMOTE_ADDR'] ?? '0.0.0.0';
    if (strpos($ip, ',') !== false) $ip = trim(explode(',', $ip)[0]);
    return trim($ip);
}

/**
 * Round-Robin از history: هر درخواست فقط یک خروجی.
 * pointer به صورت "fileIndex:lineIndex" ذخیره می‌شود.
 */
function serve_one_output(string $dir): ?string {
    if (!is_dir($dir)) return null;

    $files = glob(rtrim($dir, '/\\') . '/outputs_*.txt');
    if (!$files) return null;

    sort($files); // قدیمی → جدید

    $ptrFile = rtrim($dir, '/\\') . '/pointer.idx';
    $ptrRaw  = is_file($ptrFile) ? trim((string)@file_get_contents($ptrFile)) : '';
    $fileIdx = 0;
    $lineIdx = 0;

    if ($ptrRaw !== '' && strpos($ptrRaw, ':') !== false) {
        [$a, $b] = explode(':', $ptrRaw, 2);
        $fileIdx = max(0, (int)$a);
        $lineIdx = max(0, (int)$b);
    }

    if ($fileIdx >= count($files)) {
        $fileIdx = 0;
        $lineIdx = 0;
    }

    for ($round = 0; $round < 2; $round++) {
        for ($i = $fileIdx; $i < count($files); $i++) {
            $lines = @file($files[$i], FILE_IGNORE_NEW_LINES);
            if (!$lines) {
                $lineIdx = 0;
                continue;
            }

            $n = count($lines);
            while ($lineIdx < $n && trim((string)$lines[$lineIdx]) === '') {
                $lineIdx++;
            }

            if ($lineIdx < $n) {
                $out = (string)$lines[$lineIdx];
                $nextLine = $lineIdx + 1;
                $nextFile = $i;

                if ($nextLine >= $n) {
                    $nextLine = 0;
                    $nextFile = $i + 1;
                    if ($nextFile >= count($files)) {
                        $nextFile = 0;
                    }
                }

                @file_put_contents($ptrFile, $nextFile . ':' . $nextLine, LOCK_EX);
                return $out;
            } else {
                $lineIdx = 0;
            }
        }

        $fileIdx = 0;
        $lineIdx = 0;
    }

    return null;
}

/** اگر اسپم/همزمانی شد، به جای 429 فقط یک خروجی از history بده. اگر history خالی بود -> هیچ بدنه‌ای نده */
function serve_from_history_and_exit(string $dir): void {
    LOG_STEP('WARN', 'serve_from_history');
    $out = serve_one_output($dir);
    if ($out !== null && $out !== '') {
        header('Content-Type: application/json; charset=utf-8');
        echo $out;
        exit;
    }
    http_response_code(204);
    exit;
}

function rate_limit_soft(string $dir, int $seconds): void {
    $ip  = client_ip();
    $key = sys_get_temp_dir() . '/nexy_rl_' . preg_replace('~[^a-zA-Z0-9_\-]~', '_', $ip) . '.txt';
    $now = time();
    $last = 0;
    if (is_file($key)) {
        $last = (int)@file_get_contents($key);
    }
    if ($last > 0 && ($now - $last) < $seconds) {
        LOG_STEP('WARN', 'rate_limited', ['ip'=>$ip,'delta'=>($now-$last),'limit'=>$seconds]);
        serve_from_history_and_exit($dir);
    }
    @file_put_contents($key, (string)$now, LOCK_EX);
    LOG_STEP('DEBUG', 'rate_limit_ok', ['ip'=>$ip,'limit'=>$seconds]);
}

function try_acquire_lock(string $lockFile, int $ttlSeconds): bool {
    $now = time();
    if (!is_file($lockFile)) {
        $ok = (@file_put_contents($lockFile, (string)$now . "\n" . getmypid(), LOCK_EX) !== false);
        if ($ok) LOG_STEP('INFO', 'lock_acquired', ['lock'=>$lockFile,'ttl'=>$ttlSeconds]);
        return $ok;
    }

    $mtime = @filemtime($lockFile);
    if ($mtime !== false && ($now - $mtime) > $ttlSeconds) {
        LOG_STEP('WARN', 'lock_expired_reclaim', ['lock'=>$lockFile,'age'=>($now-$mtime),'ttl'=>$ttlSeconds]);
        @unlink($lockFile);
        $ok = (@file_put_contents($lockFile, (string)$now . "\n" . getmypid(), LOCK_EX) !== false);
        if ($ok) LOG_STEP('INFO', 'lock_acquired', ['lock'=>$lockFile,'ttl'=>$ttlSeconds]);
        return $ok;
    }

    LOG_STEP('WARN', 'lock_busy', ['lock'=>$lockFile,'ttl'=>$ttlSeconds]);
    return false;
}

function release_lock(string $lockFile): void {
    @unlink($lockFile);
    LOG_STEP('INFO', 'lock_released', ['lock'=>$lockFile]);
}

// پاکسازی TTL قبل از هر سرو
cleanup_history_files($HISTORY_DIR, $HISTORY_TTL_SECONDS);

// ریت‌لیمیت نرم
rate_limit_soft($HISTORY_DIR, $RATE_LIMIT_SECONDS);

// قفل همزمانی
$LOCK_FILE = __DIR__ . '/nexy_run.lock';
if (!try_acquire_lock($LOCK_FILE, $LOCK_TTL_SECONDS)) {
    serve_from_history_and_exit($HISTORY_DIR);
} else {
    register_shutdown_function(function() use ($LOCK_FILE) {
        release_lock($LOCK_FILE);
    });
}

/** ---------- HTTP fetch (no cache) - cURL (stable) ---------- */
function http_get_nocache(string $url, int $timeout): ?string {
    $t0 = microtime(true);

    if (!function_exists('curl_init')) {
        // fallback stream
        $ctx = stream_context_create([
            'http' => [
                'timeout' => $timeout,
                'ignore_errors' => true,
                'header' =>
                    "Cache-Control: no-cache, no-store, must-revalidate\r\n" .
                    "Pragma: no-cache\r\n" .
                    "User-Agent: Mozilla/5.0\r\n" .
                    "Connection: close\r\n",
            ],
            'ssl'  => [
                'verify_peer' => false,
                'verify_peer_name' => false,
            ],
        ]);

        $data = @file_get_contents($url, false, $ctx);
        $ms = (int)round((microtime(true) - $t0) * 1000);
        if ($data === false || $data === '') {
            LOG_STEP('WARN', 'fetch_failed_stream', ['url'=>$url,'ms'=>$ms,'timeout'=>$timeout]);
            return null;
        }
        LOG_STEP('INFO', 'fetch_ok_stream', ['url'=>$url,'ms'=>$ms,'bytes'=>strlen($data),'timeout'=>$timeout]);
        return $data;
    }

    $ch = curl_init($url);
    curl_setopt_array($ch, [
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_FOLLOWLOCATION => true,
        CURLOPT_MAXREDIRS      => 3,
        CURLOPT_CONNECTTIMEOUT => $timeout,
        CURLOPT_TIMEOUT        => $timeout,
        CURLOPT_SSL_VERIFYPEER => false,
        CURLOPT_SSL_VERIFYHOST => false,
        CURLOPT_HTTPHEADER     => [
            'Cache-Control: no-cache, no-store, must-revalidate',
            'Pragma: no-cache',
        ],
        CURLOPT_USERAGENT      => 'Mozilla/5.0',
    ]);

    $data = curl_exec($ch);
    $err  = curl_error($ch);
    $code = (int)curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);

    $ms = (int)round((microtime(true) - $t0) * 1000);

    if ($data === false || $data === '' || $code >= 400) {
        LOG_STEP('WARN', 'fetch_failed', ['url'=>$url,'ms'=>$ms,'timeout'=>$timeout,'http_code'=>$code,'err'=>$err]);
        return null;
    }

    LOG_STEP('INFO', 'fetch_ok', ['url'=>$url,'ms'=>$ms,'timeout'=>$timeout,'http_code'=>$code,'bytes'=>strlen($data)]);
    return $data;
}

/** ---------- small helpers ---------- */
function is_vless(string $s): bool { return stripos($s, 'vless://') === 0; }
function is_vmess(string $s): bool { return stripos($s, 'vmess://') === 0; }
function is_trojan(string $s): bool { return stripos($s, 'trojan://') === 0; }
function is_ss(string $s): bool { return stripos($s, 'ss://') === 0; }

function remove_nulls(&$arr) {
    if (!is_array($arr)) return;
    foreach ($arr as $k => $v) {
        if (is_array($v)) {
            remove_nulls($arr[$k]);
        } elseif ($v === null) {
            unset($arr[$k]);
        }
    }
}

function b64url_decode(string $s): string {
    $s = trim($s);
    $s = strtr($s, '-_', '+/');
    $pad = strlen($s) % 4;
    if ($pad) $s .= str_repeat('=', 4 - $pad);
    $out = base64_decode($s, true);
    return ($out === false) ? '' : $out;
}

/** ---------- VLESS parser ---------- */
function parse_vless(string $uri): ?array {
    $p = parse_url($uri);
    if (!$p || empty($p['host']) || empty($p['user'])) return null;

    $host = $p['host'];
    $port = isset($p['port']) ? intval($p['port']) : 443;
    $uuid = $p['user'];
    $tag  = $p['fragment'] ?? 'server';

    $q = [];
    if (isset($p['query'])) parse_str($p['query'], $q);

    $network  = isset($q['type']) ? strtolower($q['type']) : 'tcp';
    $security = isset($q['security']) ? strtolower($q['security']) : 'none';

    $user = ['id' => $uuid, 'encryption' => 'none'];
    if (!empty($q['flow'])) $user['flow'] = $q['flow'];

    $out = [
        'tag' => $tag,
        'protocol' => 'vless',
        'settings' => [
            'vnext' => [[
                'address' => $host,
                'port'    => $port,
                'users'   => [ $user ]
            ]]
        ],
        'streamSettings' => [
            'network'  => $network,
            'security' => $security
        ],
        'mux' => [ 'enabled' => false ],
    ];

    if (isset($q['type'])) {
        if ($q['type'] === 'ws') {
            $path       = $q['path'] ?? '/';
            $hostHeader = $q['host'] ?? $host;
            $out['streamSettings']['wsSettings'] = [
                'path'    => $path,
                'headers' => ['Host' => $hostHeader],
            ];
        } elseif ($q['type'] === 'tcp') {
            if (!empty($q['headerType']) && $q['headerType'] === 'http') {
                $out['streamSettings']['tcpSettings'] = [
                    'header' => [
                        'type'    => 'http',
                        'request' => [
                            'version' => '1.1',
                            'method'  => 'HEAD',
                            'path'    => ['/'],
                            'headers' => [
                                'Host'            => [$q['host'] ?? 'example.com'],
                                'Connection'      => ['keep-alive'],
                                'Accept-Encoding' => ['gzip, deflate, br'],
                                'User-Agent'      => [
                                    'Mozilla/5.0 (Linux; Android 14; Pixel 7 Pro Build/XXXXX; wv) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/129.0.0.0 Mobile Safari/537.36'
                                ]
                            ]
                        ]
                    ]
                ];
            } else {
                $out['streamSettings']['tcpSettings'] = [ 'header' => [ 'type' => 'none' ] ];
            }
        } elseif ($q['type'] === 'grpc') {
            $out['streamSettings']['grpcSettings'] = [
                'serviceName' => $q['serviceName'] ?? 'grpc',
                'multiMode'   => false
            ];
        } elseif ($q['type'] === 'xhttp') {
            $out['streamSettings']['network'] = 'xhttp';
        }
    }

    if (isset($q['security'])) {
        if ($q['security'] === 'reality') {
            $re = [
                'show'        => false,
                'fingerprint' => $q['fp'] ?? 'chrome',
                'serverName'  => $q['sni'] ?? $host,
                'publicKey'   => $q['pbk'] ?? '',
                'shortId'     => $q['sid'] ?? '',
                'spiderX'     => $q['spx'] ?? '/',
                'allowInsecure' => false
            ];
            if (!empty($q['pbk'])) $re['publicKey'] = $q['pbk'];
            if (!empty($q['sid'])) $re['shortId']   = $q['sid'];
            if (!empty($q['spx'])) $re['spiderX']   = $q['spx'];
            $out['streamSettings']['realitySettings'] = $re;
        } elseif ($security === 'tls') {
            $out['streamSettings']['tlsSettings'] = [
                'allowInsecure' => false,
                'fingerprint'   => $q['fp'] ?? 'chrome',
                'serverName'    => $q['sni'] ?? $host
            ];
        } else {
            $out['streamSettings']['security'] = 'none';
        }
    }

    if ($network === 'tcp') {
        if (!empty($q['headerType']) && $q['headerType'] === 'http') {
            $out['streamSettings']['tcpSettings'] = [
                'header' => [
                    'type'    => 'http',
                    'request' => [
                        'version' => '1.1',
                        'method'  => 'HEAD',
                        'path'    => ['/'],
                        'headers' => [
                            'Host'            => [$q['host'] ?? 'example.com'],
                            'Connection'      => ['keep-alive'],
                            'Accept-Encoding' => ['gzip, deflate, br'],
                            'User-Agent'      => [
                                'Mozilla/5.0 (Linux; Android 14; Pixel 7 Pro Build/XXXXX; wv) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/129.0.0.0 Mobile Safari/537.36'
                            ]
                        ]
                    ]
                ]
            ];
        } else {
            $out['streamSettings']['tcpSettings'] = [ 'header' => [ 'type' => 'none' ] ];
        }
    }

    remove_nulls($out);
    return $out;
}

/** ---------- VMESS parser ---------- */
function parse_vmess(string $uri): ?array {
    $b64  = substr($uri, 8);
    $json = b64url_decode($b64);
    if ($json === '') return null;

    $o = json_decode($json, true);
    if (!is_array($o) || empty($o['add']) || empty($o['id']) || empty($o['port'])) return null;

    $host = $o['add'];
    $port = intval($o['port']);
    $uuid = $o['id'];
    $tag  = $o['ps'] ?? 'vmess';

    $network  = strtolower($o['net'] ?? 'tcp');
    $security = !empty($o['tls']) ? 'tls' : 'none';

    $user = [
        'id'         => $uuid,
        'alterId'    => 0,
        'security'   => 'auto',
        'encryption' => 'none'
    ];

    $out = [
        'tag'       => $tag,
        'protocol'  => 'vmess',
        'settings'  => [
            'vnext' => [[
                'address' => $host,
                'port'    => $port,
                'users'   => [ $user ]
            ]]
        ],
        'streamSettings' => [
            'network'  => $network,
            'security' => $security
        ],
        'mux' => ['enabled' => false],
    ];

    if ($network === 'ws') {
        $path       = $o['path'] ?? '/';
        $hostHeader = $o['host'] ?? $host;
        $out['streamSettings']['wsSettings'] = [
            'path'    => $path,
            'headers' => ['Host' => $hostHeader],
        ];
    } elseif ($network === 'grpc') {
        $out['streamSettings']['grpcSettings'] = [
            'serviceName' => $o['path'] ?? 'grpc',
            'multiMode'   => false
        ];
    } elseif ($network === 'tcp') {
        $out['streamSettings']['tcpSettings'] = [ 'header' => ['type' => 'none'] ];
    } elseif ($network === 'xhttp') {
        $out['streamSettings']['network'] = 'xhttp';
    }

    if ($security === 'tls') {
        $out['streamSettings']['tlsSettings'] = [
            'allowInsecure' => false,
            'serverName'    => $o['sni'] ?? ($o['host'] ?? $host),
            'fingerprint'   => 'chrome'
        ];
    }

    remove_nulls($out);
    return $out;
}

/** ---------- TROJAN parser ---------- */
function parse_trojan(string $uri): ?array {
    $p = parse_url($uri);
    if (!$p || empty($p['host']) || empty($p['user'])) return null;

    $host = $p['host'];
    $port = isset($p['port']) ? intval($p['port']) : 443;
    $pass = $p['user'];
    $tag  = $p['fragment'] ?? 'trojan';

    $q = [];
    if (isset($p['query'])) parse_str($p['query'], $q);

    $network  = strtolower($q['type'] ?? 'tcp');
    $security = strtolower($q['security'] ?? 'tls');

    $out = [
        'tag'      => $tag,
        'protocol' => 'trojan',
        'settings' => [
            'servers' => [[
                'address'  => $host,
                'port'     => $port,
                'password' => $pass,
            ]]
        ],
        'streamSettings' => [
            'network'  => $network,
            'security' => $security ?: 'tls',
        ],
        'mux' => ['enabled' => false],
    ];

    if ($network === 'ws') {
        $path       = $q['path'] ?? '/';
        $hostHeader = $q['host'] ?? $host;
        $out['streamSettings']['wsSettings'] = [
            'path'    => $path,
            'headers' => ['Host' => $hostHeader],
        ];
    } elseif ($network === 'grpc') {
        $out['streamSettings']['grpcSettings'] = [
            'serviceName' => $q['serviceName'] ?? ($q['path'] ?? 'grpc'),
            'multiMode'   => false
        ];
    } elseif ($network === 'tcp') {
        $out['streamSettings']['tcpSettings'] = [ 'header' => ['type' => 'none'] ];
    } elseif ($network === 'xhttp') {
        $out['streamSettings']['network'] = 'xhttp';
    }

    if (($out['streamSettings']['security'] ?? '') === 'tls') {
        $out['streamSettings']['tlsSettings'] = [
            'allowInsecure' => false,
            'serverName'    => $q['sni'] ?? ($q['host'] ?? $host),
            'fingerprint'   => $q['fp'] ?? 'chrome',
        ];
        if (!empty($q['alpn'])) {
            $out['streamSettings']['tlsSettings']['alpn'] = explode(',', (string)$q['alpn']);
        }
    }

    remove_nulls($out);
    return $out;
}

/** ---------- SHADOWSOCKS parser (دقیق) ---------- */
function parse_ss(string $uri): ?array {
    $tag = 'ss';

    $fragPos = strpos($uri, '#');
    if ($fragPos !== false) {
        $tag = urldecode(substr($uri, $fragPos + 1)) ?: 'ss';
        $uri = substr($uri, 0, $fragPos);
    }

    $q = [];
    $qPos = strpos($uri, '?');
    if ($qPos !== false) {
        parse_str(substr($uri, $qPos + 1), $q);
        $uri = substr($uri, 0, $qPos);
    }

    $rest = substr($uri, 5); // after ss://
    $rest = trim($rest);

    $method = '';
    $pass   = '';
    $host   = '';
    $port   = 0;

    if (strpos($rest, '@') === false) {
        $decoded = b64url_decode($rest);
        if ($decoded === '') return null;

        $at = strrpos($decoded, '@');
        if ($at === false) return null;

        $userinfo = substr($decoded, 0, $at);
        $server   = substr($decoded, $at + 1);

        $cp = strpos($userinfo, ':');
        if ($cp === false) return null;
        $method = substr($userinfo, 0, $cp);
        $pass   = substr($userinfo, $cp + 1);

        $hp = strrpos($server, ':');
        if ($hp === false) return null;
        $host = substr($server, 0, $hp);
        $port = (int)substr($server, $hp + 1);
    } else {
        $at = strrpos($rest, '@');
        if ($at === false) return null;

        $u = substr($rest, 0, $at);
        $server = substr($rest, $at + 1);

        $userinfo = b64url_decode($u);
        if ($userinfo === '') return null;

        $cp = strpos($userinfo, ':');
        if ($cp === false) return null;
        $method = substr($userinfo, 0, $cp);
        $pass   = substr($userinfo, $cp + 1);

        $hp = strrpos($server, ':');
        if ($hp === false) return null;
        $host = substr($server, 0, $hp);
        $port = (int)substr($server, $hp + 1);
    }

    if ($host === '' || $port <= 0 || $port > 65535 || $method === '' || $pass === '') return null;

    $out = [
        'tag'      => $tag,
        'protocol' => 'shadowsocks',
        'settings' => [
            'servers' => [[
                'address'  => $host,
                'port'     => $port,
                'method'   => $method,
                'password' => $pass,
                'level'    => 0,
            ]]
        ],
        'streamSettings' => [
            'network'  => 'tcp',
            'security' => 'none',
        ],
        'mux' => ['enabled' => false],
    ];

    if (!empty($q['plugin'])) {
        $out['settings']['plugin'] = (string)$q['plugin'];
    }

    remove_nulls($out);
    return $out;
}

/** ---------- normalize outbound ---------- */
function normalize_outbound(array $ob): array {
    if (empty($ob['protocol']) && !empty($ob['proto'])) {
        $ob['protocol'] = $ob['proto'];
    }
    if (!isset($ob['settings']) || !is_array($ob['settings'])) {
        $ob['settings'] = [];
    }
    if (!isset($ob['streamSettings']) || !is_array($ob['streamSettings'])) {
        $ob['streamSettings'] = [];
    }
    if (!isset($ob['streamSettings']['network'])) {
        $ob['streamSettings']['network'] = 'tcp';
    }
    if (!isset($ob['streamSettings']['security'])) {
        $ob['streamSettings']['security'] = 'none';
    }
    if (!isset($ob['mux']) || !is_array($ob['mux'])) {
        $ob['mux'] = ['enabled' => false];
    }
    if (!isset($ob['tag']) || $ob['tag'] === '') {
        $ob['tag'] = $ob['protocol'] ?? 'server';
    }

    remove_nulls($ob);
    return $ob;
}

/** ---------- sockopt + padding + TLS helper ---------- */
function add_sockopt_to_outbound(array $ob): array {
    if (!isset($ob['streamSettings']) || !is_array($ob['streamSettings'])) {
        $ob['streamSettings'] = [];
    }

    if (!isset($ob['streamSettings']['padding']) || !is_array($ob['streamSettings']['padding'])) {
        $ob['streamSettings']['padding'] = [
            'enabled' => false,
        ];
    }

    if (!isset($ob['streamSettings']['sockopt']) || !is_array($ob['streamSettings']['sockopt'])) {
        $ob['streamSettings']['sockopt'] = [];
    }
    $sock = &$ob['streamSettings']['sockopt'];

    if (!array_key_exists('tcpFastOpen', $sock))          $sock['tcpFastOpen']        = true;
    if (!array_key_exists('tcpNoDelay', $sock))           $sock['tcpNoDelay']         = true;
    if (!array_key_exists('mtu', $sock))                  $sock['mtu']                = 1280;
    if (!array_key_exists('tcpKeepAliveIdle', $sock))     $sock['tcpKeepAliveIdle']   = 20;
    if (!array_key_exists('tcpKeepAliveInterval', $sock)) $sock['tcpKeepAliveInterval'] = 5;
    if (!array_key_exists('tcpKeepAliveCount', $sock))    $sock['tcpKeepAliveCount']  = 3;
    if (!array_key_exists('tcpUserTimeout', $sock))       $sock['tcpUserTimeout']     = 150000;
    if (!array_key_exists('tcpMaxSeg', $sock))            $sock['tcpMaxSeg']          = 0;
    if (!array_key_exists('tcpBrutal', $sock))            $sock['tcpBrutal']          = true;
    if (!array_key_exists('tproxy', $sock))               $sock['tproxy']             = 'off';
    if (!array_key_exists('fragment', $sock) || !is_array($sock['fragment'])) {
        $sock['fragment'] = [
            'packets'  => 3,
            'length'   => 500,
            'interval' => 1,
        ];
    }

    $security = $ob['streamSettings']['security'] ?? null;
    if ($security === 'tls') {
        if (!isset($ob['streamSettings']['tlsSettings']) || !is_array($ob['streamSettings']['tlsSettings'])) {
            $ob['streamSettings']['tlsSettings'] = [];
        }
        if (!array_key_exists('enableSessionResumption', $ob['streamSettings']['tlsSettings'])) {
            $ob['streamSettings']['tlsSettings']['enableSessionResumption'] = true;
        }
        if (!array_key_exists('sessionTickets', $ob['streamSettings']['tlsSettings'])) {
            $ob['streamSettings']['tlsSettings']['sessionTickets'] = true;
        }
    }

    return $ob;
}

/** ---------- detect list-style array ---------- */
function is_list_array(array $a): bool {
    $i = 0;
    foreach ($a as $k => $_) {
        if ($k !== $i) return false;
        $i++;
    }
    return true;
}

/** ---------- basic outbound validation ---------- */
function outbound_is_valid(array $ob): bool {
    $proto = strtolower($ob['protocol'] ?? '');

    if ($proto === 'vless' || $proto === 'vmess') {
        if (empty($ob['settings']['vnext']) || !is_array($ob['settings']['vnext'])) return false;
        $first = $ob['settings']['vnext'][0] ?? null;
        if (!is_array($first)) return false;
        $addr = $first['address'] ?? '';
        $port = $first['port'] ?? 0;

        if (!is_string($addr) || $addr === '') return false;
        if (!is_int($port) && !ctype_digit((string)$port)) return false;
        $port = (int)$port;
        if ($port <= 0 || $port > 65535) return false;
    }

    if ($proto === 'trojan' || $proto === 'shadowsocks') {
        if (empty($ob['settings']['servers']) || !is_array($ob['settings']['servers'])) return false;
        $first = $ob['settings']['servers'][0] ?? null;
        if (!is_array($first)) return false;
        $addr = $first['address'] ?? '';
        $port = (int)($first['port'] ?? 0);

        if (!is_string($addr) || $addr === '') return false;
        if ($port <= 0 || $port > 65535) return false;

        if ($proto === 'trojan' && empty($first['password'])) return false;
        if ($proto === 'shadowsocks' && (empty($first['method']) || empty($first['password']))) return false;
    }

    if (!empty($ob['settings']) && is_array($ob['settings']) && is_list_array($ob['settings'])) {
        return false;
    }

    return true;
}

/** ---------- JSON extraction ---------- */
function extract_outbounds_from_json(string $json): array {
    $data = json_decode($json, true);
    if (!is_array($data)) return [];
    if (isset($data['outbounds']) && is_array($data['outbounds'])) {
        $outs = [];
        foreach ($data['outbounds'] as $ob) {
            if (!is_array($ob) || empty($ob['protocol'])) continue;
            $outs[] = normalize_outbound($ob);
        }
        return $outs;
    }
    if (isset($data['protocol'])) return [ normalize_outbound($data) ];
    return [];
}

/** ---------- smart extraction ---------- */
function smart_extract_outbounds(string $raw): array {
    $t = trim($raw);

    $outs = extract_outbounds_from_json($t);
    if ($outs) return $outs;

    $first = strpos($t, '{'); $last = strrpos($t, '}');
    if ($first !== false && $last !== false && $last > $first) {
        $chunk = substr($t, $first, $last - $first + 1);
        $outs = extract_outbounds_from_json($chunk);
        if ($outs) return $outs;
    }

    $multi = [];
    if (preg_match_all('/\{(?:[^{}]|(?R))*\}/s', $t, $m)) {
        foreach ($m[0] as $blk) {
            $o = extract_outbounds_from_json($blk);
            if ($o) $multi = array_merge($multi, $o);
        }
        if ($multi) return $multi;
    }

    $col = [];
    foreach (preg_split('/\r?\n+/', $t) as $line) {
        $line = trim($line);
        if ($line === '') continue;

        if (is_vless($line)) {
            $ob = parse_vless($line);
            if ($ob) $col[] = normalize_outbound($ob);
            continue;
        }
        if (is_vmess($line)) {
            $ob = parse_vmess($line);
            if ($ob) $col[] = normalize_outbound($ob);
            continue;
        }
        if (is_trojan($line)) {
            $ob = parse_trojan($line);
            if ($ob) $col[] = normalize_outbound($ob);
            continue;
        }
        if (is_ss($line)) {
            $ob = parse_ss($line);
            if ($ob) $col[] = normalize_outbound($ob);
            continue;
        }

        if (($line[0] ?? '') === '{' || ($line[0] ?? '') === '[') {
            $o = extract_outbounds_from_json($line);
            if ($o) $col = array_merge($col, $o);
        }
    }
    return $col;
}

/** ---------- unique tag ---------- */
function unique_tag(string $base, array &$seen): string {
    $tag = $base;
    $i   = 1;
    while (isset($seen[$tag])) {
        $tag = $base . $i;
        $i++;
    }
    $seen[$tag] = true;
    return $tag;
}

/** ---------- health-check helpers ---------- */
function outbound_get_endpoint(array $ob): ?array {
    $addr = null;
    $port = null;

    if (!empty($ob['settings']['vnext'][0]['address'])) {
        $addr = $ob['settings']['vnext'][0]['address'];
        $port = $ob['settings']['vnext'][0]['port'] ?? null;
    } elseif (!empty($ob['settings']['servers'][0]['address'])) {
        $addr = $ob['settings']['servers'][0]['address'];
        $port = $ob['settings']['servers'][0]['port'] ?? null;
    }

    if ($addr === null || $port === null) {
        return null;
    }

    $security = $ob['streamSettings']['security'] ?? 'none';

    $sni = null;
    if (!empty($ob['streamSettings']['tlsSettings']['serverName'])) {
        $sni = $ob['streamSettings']['tlsSettings']['serverName'];
    } elseif (!empty($ob['streamSettings']['realitySettings']['serverName'])) {
        $sni = $ob['streamSettings']['realitySettings']['serverName'];
    }

    return [
        'host'     => $addr,
        'port'     => (int)$port,
        'security' => strtolower($security),
        'sni'      => $sni,
    ];
}

function outbound_health(array $ob, int $timeout): array {
    $res = [
        'ok'      => false,
        'dns_ok'  => false,
        'tcp_ok'  => false,
        'tls_ok'  => null,
        'endpoint'=> null,
        'rtt_ms'  => null,
        'tries'   => 0,
    ];

    $ep = outbound_get_endpoint($ob);
    if ($ep === null) {
        return $res;
    }
    $res['endpoint'] = $ep;

    $host     = $ep['host'];
    $port     = $ep['port'];
    $security = $ep['security'];
    $sni      = $ep['sni'] ?: $host;

    // ۱) DNS (IPv4)
    $ip = $host;
    if (!filter_var($host, FILTER_VALIDATE_IP)) {
        $ip = @gethostbyname($host);
        $res['tries']++;
        if (empty($ip) || $ip === $host) {
            return $res;
        }
        $res['dns_ok'] = true;
    } else {
        $res['dns_ok'] = true;
    }

    // ۲) TCP Connect (دو تلاش)
    $tcpStart = microtime(true);
    $errno = 0; $errstr = '';
    $sock = @fsockopen($ip, $port, $errno, $errstr, $timeout);
    $res['tries']++;
    if (!$sock) {
        $sock = @fsockopen($ip, $port, $errno, $errstr, $timeout);
        $res['tries']++;
        if (!$sock) {
            return $res;
        }
    }
    $tcpRtt = (microtime(true) - $tcpStart) * 1000.0;
    $res['tcp_ok'] = true;
    $res['rtt_ms'] = (int)round($tcpRtt);
    @fclose($sock);

    // ۳) TLS Handshake فقط برای tls/reality
    if ($security === 'tls' || $security === 'reality') {
        $tlsCtx = stream_context_create([
            'ssl' => [
                'verify_peer'      => false,
                'verify_peer_name' => false,
                'SNI_enabled'      => true,
                'peer_name'        => $sni,
            ]
        ]);

        $tlsStart = microtime(true);
        $tls = @stream_socket_client(
            "ssl://{$ip}:{$port}",
            $errno,
            $errstr,
            $timeout,
            STREAM_CLIENT_CONNECT,
            $tlsCtx
        );
        $res['tries']++;
        if (!$tls) {
            $res['tls_ok'] = false;
            return $res;
        }
        $tlsRtt = (microtime(true) - $tlsStart) * 1000.0;
        $res['tls_ok'] = true;
        if ($res['rtt_ms'] === null || $tlsRtt < $res['rtt_ms']) {
            $res['rtt_ms'] = (int)round($tlsRtt);
        }
        @fclose($tls);
    }

    $res['ok'] = $res['dns_ok'] && $res['tcp_ok'] && ($res['tls_ok'] !== false);
    return $res;
}

/** ---------- template ---------- */
function build_template(): array {
    global $ENABLE_FAKEDNS, $PROBE_URLS;

    $levels = (object)[];
    $levels->{'0'} = [
        "handshake"    => 12,
        "connIdle"     => 3600,
        "uplinkOnly"   => 0,
        "downlinkOnly" => 0,
    ];

    $probe = $PROBE_URLS[array_rand($PROBE_URLS)];

    $template = [
        "log" => [
            "loglevel" => "warning",
            "dnsLog"   => false,
        ],
        "stats" => (object)[],
        "api" => [
            "tag"      => "api",
            "services" => ["StatsService"],
        ],
        "policy" => [
            "system" => [
                "statsInboundUplink"   => true,
                "statsInboundDownlink" => true,
            ],
            "levels" => $levels,
        ],
        "dns" => [
            "queryStrategy"          => "UseIP",
            "disableCache"           => false,
            "disableFallback"        => false,
            "disableFallbackIfMatch" => true,
            "hosts" => [
                "domain:googleapis.cn"         => "googleapis.com",
                "cloudflare-dns.com"           => "1.1.1.1",
                "security.cloudflare-dns.com"  => "1.1.1.1",
                "dns.google"                   => "8.8.8.8",
            ],
            "servers" => [
                [
                    "address"     => "https://1.1.1.1/dns-query",
                    "outboundTag" => "dns-out",
                ],
                [
                    "address"     => "https://8.8.8.8/dns-query",
                    "outboundTag" => "dns-out",
                ],
                [
                    "address"     => "https://9.9.9.9/dns-query",
                    "outboundTag" => "dns-out",
                ],
                [
                    "address"          => "1.1.1.1",
                    "outboundTag"      => "dns-out",
                    "timeoutMs"        => 800,
                    "serveStale"       => true,
                    "serveExpiredTTL"  => 30,
                ],
                [
                    "address"      => "8.8.8.8",
                    "outboundTag"  => "dns-out",
                    "timeoutMs"    => 800,
                    "serveStale"   => true,
                ],
            ],
        ],
        "observatory" => [
            "subjectSelector"   => ["auto-group"],
            "probeURL"          => $probe,
            "enableConcurrency" => true,
        ],
        "inbounds" => [
            [
                "tag"      => "socks",
                "port"     => 10808,
                "listen"   => "127.0.0.1",
                "protocol" => "socks",
                "sniffing" => [
                    "enabled"      => true,
                    "routeOnly"    => false,
                    "destOverride" => ["http","tls","quic"],
                ],
                "settings" => [
                    "auth"      => "noauth",
                    "udp"       => true,
                    "ip"        => "127.0.0.1",
                    "userLevel" => 0,
                ],
            ],
            [
                "tag"      => "dns-in",
                "port"     => 10853,
                "listen"   => "127.0.0.1",
                "protocol" => "dokodemo-door",
                "settings" => [
                    "network"        => "tcp,udp",
                    "followRedirect" => false,
                    "address"        => "1.1.1.1",
                    "port"           => 53,
                ],
            ],
        ],
        "outbounds" => [
            [
                "tag"      => "direct",
                "protocol" => "freedom",
                "settings" => [
                    "domainStrategy" => "UseIP",
                ],
            ],
            [
                "tag"      => "block",
                "protocol" => "blackhole",
                "settings" => (object)[],
            ],
            [
                "tag"      => "dns-out",
                "protocol" => "dns",
                "settings" => (object)[],
            ],
        ],
        "routing" => [
            "domainStrategy" => "IPIfNonMatch",
            "domainMatcher"  => "hybrid",
            "balancers" => [
                [
                    "tag"      => "auto-group",
                    "selector" => [],
                    "strategy" => ["type" => "leastPing"],
                ],
            ],
            "rules" => [
                [
                    "type"        => "field",
                    "inboundTag"  => ["dns-in"],
                    "outboundTag" => "dns-out",
                ],
                [
                    "type"        => "field",
                    "protocol"    => ["bittorrent"],
                    "outboundTag" => "block",
                ],
                [
                    "type"        => "field",
                    "port"        => "6881-6999",
                    "outboundTag" => "block",
                ],
                [
                    "type"        => "field",
                    "ip"          => [
                        "10.0.0.0/8",
                        "172.16.0.0/12",
                        "192.168.0.0/16",
                        "127.0.0.0/8",
                    ],
                    "outboundTag" => "direct",
                ],
                [
                    "type"        => "field",
                    "network"     => "tcp,udp",
                    "balancerTag" => "auto-group",
                ],
            ],
        ],
    ];

    if ($ENABLE_FAKEDNS) {
        $template['fakedns'] = [
            [
                "ipPool"   => "198.18.0.0/15",
                "poolSize" => 10000,
            ],
        ];

        if (isset($template['inbounds'][0]['sniffing']['destOverride'])
            && is_array($template['inbounds'][0]['sniffing']['destOverride'])) {
            if (!in_array('fakedns', $template['inbounds'][0]['sniffing']['destOverride'], true)) {
                $template['inbounds'][0]['sniffing']['destOverride'][] = 'fakedns';
            }
        }
    }

    return $template;
}

/** Add advanced observatory config based on outbounds. */
function addObservatoryToConfig(array &$jsonConfig, array $preferredSelectors = []) {
    if (!isset($jsonConfig['outbounds']) || !is_array($jsonConfig['outbounds'])) return;

    $selectors = [];
    foreach ($jsonConfig['outbounds'] as $ob) {
        if (!isset($ob['tag'])) continue;
        $tag = $ob['tag'];
        if (in_array($tag, ['direct', 'dns-out', 'block'], true)) continue;
        $selectors[] = $tag;
    }

    $selectors = array_values(array_unique($selectors));

    if (!empty($preferredSelectors)) {
        $preferredSelectors = array_values(array_unique($preferredSelectors));
        $intersection       = array_values(array_intersect($preferredSelectors, $selectors));
        $selectors          = !empty($intersection) ? $intersection : $preferredSelectors;
    }

    if (empty($selectors)) return;

    $jsonConfig['observatory']['subjectSelector']   = $selectors;
    $jsonConfig['observatory']['enableConcurrency'] = true;
}

/* ---------- ست کردن allowInsecure روی همه Outbound ها ---------- */
function forceAllowInsecure(&$template) {
    if (!isset($template['outbounds']) || !is_array($template['outbounds'])) return;

    foreach ($template['outbounds'] as &$ob) {
        if (!isset($ob['streamSettings']) || !is_array($ob['streamSettings'])) continue;

        $ss =& $ob['streamSettings'];
        $security = $ss['security'] ?? '';

        if ($security === 'tls') {
            if (!isset($ss['tlsSettings']) || !is_array($ss['tlsSettings'])) $ss['tlsSettings'] = [];
            $ss['tlsSettings']['allowInsecure'] = true;
        }

        if ($security === 'reality') {
            if (!isset($ss['realitySettings']) || !is_array($ss['realitySettings'])) $ss['realitySettings'] = [];
            $ss['realitySettings']['allowInsecure'] = true;
        }
    }
}

/* ===================== FEATURE FLAGS APPLY (false => حذف کامل) ===================== */

function apply_feature_flags_to_outbound(array $ob): array {
    global $FEATURES;

    // --- MUX ---
    if (empty($FEATURES['enable_mux'])) {
        if (isset($ob['mux'])) unset($ob['mux']);
    } else {
        if (!isset($ob['mux']) || !is_array($ob['mux'])) $ob['mux'] = [];
        $ob['mux']['enabled'] = true;
        if (!isset($ob['mux']['concurrency'])) $ob['mux']['concurrency'] = 8;
        if (!isset($ob['mux']['xudpConcurrency'])) $ob['mux']['xudpConcurrency'] = 8;
        if (!isset($ob['mux']['xudpProxyUDP443'])) $ob['mux']['xudpProxyUDP443'] = 'reject';
    }

    // --- SOCKOPT ---
    if (empty($FEATURES['enable_sockopt'])) {
        if (isset($ob['streamSettings']['sockopt'])) unset($ob['streamSettings']['sockopt']);
    }

    // --- PADDING ---
    if (empty($FEATURES['enable_padding'])) {
        if (isset($ob['streamSettings']['padding'])) unset($ob['streamSettings']['padding']);
    }

    // --- FRAGMENT (زیرمجموعه sockopt) ---
    if (empty($FEATURES['enable_fragment'])) {
        if (isset($ob['streamSettings']['sockopt']['fragment'])) {
            unset($ob['streamSettings']['sockopt']['fragment']);
        }
    }

    // --- XHTTP ---
    if (empty($FEATURES['enable_xhttp'])) {
        if (isset($ob['streamSettings']['network']) && strtolower((string)$ob['streamSettings']['network']) === 'xhttp') {
            $ob['streamSettings']['network'] = 'tcp';
        }
        if (isset($ob['streamSettings']['xhttpSettings'])) unset($ob['streamSettings']['xhttpSettings']);
    }

    return $ob;
}

function apply_feature_flags_to_template(array &$template): void {
    global $FEATURES;

    if (isset($template['outbounds']) && is_array($template['outbounds'])) {
        foreach ($template['outbounds'] as $i => $ob) {
            if (!is_array($ob)) continue;
            $template['outbounds'][$i] = apply_feature_flags_to_outbound($ob);

            if (empty($FEATURES['enable_metadata'])) {
                if (isset($template['outbounds'][$i]['__health'])) unset($template['outbounds'][$i]['__health']);
                if (isset($template['outbounds'][$i]['__ping'])) unset($template['outbounds'][$i]['__ping']);
            }
        }
    }

    if (empty($FEATURES['enable_metadata'])) {
        if (isset($template['__meta'])) unset($template['__meta']);
    }
}

/** ===================== main aggregation (normal + ads + dns) ===================== */

$ALLOWED_PROTOCOLS = ['vless', 'vmess', 'trojan', 'shadowsocks'];

// اگر ADS_LINKS تعریف نشده بود، warning نده
$ADS_LINKS = $ADS_LINKS ?? [];

$servers     = [];
$adsServers  = [];
$dnsServers  = [];

$seenTags    = ['direct' => true, 'block' => true, 'dns-out' => true];

$mainTags    = [];
$adsTags     = [];
$dnsTags     = [];

LOG_STEP('INFO', 'aggregation_start', [
    'links'     => count($LINKS),
    'dns_links' => count($DNS_LINKS),
    'ads_links' => count($ADS_LINKS),
]);

/* ۱) سرورهای معمولی */
shuffle($LINKS);

foreach ($LINKS as $url) {
    $attempts = 0;
    while ($attempts < 2) {
        $attempts++;
        LOG_STEP('DEBUG', 'main_fetch_try', ['url'=>$url,'attempt'=>$attempts]);

        $raw = http_get_nocache($url, $HTTP_TIMEOUT);
        if ($raw === null) continue;

        $added = smart_extract_outbounds($raw);
        LOG_STEP('INFO', 'main_parse_outbounds', ['url'=>$url,'count'=>count($added)]);
        if (!$added) continue;

        foreach ($added as $ob) {
            $proto = strtolower($ob['protocol'] ?? '');
            if ($proto === '' || !in_array($proto, $ALLOWED_PROTOCOLS, true)) continue;
            if (!outbound_is_valid($ob)) {
                LOG_STEP('DEBUG', 'main_outbound_invalid', ['url'=>$url,'proto'=>$proto,'tag'=>($ob['tag']??'')]);
                continue;
            }

            if (!empty($FEATURES['enable_sockopt']) || !empty($FEATURES['enable_padding']) || !empty($FEATURES['enable_fragment'])) {
                $ob = add_sockopt_to_outbound($ob);
            }

            LOG_STEP('DEBUG', 'main_health_start', ['url'=>$url,'proto'=>$proto,'tag'=>($ob['tag']??'')]);
            $health = outbound_health($ob, $HEALTH_TIMEOUT);
            LOG_STEP($health['ok'] ? 'INFO' : 'DEBUG', 'main_health_result', [
                'url'    => $url,
                'ok'     => $health['ok'],
                'rtt_ms' => $health['rtt_ms'],
                'dns_ok' => $health['dns_ok'],
                'tcp_ok' => $health['tcp_ok'],
                'tls_ok' => $health['tls_ok'],
                'host'   => $health['endpoint']['host'] ?? null,
                'port'   => $health['endpoint']['port'] ?? null,
            ]);
            if (!$health['ok']) continue;

            $base      = $ob['tag'] ?? ($proto . '_');
            $ob['tag'] = unique_tag(preg_replace('~[^a-zA-Z0-9_\-]~', '_', $base . '_'), $seenTags);

            if (!empty($FEATURES['enable_metadata'])) {
                $ob['__health'] = $health;
                $ob['__ping']   = $health['rtt_ms'] ?? -1;
            }

            $servers[]  = $ob;
            $mainTags[] = $ob['tag'];
        }

        break;
    }
}

/* ۲) سرورهای مخصوص ادز */
if (!empty($ADS_LINKS)) {
    shuffle($ADS_LINKS);

    foreach ($ADS_LINKS as $url) {
        $attempts = 0;
        while ($attempts < 2) {
            $attempts++;
            LOG_STEP('DEBUG', 'ads_fetch_try', ['url'=>$url,'attempt'=>$attempts]);

            $raw = http_get_nocache($url, $HTTP_TIMEOUT);
            if ($raw === null) continue;

            $added = smart_extract_outbounds($raw);
            LOG_STEP('INFO', 'ads_parse_outbounds', ['url'=>$url,'count'=>count($added)]);
            if (!$added) continue;

            foreach ($added as $ob) {
                $proto = strtolower($ob['protocol'] ?? '');
                if ($proto === '' || !in_array($proto, $ALLOWED_PROTOCOLS, true)) continue;
                if (!outbound_is_valid($ob)) {
                    LOG_STEP('DEBUG', 'ads_outbound_invalid', ['url'=>$url,'proto'=>$proto,'tag'=>($ob['tag']??'')]);
                    continue;
                }

                if (!empty($FEATURES['enable_sockopt']) || !empty($FEATURES['enable_padding']) || !empty($FEATURES['enable_fragment'])) {
                    $ob = add_sockopt_to_outbound($ob);
                }

                LOG_STEP('DEBUG', 'ads_health_start', ['url'=>$url,'proto'=>$proto,'tag'=>($ob['tag']??'')]);
                $health = outbound_health($ob, $HEALTH_TIMEOUT);
                LOG_STEP($health['ok'] ? 'INFO' : 'DEBUG', 'ads_health_result', [
                    'url'    => $url,
                    'ok'     => $health['ok'],
                    'rtt_ms' => $health['rtt_ms'],
                    'dns_ok' => $health['dns_ok'],
                    'tcp_ok' => $health['tcp_ok'],
                    'tls_ok' => $health['tls_ok'],
                    'host'   => $health['endpoint']['host'] ?? null,
                    'port'   => $health['endpoint']['port'] ?? null,
                ]);
                if (!$health['ok']) continue;

                $base      = $ob['tag'] ?? ($proto . '_');
                $ob['tag'] = unique_tag(preg_replace('~[^a-zA-Z0-9_\-]~', '_', $base . '_'), $seenTags);

                if (!empty($FEATURES['enable_metadata'])) {
                    $ob['__health'] = $health;
                    $ob['__ping']   = $health['rtt_ms'] ?? -1;
                }

                $adsServers[] = $ob;
                $adsTags[]    = $ob['tag'];
            }

            break;
        }
    }
}

/* ۳) DNS PROXY SERVERS (مثل سرور اصلی: health + failover) */
if (!empty($DNS_LINKS)) {
    $tmpLinks = $DNS_LINKS;
    shuffle($tmpLinks);

    foreach ($tmpLinks as $url) {
        $attempts = 0;
        while ($attempts < 2) {
            $attempts++;
            LOG_STEP('DEBUG', 'dns_fetch_try', ['url'=>$url,'attempt'=>$attempts]);

            $raw = http_get_nocache($url, $HTTP_TIMEOUT);
            if ($raw === null) continue;

            $added = smart_extract_outbounds($raw);
            LOG_STEP('INFO', 'dns_parse_outbounds', ['url'=>$url,'count'=>count($added)]);
            if (!$added) continue;

            foreach ($added as $ob) {
                $proto = strtolower($ob['protocol'] ?? '');
                if ($proto === '' || !in_array($proto, $ALLOWED_PROTOCOLS, true)) continue;
                if (!outbound_is_valid($ob)) {
                    LOG_STEP('DEBUG', 'dns_outbound_invalid', ['url'=>$url,'proto'=>$proto,'tag'=>($ob['tag']??'')]);
                    continue;
                }

                if (!empty($FEATURES['enable_sockopt']) || !empty($FEATURES['enable_padding']) || !empty($FEATURES['enable_fragment'])) {
                    $ob = add_sockopt_to_outbound($ob);
                }

                LOG_STEP('DEBUG', 'dns_health_start', ['url'=>$url,'proto'=>$proto,'tag'=>($ob['tag']??'')]);
                $health = outbound_health($ob, $HEALTH_TIMEOUT);
                LOG_STEP($health['ok'] ? 'INFO' : 'DEBUG', 'dns_health_result', [
                    'url'    => $url,
                    'ok'     => $health['ok'],
                    'rtt_ms' => $health['rtt_ms'],
                    'dns_ok' => $health['dns_ok'],
                    'tcp_ok' => $health['tcp_ok'],
                    'tls_ok' => $health['tls_ok'],
                    'host'   => $health['endpoint']['host'] ?? null,
                    'port'   => $health['endpoint']['port'] ?? null,
                ]);
                if (!$health['ok']) continue;

                // tag را با dns_ شروع کن که تمیز جدا باشد
                $base      = 'dns_' . ($ob['tag'] ?? ($proto . '_'));
                $ob['tag'] = unique_tag(preg_replace('~[^a-zA-Z0-9_\-]~', '_', $base . '_'), $seenTags);

                if (!empty($FEATURES['enable_metadata'])) {
                    $ob['__health'] = $health;
                    $ob['__ping']   = $health['rtt_ms'] ?? -1;
                }

                $dnsServers[] = $ob;
                $dnsTags[]    = $ob['tag'];
            }

            break;
        }
    }
}

/* ===================== PATCH #1: اگر هیچ outbound سالمی نبود، هیچ چیز چاپ نشود ===================== */
if (empty($servers) && empty($adsServers) && empty($dnsServers)) {
    LOG_STEP('WARN', 'no_outbound_ok');
    http_response_code(204); // No Content
    exit;
}

/* ۴) ساخت template نهایی */
$template = build_template();

foreach ($servers as $ob)     { $template['outbounds'][] = $ob; }
foreach ($adsServers as $ob)  { $template['outbounds'][] = $ob; }
foreach ($dnsServers as $ob)  { $template['outbounds'][] = $ob; }

forceAllowInsecure($template);

/* ===================== PATCH #2: منطق درست جایگزینی (Fallback Selectors) ===================== */
$mainSelector = $mainTags;
if (empty($mainSelector) && !empty($adsTags)) $mainSelector = $adsTags; // اضطراری
if (empty($mainSelector) && !empty($dnsTags)) $mainSelector = $dnsTags; // اضطراری

$adsSelector  = !empty($adsTags) ? $adsTags : $mainSelector; // اگر ads نبود -> main
$dnsSelector  = !empty($dnsTags) ? $dnsTags : $mainSelector; // اگر dns نبود -> main

/* تنظیم بالانسر اصلی (auto-group همیشه main) */
$template['routing']['balancers'][0]['selector'] = $mainSelector;

/* balancer ادز (همیشه ساخته می‌شود، با fallback) */
$template['routing']['balancers'][] = [
    'tag'      => 'auto-ads',
    'selector' => $adsSelector,
    'strategy' => ['type' => 'leastPing'],
];

/* balancer DNS (همیشه ساخته می‌شود، با fallback) */
$template['routing']['balancers'][] = [
    'tag'      => 'auto-dns',
    'selector' => $dnsSelector,
    'strategy' => ['type' => 'leastPing'],
];

/* observatory برای همه سرورها */
$allTags = array_values(array_unique(array_merge($mainTags, $adsTags, $dnsTags)));
addObservatoryToConfig($template, $allTags);

/* meta */
$template['__meta'] = [
    'health'  => ['requireAll' => CHECK_REQUIRE_ALL],
    'network' => ['ipv6Blocked' => false],
];

/* ۵) قوانین روتینگ برای ترافیک گوگل/یوتیوب/ادز → auto-ads
 * PATCH #3: شرط بر اساس adsSelector (نه adsTags)
 */
if (!empty($adsSelector)) {
    $googleDomains = [
        "googleads.g.doubleclick.net",
        "geosite:category-ads-all",
        "geosite:google",
        "pagead2.googlesyndication.com",
        "tpc.googlesyndication.com",
        "www.googleadservices.com",
        "adservice.google.com",
        "adservice.google.*",
        "stats.g.doubleclick.net",

        "www.googletagmanager.com",
        "www.google-analytics.com",
        "ssl.google-analytics.com",
        "app-measurement.com",
        "www.googletagservices.com",

        "fcm.googleapis.com",
        "firebaseinstallations.googleapis.com",
        "firebaseremoteconfig.googleapis.com",
        "crashlytics.googleapis.com",

        "play.google.com",
        "play.googleapis.com",
        "android.googleapis.com",
        "android.clients.google.com",

        "www.google.com",
        "www.googleapis.com",
        "oauth2.googleapis.com",
        "accounts.google.com",
        "gstaticadssl.l.google.com",
        "fonts.gstatic.com",
        "www.gstatic.com",
        "clients3.google.com",
        "clients4.google.com",
        "clients5.google.com",

        "www.youtube.com",
        "m.youtube.com",
        "youtubei.googleapis.com",
        "i.ytimg.com",
        "s.youtube.com",
        "yt3.ggpht.com",
        "googlevideo.com",
        "r.googlevideo.com",

        "google.com",
        "googleapis.com",
        "googleusercontent.com",
        "gvt1.com",
        "gvt2.com",
        "gstatic.com",
    ];

    $rules    = $template['routing']['rules'] ?? [];
    $newRules = [];

    $newRules[] = [
        "type"        => "field",
        "network"     => "tcp,udp",
        "domain"      => $googleDomains,
        "balancerTag" => "auto-ads",
    ];

    foreach ($rules as $rule) { $newRules[] = $rule; }
    $template['routing']['rules'] = $newRules;
}

/* ۶) قفل DNS: جلوگیری از رفتن DoH/DoT/QUIC به direct و عبور از auto-dns */
$dnsDomains = [
    "dns.google",
    "cloudflare-dns.com",
    "security.cloudflare-dns.com",
    "one.one.one.one",
    "family.cloudflare-dns.com",
    "dns.quad9.net",
];

$dnsIps = [
    "1.1.1.1",
    "1.0.0.1",
    "8.8.8.8",
    "8.8.4.4",
    "9.9.9.9",
    "149.112.112.112",
];

// prepend rules (خیلی مهم: باید قبل از rule عمومی بیاد)
$rules = $template['routing']['rules'] ?? [];
$prepend = [];

// DoT tcp/853
$prepend[] = [
    "type"        => "field",
    "network"     => "tcp",
    "ip"          => $dnsIps,
    "port"        => "853",
    "balancerTag" => "auto-dns",
];

// DoH tcp/443 به IP
$prepend[] = [
    "type"        => "field",
    "network"     => "tcp",
    "ip"          => $dnsIps,
    "port"        => "443",
    "balancerTag" => "auto-dns",
];

// QUIC / DoH3 udp/443 به IP
$prepend[] = [
    "type"        => "field",
    "network"     => "udp",
    "ip"          => $dnsIps,
    "port"        => "443",
    "balancerTag" => "auto-dns",
];

// DNS udp/53 به IP
$prepend[] = [
    "type"        => "field",
    "network"     => "udp",
    "ip"          => $dnsIps,
    "port"        => "53",
    "balancerTag" => "auto-dns",
];

// DNS domains (tcp/udp هرچی) → auto-dns
$prepend[] = [
    "type"        => "field",
    "domain"      => $dnsDomains,
    "network"     => "tcp,udp",
    "balancerTag" => "auto-dns",
];

$template['routing']['rules'] = array_merge($prepend, $rules);

/**
 * ====== اعمال نهایی فلگ‌ها ======
 * باید بعد از ساخت کامل template باشد
 */
apply_feature_flags_to_template($template);

LOG_STEP('INFO', 'building_output', [
    'main_count' => count($servers),
    'ads_count'  => count($adsServers),
    'dns_count'  => count($dnsServers),
    'outbounds_total' => isset($template['outbounds']) && is_array($template['outbounds']) ? count($template['outbounds']) : 0
]);

// خروجی نهایی + ذخیره history (هر خط یک خروجی)
$outJson = json_encode($template, JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE);
if ($outJson === false) {
    LOG_STEP('ERROR', 'json_encode_failed', ['json_error' => json_last_error_msg()]);
    http_response_code(500);
    exit;
}
LOG_STEP('INFO', 'json_ready', ['bytes'=>strlen($outJson)]);

if ($SAVE_OUTPUT_HISTORY) {
    cleanup_history_files($HISTORY_DIR, $HISTORY_TTL_SECONDS);
    save_output_line($HISTORY_DIR, $outJson);
    LOG_STEP('INFO', 'history_saved', ['dir'=>$HISTORY_DIR]);
}

LOG_STEP('INFO', 'request_done');
echo $outJson;