<?php
declare(strict_types=1);

/**
 * trun.php — stable config fetcher (never die, always fallback)
 * PHP 7.4+
 */

/* ===================== SETTINGS ===================== */

// منبع/منابع دریافت کانفیگ
$SOURCE_URLS = [
    "https://fitn1.ir/Api/Bax/trun.php",
];

// کش
$CACHE_FILE = __DIR__ . "/cnnnhhhacjkdkkdhe.txt";
$CACHE_TTL  = 6 * 60; // 6 دقیقه

// خروجی سالم آخر (fallback)
$LAST_GOOD_FILE = __DIR__ . "/last_good_output.txt";

// انتخاب دونه‌دونه (ترتیبی)
$POINTER_FILE = __DIR__ . "/pointer.idx";

// لاگ
$LOG_DIR  = __DIR__ . "/logs";
$LOG_FILE = $LOG_DIR . "/trun.log";

// تایم‌اوت‌ها
$HTTP_TIMEOUT   = 5;   // کل درخواست
$CONNECT_TIMEOUT= 3;   // اتصال
$RETRY          = 2;   // تلاش مجدد
$REQUEST_TIMEOUT = 15; // سقف اجرای همین اسکریپت

header("Content-Type: text/plain; charset=utf-8");
@set_time_limit($REQUEST_TIMEOUT);
@ini_set('max_execution_time', (string)$REQUEST_TIMEOUT);

/* ===================== LOGGER ===================== */

function ensure_dir(string $dir): void {
    if (!is_dir($dir)) @mkdir($dir, 0777, true);
}

function mem_str(): string {
    $m = memory_get_usage(true);
    $p = memory_get_peak_usage(true);
    return "mem=" . (int)round($m/1024/1024) . "MB peak=" . (int)round($p/1024/1024) . "MB";
}

$RID = date("Ymd-His") . "-" . substr(bin2hex(random_bytes(8)), 0, 8);
$T0  = microtime(true);

function log_step(string $lvl, string $step, array $ctx = []): void {
    global $LOG_FILE, $RID, $T0;
    ensure_dir(dirname($LOG_FILE));
    $ms = (int)round((microtime(true) - $T0) * 1000);

    $base = [
        "ts"   => date("Y-m-d H:i:s"),
        "rid"  => $RID,
        "ms"   => $ms,
        "lvl"  => $lvl,
        "step" => $step,
        "mem"  => mem_str(),
    ];

    $line = json_encode(array_merge($base, $ctx), JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
    @file_put_contents($LOG_FILE, $line . "\n", FILE_APPEND | LOCK_EX);
}

// هندل خطاها و فتال
set_error_handler(function($severity, $message, $file, $line) {
    log_step("ERROR", "php_error", [
        "severity" => $severity,
        "msg" => $message,
        "file" => $file,
        "line" => $line,
    ]);
    return true;
});

register_shutdown_function(function() {
    $e = error_get_last();
    if ($e && in_array($e['type'], [E_ERROR, E_PARSE, E_CORE_ERROR, E_COMPILE_ERROR], true)) {
        log_step("ERROR", "php_fatal", [
            "type" => $e['type'],
            "msg"  => $e['message'],
            "file" => $e['file'],
            "line" => $e['line'],
        ]);
    }
    log_step("INFO", "request_end");
});

log_step("INFO", "request_start", [
    "ip" => $_SERVER['HTTP_CF_CONNECTING_IP'] ?? $_SERVER['REMOTE_ADDR'] ?? "",
    "uri" => $_SERVER['REQUEST_URI'] ?? "",
    "method" => $_SERVER['REQUEST_METHOD'] ?? "",
    "ua" => $_SERVER['HTTP_USER_AGENT'] ?? "",
]);

/* ===================== HTTP FETCH (robust) ===================== */

function http_get(string $url, int $connectTimeout, int $timeout): array {
    $t0 = microtime(true);

    if (!function_exists("curl_init")) {
        // fallback stream
        $ctx = stream_context_create([
            "http" => [
                "timeout" => $timeout,
                "ignore_errors" => true,
                "header" =>
                    "User-Agent: Mozilla/5.0\r\n" .
                    "Cache-Control: no-cache\r\n" .
                    "Pragma: no-cache\r\n" .
                    "Connection: close\r\n",
            ],
            "ssl" => [
                "verify_peer" => false,
                "verify_peer_name" => false,
            ]
        ]);

        $data = @file_get_contents($url, false, $ctx);
        $ms = (int)round((microtime(true) - $t0) * 1000);
        return [
            "ok" => ($data !== false && $data !== ""),
            "code" => ($data !== false ? 200 : 0),
            "body" => ($data !== false ? $data : ""),
            "err" => ($data === false ? "stream_failed" : ""),
            "ms" => $ms,
            "bytes" => ($data !== false ? strlen($data) : 0),
            "hdr" => [],
            "meta" => [],
        ];
    }

    $hdr = [];
    $ch = curl_init($url);
    curl_setopt_array($ch, [
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_FOLLOWLOCATION => true,
        CURLOPT_MAXREDIRS      => 3,
        CURLOPT_CONNECTTIMEOUT => $connectTimeout,
        CURLOPT_TIMEOUT        => $timeout,
        CURLOPT_SSL_VERIFYPEER => false,
        CURLOPT_SSL_VERIFYHOST => false,
        CURLOPT_USERAGENT      => "Mozilla/5.0",
        CURLOPT_HTTPHEADER     => [
            "Cache-Control: no-cache, no-store, must-revalidate",
            "Pragma: no-cache",
        ],
        // خیلی مهم: gzip/br decode
        CURLOPT_ENCODING       => "",
        CURLOPT_HEADERFUNCTION => function($ch, $line) use (&$hdr) {
            $len = strlen($line);
            $line = trim($line);
            if ($line === "" || strpos($line, ":") === false) return $len;
            [$k,$v] = explode(":", $line, 2);
            $hdr[strtolower(trim($k))] = trim($v);
            return $len;
        },
    ]);

    $body = curl_exec($ch);
    $err  = curl_error($ch);
    $code = (int)curl_getinfo($ch, CURLINFO_HTTP_CODE);

    $meta = [
        "content_type" => (string)curl_getinfo($ch, CURLINFO_CONTENT_TYPE),
        "primary_ip"   => (string)curl_getinfo($ch, CURLINFO_PRIMARY_IP),
        "total_ms"     => (int)round(curl_getinfo($ch, CURLINFO_TOTAL_TIME) * 1000),
        "connect_ms"   => (int)round(curl_getinfo($ch, CURLINFO_CONNECT_TIME) * 1000),
        "ttfb_ms"      => (int)round(curl_getinfo($ch, CURLINFO_STARTTRANSFER_TIME) * 1000),
    ];

    curl_close($ch);

    $ms = (int)round((microtime(true) - $t0) * 1000);
    $ok = ($body !== false && $body !== "" && $code >= 200 && $code < 400);

    return [
        "ok" => $ok,
        "code" => $code,
        "body" => ($body !== false ? $body : ""),
        "err" => ($ok ? "" : ($err ?: "http_failed")),
        "ms" => $ms,
        "bytes" => ($body !== false ? strlen((string)$body) : 0),
        "hdr" => $hdr,
        "meta" => $meta,
    ];
}

/* ===================== CACHE HELPERS ===================== */

function cache_is_fresh(string $file, int $ttl): bool {
    return is_file($file) && (time() - (int)@filemtime($file)) < $ttl;
}

function cache_write_atomic(string $file, string $data): void {
    $tmp = $file . ".tmp";
    @file_put_contents($tmp, $data, LOCK_EX);
    @rename($tmp, $file);
}

function read_file_safe(string $file): string {
    $s = @file_get_contents($file);
    return ($s === false) ? "" : $s;
}

/* ===================== EXTRACTION ===================== */

function extract_link_configs(string $text): array {
    // دقیق: فقط بر اساس prefix
    $out = [];

    // روی هر خط + همچنین داخل متن
    $patterns = [
        '/\bvless:\/\/[^\s<>"\']+/i',
        '/\bvmess:\/\/[^\s<>"\']+/i',
        '/\btrojan:\/\/[^\s<>"\']+/i',
        '/\bss:\/\/[^\s<>"\']+/i',
    ];

    foreach ($patterns as $re) {
        if (preg_match_all($re, $text, $m)) {
            foreach ($m[0] as $s) $out[] = trim($s);
        }
    }
    return $out;
}

function extractJsonConfigs(string $text): array {
    $results = [];
    $len = strlen($text);
    $i = 0;

    while ($i < $len) {
        $ch = $text[$i];
        if ($ch === '{' || $ch === '[') {
            $start = $i;
            $depth = 0;
            $inString = false;
            $escape = false;

            for (; $i < $len; $i++) {
                $c = $text[$i];

                if ($inString) {
                    if ($escape) {
                        $escape = false;
                    } else {
                        if ($c === '\\') $escape = true;
                        elseif ($c === '"') $inString = false;
                    }
                } else {
                    if ($c === '"') $inString = true;
                    elseif ($c === '{' || $c === '[') $depth++;
                    elseif ($c === '}' || $c === ']') {
                        $depth--;
                        if ($depth === 0) {
                            $jsonStr = substr($text, $start, $i - $start + 1);
                            $decoded = json_decode($jsonStr, true);
                            if (is_array($decoded) && json_last_error() === JSON_ERROR_NONE) {
                                // فقط چیزی که واقعاً config باشه
                                if (isset($decoded['outbounds']) || isset($decoded['inbounds']) || isset($decoded['protocol'])) {
                                    $results[] = $jsonStr;
                                }
                            }
                            break;
                        }
                    }
                }
            }
        }
        $i++;
    }

    return array_values(array_unique($results));
}

function dedupe_preserve_order(array $items): array {
    $seen = [];
    $out = [];
    foreach ($items as $s) {
        $k = sha1($s);
        if (isset($seen[$k])) continue;
        $seen[$k] = true;
        $out[] = $s;
    }
    return $out;
}

/* ===================== POINTER (one-by-one) ===================== */

function pick_next_config(array $configs, string $pointerFile): string {
    $n = count($configs);
    if ($n === 0) return "";

    $idx = 0;
    if (is_file($pointerFile)) {
        $raw = trim((string)@file_get_contents($pointerFile));
        if ($raw !== '' && ctype_digit($raw)) $idx = (int)$raw;
    }

    if ($idx < 0 || $idx >= $n) $idx = 0;

    $chosen = $configs[$idx];

    $next = $idx + 1;
    if ($next >= $n) $next = 0;

    @file_put_contents($pointerFile, (string)$next, LOCK_EX);
    return $chosen;
}

/* ===================== MAIN ===================== */

log_step("INFO", "run_start", [
    "sources" => count($SOURCE_URLS),
    "cache_ttl" => $CACHE_TTL,
]);

$response = "";

// 1) اگر کش تازه است، از کش
if (cache_is_fresh($CACHE_FILE, $CACHE_TTL)) {
    $response = read_file_safe($CACHE_FILE);
    log_step("INFO", "cache_hit", ["bytes" => strlen($response)]);
} else {
    // 2) اگر کش تازه نیست، fetch با retry
    $ok = false;
    shuffle($SOURCE_URLS);

    foreach ($SOURCE_URLS as $u) {
        for ($a = 1; $a <= $RETRY; $a++) {
            log_step("DEBUG", "fetch_try", ["url"=>$u, "attempt"=>$a]);

            $r = http_get($u, $CONNECT_TIMEOUT, $HTTP_TIMEOUT);

            if ($r["ok"]) {
                $response = $r["body"];
                $ok = true;
                log_step("INFO", "fetch_ok", [
                    "url"=>$u, "http_code"=>$r["code"], "bytes"=>$r["bytes"], "ms"=>$r["ms"],
                    "meta"=>$r["meta"], "hdr"=>$r["hdr"]
                ]);
                break 2;
            } else {
                log_step("WARN", "fetch_failed", [
                    "url"=>$u, "http_code"=>$r["code"], "bytes"=>$r["bytes"], "ms"=>$r["ms"],
                    "err"=>$r["err"], "meta"=>$r["meta"], "hdr"=>$r["hdr"]
                ]);
            }
        }
    }

    // 3) اگر fetch موفق شد → کش را به‌روز کن
    if ($ok && $response !== "") {
        cache_write_atomic($CACHE_FILE, $response);
        log_step("INFO", "cache_write", ["bytes" => strlen($response)]);
    } else {
        // 4) اگر fetch شکست خورد → از کش قدیمی (stale) استفاده کن
        $stale = read_file_safe($CACHE_FILE);
        if ($stale !== "") {
            $response = $stale;
            log_step("WARN", "use_stale_cache", ["bytes" => strlen($response)]);
        } else {
            log_step("ERROR", "no_cache_no_fetch", []);
        }
    }
}

// 5) استخراج کانفیگ‌ها
$linkConfigs = extract_link_configs($response);
$jsonConfigs = extractJsonConfigs($response);

$configs = array_merge($linkConfigs, $jsonConfigs);
$configs = dedupe_preserve_order($configs);

log_step("INFO", "extract_done", [
    "link_count" => count($linkConfigs),
    "json_count" => count($jsonConfigs),
    "total" => count($configs),
    "resp_bytes" => strlen($response),
]);

// اگر هیچ config نبود: fallback از آخرین خروجی سالم
if (empty($configs)) {
    $last = read_file_safe($LAST_GOOD_FILE);
    if ($last !== "") {
        log_step("WARN", "fallback_last_good", ["bytes" => strlen($last)]);
        echo $last;
        exit;
    }
    // هیچ چی نداریم -> 204 (بدون توقف/بدون die)
    log_step("WARN", "no_configs_204", []);
    http_response_code(204);
    exit;
}

// 6) انتخاب دونه‌دونه
$chosen = pick_next_config($configs, $POINTER_FILE);
if ($chosen === "") {
    $last = read_file_safe($LAST_GOOD_FILE);
    if ($last !== "") {
        log_step("WARN", "picked_empty_fallback_last_good", ["bytes" => strlen($last)]);
        echo $last;
        exit;
    }
    log_step("WARN", "picked_empty_204", []);
    http_response_code(204);
    exit;
}

// 7) ذخیره آخرین خروجی سالم
@file_put_contents($LAST_GOOD_FILE, $chosen, LOCK_EX);

log_step("INFO", "serve_ok", [
    "chosen_len" => strlen($chosen),
    "chosen_sha1" => sha1($chosen),
]);

echo $chosen;