<?php
/**
 * trun.php — Single-config printer with:
 * - strict extraction (vless/vmess/trojan/ss + real xray/v2ray JSON blocks)
 * - never stop: on any error -> serve from history or 204
 * - step-by-step JSONL logging
 * - cache + lock + soft rate-limit
 * PHP 7.4+
 */

/* ===================== SETTINGS ===================== */

// سورس اصلی که این trun ازش می‌کشه (همون فایل بزرگ/اگریگیتور/لیست)
$SOURCE_URL = "https://fitn1.ir/Api/Bax/trun.php";

// Cache
$CACHE_DIR  = __DIR__ . "/cache";
$CACHE_TTL  = 6 * 60; // 6 minutes

// History (برای وقتی خطا شد یا ریت‌لیمیت/لاک شد)
$HISTORY_DIR = __DIR__ . "/history";
$HISTORY_TTL = 10 * 60; // 10 minutes
$SAVE_HISTORY = true;

// Lock + rate-limit
$LOCK_FILE = __DIR__ . "/trun.lock";
$LOCK_TTL  = 50;          // seconds
$RATE_LIMIT_SECONDS = 2;  // per IP

// Timeouts
$HTTP_TIMEOUT = 4; // seconds
$REQUEST_TIMEOUT = 25; // hard per request

// Logging
$LOG_DIR  = __DIR__ . "/logs";
$LOG_FILE = $LOG_DIR . "/trun.log";
$LOG_LEVEL = "DEBUG"; // DEBUG|INFO|WARN|ERROR

// Force HTTP on inbound request
$FORCE_HTTP = true;

/* ===================== BOOTSTRAP ===================== */

@set_time_limit((int)$REQUEST_TIMEOUT);
@ini_set('max_execution_time', (string)((int)$REQUEST_TIMEOUT));

header('Content-Type: text/plain; charset=utf-8');

function ensure_dir(string $dir): void {
    if (!is_dir($dir)) @mkdir($dir, 0777, true);
}

ensure_dir($CACHE_DIR);
ensure_dir($HISTORY_DIR);
ensure_dir($LOG_DIR);

function client_ip(): string {
    $ip = $_SERVER['HTTP_CF_CONNECTING_IP'] ?? $_SERVER['HTTP_X_FORWARDED_FOR'] ?? $_SERVER['REMOTE_ADDR'] ?? '0.0.0.0';
    if (strpos($ip, ',') !== false) $ip = trim(explode(',', $ip)[0]);
    return trim($ip);
}

function now_ms(): int {
    return (int)round(microtime(true) * 1000);
}

function mem_str(): string {
    $m = memory_get_usage(true);
    $p = memory_get_peak_usage(true);
    return "mem=" . round($m/1024/1024) . "MB peak=" . round($p/1024/1024) . "MB";
}

function rid(): string {
    $t = date('Ymd-His');
    $r = bin2hex(random_bytes(4));
    return $t . '-' . $r;
}

function lvl_rank(string $lvl): int {
    static $map = ['DEBUG'=>10,'INFO'=>20,'WARN'=>30,'ERROR'=>40];
    return $map[strtoupper($lvl)] ?? 20;
}

function log_event(array $ctx, string $lvl, string $step, array $data = []): void {
    global $LOG_FILE, $LOG_LEVEL;
    if (lvl_rank($lvl) < lvl_rank($LOG_LEVEL)) return;

    $row = array_merge([
        "ts"   => date('Y-m-d H:i:s'),
        "rid"  => $ctx['rid'],
        "ms"   => max(0, now_ms() - $ctx['t0']),
        "lvl"  => strtoupper($lvl),
        "step" => $step,
        "mem"  => mem_str(),
    ], $data);

    $line = json_encode($row, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
    @file_put_contents($LOG_FILE, $line . "\n", FILE_APPEND | LOCK_EX);
}

/* ===================== FORCE HTTP (ورودی کاربر) ===================== */
if ($FORCE_HTTP) {
    $httpsOn =
        (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off') ||
        (!empty($_SERVER['SERVER_PORT']) && (int)$_SERVER['SERVER_PORT'] === 443) ||
        (!empty($_SERVER['HTTP_X_FORWARDED_PROTO']) && strtolower($_SERVER['HTTP_X_FORWARDED_PROTO']) === 'https') ||
        (!empty($_SERVER['HTTP_CF_VISITOR']) && stripos($_SERVER['HTTP_CF_VISITOR'], '"scheme":"https"') !== false);

    if ($httpsOn) {
        $host = $_SERVER['HTTP_HOST'] ?? '';
        $uri  = $_SERVER['REQUEST_URI'] ?? '/';
        header('Location: http://' . $host . $uri, true, 302);
        header('Cache-Control: no-store, no-cache, must-revalidate, max-age=0');
        header('Pragma: no-cache');
        exit;
    }
}

/* ===================== ERROR HANDLERS ===================== */
$CTX = [
    'rid' => rid(),
    't0'  => now_ms(),
];

set_error_handler(function($errno, $errstr, $errfile, $errline) use (&$CTX) {
    log_event($CTX, "ERROR", "php_error", [
        "errno" => $errno,
        "err"   => $errstr,
        "file"  => $errfile,
        "line"  => $errline,
    ]);
    return true; // prevent default output
});

set_exception_handler(function($e) use (&$CTX) {
    log_event($CTX, "ERROR", "php_exception", [
        "err"  => $e->getMessage(),
        "file" => $e->getFile(),
        "line" => $e->getLine(),
    ]);
    serve_from_history_or_204($CTX, "exception");
});

register_shutdown_function(function() use (&$CTX) {
    $err = error_get_last();
    if ($err && in_array($err['type'], [E_ERROR,E_PARSE,E_CORE_ERROR,E_COMPILE_ERROR], true)) {
        log_event($CTX, "ERROR", "php_fatal", [
            "type" => $err['type'],
            "err"  => $err['message'],
            "file" => $err['file'],
            "line" => $err['line'],
        ]);
        // اگر فتال شد و هنوز خروجی نداده، از history بده
        if (!headers_sent()) {
            serve_from_history_or_204($CTX, "fatal");
        }
    }
});

/* ===================== HISTORY (Round-Robin) ===================== */

function cleanup_dir_ttl(string $dir, int $ttl): void {
    if (!is_dir($dir)) return;
    $now = time();
    foreach (glob(rtrim($dir,'/\\') . "/*.txt") ?: [] as $f) {
        $mt = @filemtime($f);
        if ($mt !== false && ($now - $mt) > $ttl) @unlink($f);
    }
}

function save_history_line(string $dir, string $line): void {
    ensure_dir($dir);
    $name = "outputs_" . date("Ymd_Hi") . ".txt";
    $path = rtrim($dir,'/\\') . "/" . $name;
    $fp = @fopen($path, "ab");
    if ($fp) {
        @fwrite($fp, $line . "\n");
        @fclose($fp);
    }
}

function history_one(string $dir): ?string {
    if (!is_dir($dir)) return null;
    $files = glob(rtrim($dir,'/\\') . "/outputs_*.txt");
    if (!$files) return null;
    sort($files);

    $ptr = rtrim($dir,'/\\') . "/pointer.idx";
    $raw = is_file($ptr) ? trim((string)@file_get_contents($ptr)) : "";
    $fi = 0; $li = 0;
    if ($raw !== "" && strpos($raw, ":") !== false) {
        [$a,$b] = explode(":", $raw, 2);
        $fi = max(0, (int)$a);
        $li = max(0, (int)$b);
    }
    if ($fi >= count($files)) { $fi = 0; $li = 0; }

    for ($round=0; $round<2; $round++) {
        for ($i=$fi; $i<count($files); $i++) {
            $lines = @file($files[$i], FILE_IGNORE_NEW_LINES);
            if (!$lines) { $li=0; continue; }

            $n = count($lines);
            while ($li < $n && trim((string)$lines[$li]) === "") $li++;

            if ($li < $n) {
                $out = (string)$lines[$li];
                $nextLi = $li + 1;
                $nextFi = $i;
                if ($nextLi >= $n) { $nextLi = 0; $nextFi = $i + 1; if ($nextFi >= count($files)) $nextFi = 0; }
                @file_put_contents($ptr, $nextFi . ":" . $nextLi, LOCK_EX);
                return $out;
            } else {
                $li=0;
            }
        }
        $fi=0; $li=0;
    }
    return null;
}

function serve_from_history_or_204(array $CTX, string $reason): void {
    global $HISTORY_DIR;
    cleanup_dir_ttl($HISTORY_DIR, (int)($GLOBALS['HISTORY_TTL'] ?? 600));
    $out = history_one($HISTORY_DIR);
    if ($out !== null && $out !== "") {
        log_event($CTX, "WARN", "serve_history", ["reason"=>$reason]);
        echo $out;
        exit;
    }
    log_event($CTX, "WARN", "serve_204", ["reason"=>$reason]);
    http_response_code(204);
    exit;
}

/* ===================== RATE LIMIT (SOFT) ===================== */

function rate_limit_soft(array $CTX, int $seconds): void {
    global $HISTORY_DIR;
    $ip = client_ip();
    $key = sys_get_temp_dir() . "/trun_rl_" . preg_replace('~[^a-zA-Z0-9_\-]~','_', $ip) . ".txt";
    $now = time();
    $last = is_file($key) ? (int)@file_get_contents($key) : 0;

    if ($last > 0 && ($now - $last) < $seconds) {
        log_event($CTX, "DEBUG", "rate_limited", ["ip"=>$ip, "limit"=>$seconds]);
        serve_from_history_or_204($CTX, "rate_limit");
    }
    @file_put_contents($key, (string)$now, LOCK_EX);
    log_event($CTX, "DEBUG", "rate_limit_ok", ["ip"=>$ip, "limit"=>$seconds]);
}

/* ===================== LOCK ===================== */

function try_lock(array $CTX, string $file, int $ttl): bool {
    $now = time();
    if (!is_file($file)) {
        $ok = (@file_put_contents($file, (string)$now . "\n" . getmypid(), LOCK_EX) !== false);
        if ($ok) log_event($CTX, "INFO", "lock_acquired", ["lock"=>$file, "ttl"=>$ttl]);
        return $ok;
    }
    $mt = @filemtime($file);
    if ($mt !== false && ($now - $mt) > $ttl) {
        @unlink($file);
        $ok = (@file_put_contents($file, (string)$now . "\n" . getmypid(), LOCK_EX) !== false);
        if ($ok) log_event($CTX, "INFO", "lock_reacquired", ["lock"=>$file, "ttl"=>$ttl]);
        return $ok;
    }
    return false;
}
function unlock(string $file): void { @unlink($file); }

/* ===================== FETCH (CURL) ===================== */

function http_get(string $url, int $timeout, array $CTX): array {
    $ch = curl_init($url);
    curl_setopt_array($ch, [
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_FOLLOWLOCATION => true,
        CURLOPT_CONNECTTIMEOUT => $timeout,
        CURLOPT_TIMEOUT        => $timeout,
        CURLOPT_SSL_VERIFYPEER => false,
        CURLOPT_SSL_VERIFYHOST => 0,
        CURLOPT_USERAGENT      => "Mozilla/5.0",
        CURLOPT_HTTPHEADER     => [
            "Cache-Control: no-cache, no-store, must-revalidate",
            "Pragma: no-cache",
            "Connection: close",
        ],
    ]);

    $body = curl_exec($ch);
    $err  = curl_error($ch);
    $code = (int)curl_getinfo($ch, CURLINFO_HTTP_CODE);
    $bytes = is_string($body) ? strlen($body) : 0;
    curl_close($ch);

    if (!is_string($body) || $body === '') {
        return ["ok"=>false, "code"=>$code, "err"=>$err ?: "empty", "body"=>null, "bytes"=>$bytes];
    }
    return ["ok"=>true, "code"=>$code, "err"=>null, "body"=>$body, "bytes"=>$bytes];
}

/* ===================== EXTRACTORS ===================== */

function normalize_lines(string $raw): string {
    // حذف BOM
    if (substr($raw, 0, 3) === "\xEF\xBB\xBF") $raw = substr($raw, 3);
    return str_replace(["\r\n","\r"], "\n", $raw);
}

function looks_like_html_or_error(string $s): bool {
    $t = trim($s);
    if ($t === '') return true;
    // اگر HTML یا پیام خطا یا هدرهای عجیب باشد
    if (stripos($t, '<html') !== false || stripos($t, '<!doctype') !== false) return true;
    if (stripos($t, 'خطا') !== false || stripos($t, 'error') !== false) {
        // ولی اگر خط vless/vmess/... باشد نباید رد شود
        if (preg_match('~^(vless|vmess|trojan|ss)://~i', $t)) return false;
        return true;
    }
    return false;
}

function is_proto_line(string $line): bool {
    return (bool)preg_match('~^(vless|vmess|trojan|ss)://~i', trim($line));
}

function extract_link_lines(string $text): array {
    $out = [];

    $lines = explode("\n", normalize_lines($text));
    foreach ($lines as $ln) {
        $ln = trim($ln);
        if ($ln === '') continue;

        // رد کردن چیزهایی مثل https:// یا http://
        if (preg_match('~^https?://~i', $ln)) continue;

        // فقط پروتکل‌های واقعی
        if (is_proto_line($ln)) {
            // پاکسازی فضای اضافی و کاراکترهای کنترل
            $ln = preg_replace('~[\x00-\x1F\x7F]+~', '', $ln);
            $out[] = $ln;
        }
    }

    // همچنین اگر داخل متن، لینک‌ها وسط متن باشند:
    if (preg_match_all('~\b(vless|vmess|trojan|ss)://[^\s<>"\']+~i', $text, $m)) {
        foreach ($m[0] as $hit) {
            $hit = trim($hit);
            if (preg_match('~^https?://~i', $hit)) continue;
            $hit = preg_replace('~[\x00-\x1F\x7F]+~', '', $hit);
            $out[] = $hit;
        }
    }

    $out = array_values(array_unique($out));
    return $out;
}

/**
 * استخراج دقیق بلوک‌های JSON متوازن {..} یا [..]
 * فقط اگر واقعاً کانفیگ xray/v2ray باشد (کلید outbounds یا inbounds داشته باشد)
 * خروجی: JSON خام (string)
 */
function extract_json_configs(string $text): array {
    $res = [];
    $s = normalize_lines($text);
    $len = strlen($s);
    $i = 0;

    while ($i < $len) {
        $ch = $s[$i];
        if ($ch === '{' || $ch === '[') {
            $start = $i;
            $depth = 0;
            $inString = false;
            $escape = false;

            for (; $i < $len; $i++) {
                $c = $s[$i];

                if ($inString) {
                    if ($escape) {
                        $escape = false;
                    } else {
                        if ($c === '\\') $escape = true;
                        elseif ($c === '"') $inString = false;
                    }
                } else {
                    if ($c === '"') $inString = true;
                    elseif ($c === '{' || $c === '[') $depth++;
                    elseif ($c === '}' || $c === ']') {
                        $depth--;
                        if ($depth === 0) {
                            $end = $i;
                            $jsonStr = substr($s, $start, $end - $start + 1);

                            $decoded = json_decode($jsonStr, true);
                            if ($decoded !== null && json_last_error() === JSON_ERROR_NONE) {
                                if (is_array($decoded) && (isset($decoded['outbounds']) || isset($decoded['inbounds']))) {
                                    $res[] = $jsonStr;
                                }
                            }
                            break;
                        }
                    }
                }
            }
        }
        $i++;
    }

    return array_values(array_unique($res));
}

/**
 * فیلتر نهایی: اگر خروجی خیلی مشکوک/خراب بود، رد کن
 */
function is_valid_output(string $item): bool {
    $t = trim($item);
    if ($t === '') return false;
    if (looks_like_html_or_error($t)) return false;

    // اگر لینک است
    if (preg_match('~^(vless|vmess|trojan|ss)://~i', $t)) {
        // اگر خیلی کوتاه باشد معمولاً خراب است
        if (strlen($t) < 15) return false;
        return true;
    }

    // اگر JSON است
    if (($t[0] ?? '') === '{' || ($t[0] ?? '') === '[') {
        $d = json_decode($t, true);
        if ($d === null || json_last_error() !== JSON_ERROR_NONE) return false;
        if (is_array($d) && (isset($d['outbounds']) || isset($d['inbounds']))) return true;
        return false;
    }

    return false;
}

/* ===================== MAIN ===================== */

log_event($CTX, "INFO", "request_start", [
    "ip"     => client_ip(),
    "uri"    => ($_SERVER['REQUEST_URI'] ?? ''),
    "method" => ($_SERVER['REQUEST_METHOD'] ?? ''),
    "ua"     => ($_SERVER['HTTP_USER_AGENT'] ?? ''),
]);

cleanup_dir_ttl($HISTORY_DIR, $HISTORY_TTL);

// نرم‌ریت‌لیمیت
rate_limit_soft($CTX, $RATE_LIMIT_SECONDS);

// قفل اجرای همزمان (اگر قفل بود -> history)
if (!try_lock($CTX, $LOCK_FILE, $LOCK_TTL)) {
    log_event($CTX, "WARN", "lock_busy", ["lock"=>$LOCK_FILE]);
    serve_from_history_or_204($CTX, "lock_busy");
}
register_shutdown_function(function() use ($LOCK_FILE) { unlock($LOCK_FILE); });

log_event($CTX, "INFO", "run_start", [
    "source" => $SOURCE_URL,
    "cache_ttl" => $CACHE_TTL,
]);

/* ---- Read cache or fetch ---- */
$cacheKey = md5($SOURCE_URL);
$cacheFile = rtrim($CACHE_DIR,'/\\') . "/src_" . $cacheKey . ".txt";

$raw = null;
if (is_file($cacheFile) && (time() - filemtime($cacheFile)) < $CACHE_TTL) {
    $raw = @file_get_contents($cacheFile);
    log_event($CTX, "INFO", "cache_hit", [
        "file" => $cacheFile,
        "bytes" => is_string($raw) ? strlen($raw) : 0,
    ]);
} else {
    // Fetch with 2 tries
    for ($attempt=1; $attempt<=2; $attempt++) {
        log_event($CTX, "DEBUG", "fetch_try", [
            "url" => $SOURCE_URL,
            "attempt" => $attempt,
            "timeout" => $HTTP_TIMEOUT,
        ]);

        $r = http_get($SOURCE_URL, $HTTP_TIMEOUT, $CTX);
        if ($r['ok'] && $r['code'] >= 200 && $r['code'] < 500) {
            $raw = $r['body'];
            @file_put_contents($cacheFile, $raw, LOCK_EX);
            log_event($CTX, "INFO", "fetch_ok", [
                "url" => $SOURCE_URL,
                "http_code" => $r['code'],
                "bytes" => $r['bytes'],
            ]);
            break;
        } else {
            log_event($CTX, "WARN", "fetch_failed", [
                "url" => $SOURCE_URL,
                "http_code" => $r['code'],
                "err" => $r['err'],
            ]);
        }
    }
}

if (!is_string($raw) || trim($raw) === '' || looks_like_html_or_error($raw)) {
    log_event($CTX, "ERROR", "source_bad", [
        "reason" => "empty_or_html_or_error",
    ]);
    serve_from_history_or_204($CTX, "source_bad");
}

/* ---- Extract ---- */
$linkItems = extract_link_lines($raw);
$jsonItems = extract_json_configs($raw);

log_event($CTX, "INFO", "extract_done", [
    "links" => count($linkItems),
    "jsons" => count($jsonItems),
]);

$items = array_values(array_unique(array_merge($linkItems, $jsonItems)));

// فیلتر نهایی
$valid = [];
foreach ($items as $it) {
    if (is_valid_output($it)) $valid[] = $it;
}

log_event($CTX, "INFO", "validate_done", [
    "total" => count($items),
    "valid" => count($valid),
]);

if (empty($valid)) {
    log_event($CTX, "WARN", "no_valid_items", []);
    serve_from_history_or_204($CTX, "no_valid_items");
}

/* ---- Pick one ---- */
$out = $valid[array_rand($valid)];
$out = trim($out);

/* ---- Save history ---- */
if ($SAVE_HISTORY) {
    save_history_line($HISTORY_DIR, $out);
    log_event($CTX, "INFO", "history_saved", [
        "bytes" => strlen($out),
    ]);
}

log_event($CTX, "INFO", "request_end", [
    "printed" => (preg_match('~^(vless|vmess|trojan|ss)://~i', $out) ? "link" : (($out[0]??'')==='{' ? "json" : "other")),
    "bytes" => strlen($out),
]);

echo $out;
exit;