<?php
// convert_vless.php
// گرفتن محتوا از URL، استخراج کانفیگ‌های vless و تولید یک config JSON شبیه نمونه
// ویژگی‌ها:
// - کش محلی با مدت پیش‌فرض 63 دقیقه
// - پشتیبانی از چندین vless در خروجی (هر کدام یک outbound)
// - پارس پارامترهای مهم مثل encryption, security, type, sni, pbk, sid, fp, spx, headerType
// - اضافه شدن "allowInsecure": true در realitySettings
// - خروجی JSON pretty و با یونیکد/اسلش‌های بدون escape

// ====== تنظیمات ======
$sourceUrl = 'https://fitn1.ir/Api/Ash/trun.php';
$cacheFile = __DIR__ . '/vless_cache.txt';
$cacheTimeSeconds = 1; // 63 دقیقه کش (اگر نمی‌خواهی cache کنه، مقدار را 0 بگذار)
// =======================

// تابع دریافت محتوا با cache ساده
function fetch_with_cache($url, $cacheFile, $cacheTimeSeconds) {
    if ($cacheTimeSeconds > 0 && file_exists($cacheFile) && (time() - filemtime($cacheFile)) < $cacheTimeSeconds) {
        return @file_get_contents($cacheFile);
    }
    $opts = [
        "http" => [
            "method" => "GET",
            "timeout" => 10,
            "header" => "User-Agent: curl/7.68.0\r\n"
        ]
    ];
    $context = stream_context_create($opts);
    $content = @file_get_contents($url, false, $context);
    if ($content !== false && $cacheTimeSeconds > 0) {
        @file_put_contents($cacheFile, $content);
    }
    return $content;
}

// تابع parse برای یک خط vless و ساخت آرایه outbound مطابق نمونه
function parse_vless_line($line) {
    $line = trim($line);
    if ($line === '') return null;
    // اگر خط شامل چند مورد یا فاصله باشه، ابتدا فقط بخش vless://... تا پایان خط را میگیریم
    $pos = stripos($line, 'vless://');
    if ($pos === false) return null;
    $candidate = substr($line, $pos);

    // جدا کردن fragment (بعد از #) اگر وجود داشته باشه
    $fragment = null;
    if (strpos($candidate, '#') !== false) {
        list($candidate, $fragment) = explode('#', $candidate, 2);
        $fragment = trim($fragment);
    }

    // parse_url احتیاج به scheme درست دارد
    // اگر بعد از vless:// کاراکتر "/" داشته باشیم ممکنه parse_url درست نشه، پس از ساختار استفاده می‌کنیم
    $tmp = $candidate;
    // اگر لینک با vless:// شروع میشه که حتما هست
    // استفاده از parse_url
    $parts = parse_url($tmp);
    if ($parts === false) return null;

    // user id احتمالا در بخش user قرار دارد (قبل از @)
    $userId = isset($parts['user']) ? $parts['user'] : null;
    $host = isset($parts['host']) ? $parts['host'] : '';
    $port = isset($parts['port']) ? intval($parts['port']) : 443;

    // query string
    $queryStr = isset($parts['query']) ? $parts['query'] : '';
    parse_str($queryStr, $params);

    // تعیین spx/path
    $spx = '/';
    if (!empty($params['spx'])) {
        $spx = urldecode($params['spx']);
    } elseif (!empty($params['path'])) {
        $spx = $params['path'];
    } elseif (!empty($params['headerType']) && $params['headerType'] === 'http') {
        // fallback
        $spx = '/';
    }

    $network = isset($params['type']) ? $params['type'] : 'tcp';
    $security = isset($params['security']) ? $params['security'] : '';
    $encryption = isset($params['encryption']) ? $params['encryption'] : 'none';
    $headerType = isset($params['headerType']) ? $params['headerType'] : 'none';

    $serverName = isset($params['sni']) ? $params['sni'] : (isset($params['servername']) ? $params['servername'] : $host);
    $publicKey = isset($params['pbk']) ? $params['pbk'] : (isset($params['publickey']) ? $params['publickey'] : null);
    $shortId = isset($params['sid']) ? $params['sid'] : (isset($params['shortid']) ? $params['shortid'] : null);
    $fingerprint = isset($params['fp']) ? $params['fp'] : (isset($params['fingerprint']) ? $params['fingerprint'] : null);

    // تعیین flow برای reality (مطابق نمونه کاربر)
    $flow = null;
    if (strtolower($security) === 'reality') {
        $flow = 'xtls-rprx-vision';
    }

    // ساختار outbound
    $outbound = [
        "tag" => "proxy",
        "protocol" => "vless",
        "settings" => [
            "vnext" => [
                [
                    "address" => $host,
                    "port" => $port,
                    "users" => [
                        [
                            "id" => $userId,
                            "encryption" => $encryption
                        ]
                    ]
                ]
            ]
        ],
        "streamSettings" => [
            "network" => $network
        ]
    ];

    if ($flow !== null) {
        $outbound["settings"]["vnext"][0]["users"][0]["flow"] = $flow;
    }

    if (!empty($security)) {
        $outbound["streamSettings"]["security"] = $security;
        if (strtolower($security) === 'reality') {
            $outbound["streamSettings"]["realitySettings"] = [
                "serverName" => $serverName,
                "publicKey" => $publicKey,
                "shortId" => $shortId,
                "fingerprint" => $fingerprint,
                "show" => false,
                "spiderX" => $spx,
                // اضافه شدن allowInsecure طبق درخواست
                "allowInsecure" => true
            ];
        } else {
            // برای TLS یا سایر موارد می‌توان tlsSettings اضافه کرد (آینده‌نگری)
            if (strtolower($security) === 'tls') {
                // نمونهٔ پایه tlsSettings شامل allowInsecure هم اگر خواسته شد
                $outbound["streamSettings"]["tlsSettings"] = [
                    "allowInsecure" => true
                ];
            }
        }
    }

    // تنظیمات TCP header در صورت نیاز
    if (strtolower($network) === 'tcp') {
        $outbound["streamSettings"]["tcpSettings"] = [
            "header" => [
                "type" => $headerType
            ]
        ];
    }

    // قرار دادن تگ بر اساس fragment (کشور یا نام دلخواه) برای خوانایی
    if (!empty($fragment)) {
        $cleanTag = preg_replace('/[^a-zA-Z0-9_-]/', '', $fragment);
        if ($cleanTag !== '') {
            $outbound["tag"] = "proxy-" . $cleanTag;
        }
    }

    return $outbound;
}


// ======= اجرای تبدیل =======
$content = fetch_with_cache($sourceUrl, $cacheFile, $cacheTimeSeconds);
if ($content === false || $content === null || trim($content) === '') {
    http_response_code(500);
    header('Content-Type: application/json; charset=utf-8');
    echo json_encode(["error" => "failed_to_fetch_source_or_empty"], JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE);
    exit;
}

// جدا کردن خطوط
$lines = preg_split('/\r\n|\r|\n/', $content);
$outbounds = [];
foreach ($lines as $line) {
    $line = trim($line);
    if ($line === '') continue;

    // اگر چند کانکشن در یک خط هست، با regex همه vless://... را پیدا کن
    if (preg_match_all('/vless:\/\/[^\s\'"]+/i', $line, $matches)) {
        foreach ($matches[0] as $match) {
            $parsed = parse_vless_line($match);
            if ($parsed) $outbounds[] = $parsed;
        }
    } else {
        // fallback: بررسی وجود vless در خط و parse مستقیم
        if (stripos($line, 'vless://') !== false) {
            $parsed = parse_vless_line($line);
            if ($parsed) $outbounds[] = $parsed;
        }
    }
}

// اگر هیچ outbound پیدا نشد، تلاش دوم: کل متن را جستجو کن
if (count($outbounds) === 0) {
    if (preg_match_all('/vless:\/\/[^\s\'"]+/i', $content, $matches)) {
        foreach ($matches[0] as $match) {
            $parsed = parse_vless_line($match);
            if ($parsed) $outbounds[] = $parsed;
        }
    }
}

// اگر هنوز هم نبود، خروجی نمونه با direct و block بده
if (count($outbounds) === 0) {
    $config = [
        "log" => ["loglevel" => "warning"],
        "inbounds" => [
            [
                "tag" => "socks",
                "protocol" => "socks",
                "listen" => "127.0.0.1",
                "port" => 10808,
                "settings" => ["udp" => true]
            ],
            [
                "tag" => "http",
                "protocol" => "http",
                "listen" => "127.0.0.1",
                "port" => 10809
            ]
        ],
        "outbounds" => [
            ["tag" => "direct", "protocol" => "freedom"],
            ["tag" => "block", "protocol" => "blackhole"]
        ],
        "routing" => [
            "domainStrategy" => "AsIs",
            "rules" => [
                [
                    "type" => "field",
                    "ip" => ["geoip:private"],
                    "outboundTag" => "direct"
                ]
            ]
        ],
        "dns" => [
            "queryStrategy" => "UseIPv4",
            "servers" => ["local", "8.8.8.8"]
        ]
    ];
    header('Content-Type: application/json; charset=utf-8');
    echo json_encode($config, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE);
    exit;
}

// ساخت config نهایی: inbounds ثابت + outbounds (parsed) + direct & block
$config = [
    "log" => ["loglevel" => "warning"],
    "inbounds" => [
        [
            "tag" => "socks",
            "protocol" => "socks",
            "listen" => "127.0.0.1",
            "port" => 10808,
            "settings" => ["udp" => true]
        ],
        [
            "tag" => "http",
            "protocol" => "http",
            "listen" => "127.0.0.1",
            "port" => 10809
        ]
    ],
    "outbounds" => array_merge(
        $outbounds,
        [
            ["tag" => "direct", "protocol" => "freedom"],
            ["tag" => "block", "protocol" => "blackhole"]
        ]
    ),
    "routing" => [
        "domainStrategy" => "AsIs",
        "rules" => [
            [
                "type" => "field",
                "ip" => ["geoip:private"],
                "outboundTag" => "direct"
            ]
        ]
    ],
    "dns" => [
        "queryStrategy" => "UseIPv4",
        "servers" => ["local", "8.8.8.8"]
    ]
];

// چاپ JSON نهایی
header('Content-Type: application/json; charset=utf-8');
echo json_encode($config, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE);