<?php
// filter_canada.php
// UTF-8
declare(strict_types=1);

/* ========= تنظیمات ========= */
const BASE_DIR        = __DIR__;
const DATA_DIR        = BASE_DIR . '/data';
const CACHE_DIR       = DATA_DIR . '/cache';
const CACHE_DNS_DIR   = CACHE_DIR . '/dns';
const CACHE_GEO_DIR   = CACHE_DIR . '/geo';
const OUT_DIR         = BASE_DIR . '/out';
const OUT_FILE        = OUT_DIR . '/canada.txt';
const DNS_TTL_SEC     = 600;  // 10 دقیقه
const GEO_TTL_SEC     = 600;  // 10 دقیقه
const USER_AGENT      = 'Config-CA-Filter/1.0';
const GEO_API_URL     = 'http://45.150.32.240/datacenter/out/aws.txt'; // countryCode, isp, as, org ...

/* ========= کمکی‌های فایل/کش ========= */
function ensure_dir(string $d): void { if (!is_dir($d)) { @mkdir($d, 0775, true); } }
function now(): int { return time(); }
function is_ip(string $h): bool { return (bool)filter_var($h, FILTER_VALIDATE_IP, FILTER_FLAG_IPV4 | FILTER_FLAG_IPV6); }

function cache_get(string $path, int $ttl): ?array {
  if (!is_file($path)) return null;
  $raw = @file_get_contents($path);
  if ($raw === false) return null;
  $j = json_decode($raw, true);
  if (!is_array($j) || !isset($j['_ts'])) return null;
  if (now() - (int)$j['_ts'] > $ttl) return null;
  return $j;
}
function cache_put(string $path, array $data): void {
  $data['_ts'] = now();
  ensure_dir(dirname($path));
  @file_put_contents($path, json_encode($data, JSON_UNESCAPED_UNICODE|JSON_UNESCAPED_SLASHES));
}

/* ========= دانلود ورودی ========= */
function http_get(string $url): string {
  $ch = curl_init($url);
  curl_setopt_array($ch, [
    CURLOPT_RETURNTRANSFER => true,
    CURLOPT_FOLLOWLOCATION => true,
    CURLOPT_TIMEOUT => 15,
    CURLOPT_CONNECTTIMEOUT => 8,
    CURLOPT_USERAGENT => USER_AGENT,
  ]);
  $res = curl_exec($ch);
  $err = curl_error($ch);
  $code = (int)curl_getinfo($ch, CURLINFO_RESPONSE_CODE);
  curl_close($ch);
  if ($res === false || $code >= 400) {
    throw new RuntimeException("HTTP GET failed ($code): $err");
  }
  return $res;
}

/* ========= رزولوشن DNS با کش ========= */
function resolve_ipv4_cached(string $host): ?string {
  // اگر خود Host آی‌پی باشد، همان
  if (is_ip($host)) return $host;

  $key = preg_replace('/[^A-Za-z0-9_.-]/', '_', $host);
  $cacheFile = CACHE_DNS_DIR . "/{$key}.json";
  $hit = cache_get($cacheFile, DNS_TTL_SEC);
  if ($hit && isset($hit['ip']) && is_string($hit['ip'])) {
    return $hit['ip'] ?: null;
  }

  // سعی کن A record بگیری
  $ip = null;
  $recs = @dns_get_record($host, DNS_A);
  if (is_array($recs) && count($recs)) {
    foreach ($recs as $r) {
      if (!empty($r['ip']) && is_ip($r['ip'])) { $ip = $r['ip']; break; }
    }
  }
  // fallback ساده
  if (!$ip) {
    $g = @gethostbyname($host);
    if ($g && $g !== $host && is_ip($g)) $ip = $g;
  }

  cache_put($cacheFile, ['ip' => $ip ?? '']);
  return $ip;
}

/* ========= GeoIP با کش (ip-api) ========= */
function geo_country_cached(string $ip): ?string {
  $key = str_replace([':', '.'], '_', $ip);
  $cacheFile = CACHE_GEO_DIR . "/{$key}.json";
  $hit = cache_get($cacheFile, GEO_TTL_SEC);
  if ($hit && isset($hit['countryCode'])) {
    return is_string($hit['countryCode']) ? $hit['countryCode'] : null;
  }

  $url = GEO_API_URL . rawurlencode($ip) . '?fields=status,country,countryCode,as,isp,org,query';
  $raw = http_get($url);
  $j = json_decode($raw, true);
  if (!is_array($j) || ($j['status'] ?? '') !== 'success') {
    // کش ناموفق بودن برای جلوگیری از تکرار
    cache_put($cacheFile, ['countryCode' => null]);
    return null;
  }
  cache_put($cacheFile, $j);
  return $j['countryCode'] ?? null;
}

/* ========= استخراج Host از انواع کانفیگ ========= */
function parse_host_from_vlink(string $line): ?string {
  // vless/trojan/… => scheme://userinfo@host:port?...
  $schemePos = strpos($line, '://');
  if ($schemePos === false) return null;
  $scheme = strtolower(substr($line, 0, $schemePos));
  if (!in_array($scheme, ['vless','vmess','trojan','ss'], true)) return null;

  if ($scheme === 'vmess') {
    // vmess://<base64>
    $b64 = substr($line, strlen('vmess://'));
    $json = base64_decode(strtr($b64, '-_', '+/'), true);
    if ($json === false) return null;
    $v = json_decode($json, true);
    if (!is_array($v)) return null;
    // اول add (آدرس اصلی)
    $cand = $v['add'] ?? null;
    // اگر add آی‌پی نبود و host/sni بود، از آن هم می‌توان رزولوشن گرفت
    if (!$cand) $cand = $v['host'] ?? null;
    if (!$cand) $cand = $v['sni'] ?? null;
    return is_string($cand) ? trim($cand) : null;
  }

  if ($scheme === 'ss') {
    // حالت‌های مختلف ss:
    // ss://base64(user:pass)@host:port#tag
    // ss://method:password@host:port#tag
    $u = parse_url($line);
    if (is_array($u) && !empty($u['host'])) {
      return $u['host'];
    }
    // تلاش برای دی‌کد base64 قبل از @
    $payload = substr($line, strlen('ss://'));
    $at = strpos($payload, '@');
    if ($at !== false) {
      $after = substr($payload, $at + 1);
      $host = $after;
      // اگر #tag یا ?params دارد، جدا کن
      $host = preg_split('/[?#]/', $host, 2)[0];
      // host:port
      $parts = explode('@', $payload);
      $tail = end($parts);
      $tail = preg_split('/[?#]/', $tail, 2)[0];
      $hp = explode('@', $tail);
      $hostPort = end($hp);
      if (strpos($hostPort, '@') !== false) {
        $hostPort = substr($hostPort, strrpos($hostPort, '@')+1);
      }
      $hostOnly = explode(':', $hostPort)[0] ?? '';
      return $hostOnly ?: null;
    }
    return null;
  }

  // vless / trojan
  $u = parse_url($line);
  if (is_array($u) && !empty($u['host'])) {
    return $u['host'];
  }
  return null;
}

function parse_host_from_json_line(string $line): ?string {
  // اگر یک بلوک JSON کامل باشد، سعی کن address را پیدا کنی
  $line = trim($line);
  if ($line === '' || ($line[0] !== '{' && $line[0] !== '[')) return null;

  $j = json_decode($line, true);
  if (!is_array($j)) {
    // fallback: regex ساده برای "address":"..."
    if (preg_match('/"address"\s*:\s*"([^"]+)"/', $line, $m)) {
      return $m[1];
    }
    return null;
  }

  // جست‌وجوی عمیق برای کلید address
  $stack = [$j];
  while ($stack) {
    $cur = array_pop($stack);
    if (is_array($cur)) {
      foreach ($cur as $k=>$v) {
        if (is_array($v)) $stack[] = $v;
        if (is_string($k) && strtolower($k) === 'address' && is_string($v)) {
          return $v;
        }
      }
    }
  }
  return null;
}

function extract_candidate_host(string $line): ?string {
  $line = trim($line);
  if ($line === '' || strpos($line, '#') === 0) return null;
  // ابتدا لینک‌های متداول
  $h = parse_host_from_vlink($line);
  if ($h) return $h;
  // سپس JSON
  $h = parse_host_from_json_line($line);
  return $h;
}

/* ========= خواندن ورودی و فیلتر کانادا ========= */
function load_existing_out(): array {
  if (!is_file(OUT_FILE)) return [];
  $lines = @file(OUT_FILE, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES);
  return $lines ? array_fill_keys($lines, true) : [];
}

function save_append_unique(array $newLines): void {
  ensure_dir(OUT_DIR);
  if (!$newLines) return;
  $fh = @fopen(OUT_FILE, 'ab');
  if (!$fh) return;
  foreach ($newLines as $ln) {
    @fwrite($fh, $ln . "\n");
  }
  @fclose($fh);
}

/* ========= ورودی src از GET/CLI ========= */
function get_src_arg(): ?string {
  // وب
  if (PHP_SAPI !== 'cli') {
    $src = $_GET['src'] ?? null;
    return $src ? trim((string)$src) : null;
  }
  // CLI
  global $argv;
  foreach ($argv as $a) {
    if (str_starts_with($a, '--src=')) {
      return trim(substr($a, 6));
    }
  }
  return null;
}

/* ========= Main ========= */
header('Content-Type: text/plain; charset=utf-8');

ensure_dir(DATA_DIR);
ensure_dir(CACHE_DIR);
ensure_dir(CACHE_DNS_DIR);
ensure_dir(CACHE_GEO_DIR);
ensure_dir(OUT_DIR);

$src = get_src_arg();
if (!$src) {
  echo "Usage (web):   ?src=http://host/path.txt\n";
  echo "Usage (cli):   php filter_canada.php --src=\"http://host/path.txt\"\n";
  exit(0);
}

try {
  $content = http_get($src);
} catch (Throwable $e) {
  http_response_code(500);
  echo "Download error: " . $e->getMessage() . "\n";
  exit(1);
}

$lines = preg_split('/\R/u', $content);
$already = load_existing_out();
$outNew = [];

foreach ($lines as $line) {
  $line = trim($line);
  if ($line === '' || strpos($line, '#!skip') === 0) continue;

  $host = extract_candidate_host($line);
  if (!$host) continue;

  // اگر Host دامنه بود، آی‌پی را Resolve کن؛ اگر خودش آی‌پی است همان
  $ip = resolve_ipv4_cached($host);
  if (!$ip) continue;

  $cc = geo_country_cached($ip);
  if ($cc === 'CA') {
    // چاپ و ذخیره بدون تکرار
    if (!isset($already[$line])) {
      $outNew[] = $line;
      $already[$line] = true;
    }
    echo $line . "\n";
  }
}

save_append_unique($outNew);
if (!$outNew) {
  // پیام پایانی مختصر
  // echo "\nNo new CA lines. (Duplicates filtered or none found)\n";
}