<?php
header('Content-Type: application/json; charset=utf-8');

define('CACHE_FILE', __DIR__ . '/cache.json'); // کش لیست سرورها
define('INDEX_FILE', __DIR__ . '/index.txt');  // ایندکس چرخش
define('CACHE_TTL', 3600);                     // کش ساعتی (ثانیه)

/* ---------- helpers ---------- */
function array_get($arr, $path, $default=null) {
    $cur = $arr;
    foreach (explode('.', $path) as $key) {
        if (is_array($cur) && array_key_exists($key, $cur)) $cur = $cur[$key];
        else return $default;
    }
    return $cur;
}

/* ---------- fetch & cache ---------- */
function fetch_data(): array {
    // 1) backendUrls
    $ch = curl_init();
    curl_setopt_array($ch, [
        CURLOPT_URL => 'https://urls.appsongplay.com/eu13urls.txt',
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_TIMEOUT => 12,
    ]);
    $raw = curl_exec($ch);
    curl_close($ch);

    $url = '';
    if ($raw !== false) {
        $parsed = json_decode($raw, true);
        $url = $parsed['backendUrls'][0] ?? '';
    }
    if (!$url) return [];

    // 2) servers
    $ch = curl_init();
    curl_setopt_array($ch, [
        CURLOPT_URL => $url . '?&package=com.vpn.euvpn',
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_TIMEOUT => 15,
    ]);
    $raw2 = curl_exec($ch);
    curl_close($ch);

    if ($raw2 === false) return [];
    $res = json_decode($raw2, true);
    $servers = $res['servers'] ?? [];

    $out = [];
    if (is_array($servers)) {
        foreach ($servers as $s) {
            if (is_array($s) && isset($s['config'])) $out[] = $s;
        }
    }
    return $out;
}

function load_servers_with_cache(): array {
    $use_cache = false;
    if (file_exists(CACHE_FILE)) {
        $age = time() - filemtime(CACHE_FILE);
        if ($age < CACHE_TTL) $use_cache = true;
    }

    if ($use_cache) {
        $cached = json_decode(@file_get_contents(CACHE_FILE), true);
        return is_array($cached) ? $cached : [];
    }

    $servers = fetch_data();
    if (!empty($servers)) {
        @file_put_contents(CACHE_FILE, json_encode($servers, JSON_UNESCAPED_SLASHES));
        return $servers;
    }

    if (file_exists(CACHE_FILE)) {
        $cached = json_decode(@file_get_contents(CACHE_FILE), true);
        if (is_array($cached)) return $cached;
    }
    return [];
}

/* ---------- rotation ---------- */
function next_index(int $count): int {
    $idx = -1;
    if (file_exists(INDEX_FILE)) {
        $raw = trim(@file_get_contents(INDEX_FILE));
        if ($raw !== '' && ctype_digit($raw)) $idx = (int)$raw;
    }
    if ($idx < 0 || $idx >= $count) $idx = -1;
    $idx = ($idx + 1) % $count;
    @file_put_contents(INDEX_FILE, (string)$idx);
    return $idx;
}

/* ---------- main ---------- */
$servers = load_servers_with_cache();
if (empty($servers)) {
    echo json_encode(["error" => "no servers available"]);
    exit;
}

$index = next_index(count($servers));
$server = $servers[$index];

// decode inner config
$innerRaw = $server['config'] ?? null;
$inner = null;
if (is_string($innerRaw)) $inner = json_decode($innerRaw, true);
elseif (is_array($innerRaw)) $inner = $innerRaw;

if (!is_array($inner)) {
    echo json_encode([
        "error" => "failed to decode inner config",
        "index" => $index,
        "json_error" => json_last_error_msg()
    ]);
    exit;
}

/* --- extract VLESS essentials --- */
$outbounds = $inner['outbounds'] ?? [];
$proxy = null;
foreach ($outbounds as $ob) {
    if (($ob['protocol'] ?? '') === 'vless') { $proxy = $ob; break; }
}
if (!$proxy) {
    echo json_encode(["error" => "no VLESS outbound found", "index" => $index]);
    exit;
}

$vnext0  = array_get($proxy, 'settings.vnext.0', []);
$users0  = array_get($vnext0, 'users.0', []);
$address = $vnext0['address'] ?? null;
$port    = $vnext0['port']    ?? 443;
$uuid    = $users0['id']      ?? null;
$flow    = $users0['flow']    ?? null;

$stream      = $proxy['streamSettings'] ?? [];
$network     = $stream['network']  ?? 'tcp';
$security    = $stream['security'] ?? 'none';
$reality     = $stream['realitySettings'] ?? [];
$serverName  = $reality['serverName'] ?? null;
$publicKey   = $reality['publicKey']  ?? null;
$shortId     = $reality['shortId']    ?? null;
$fingerprint = $reality['fingerprint']?? 'chrome';
$spiderX     = $reality['spiderX']    ?? '/';

$wsSettings  = $stream['wsSettings'] ?? [];
$wsHost      = array_get($wsSettings, 'headers.Host', null);
$wsPath      = $wsSettings['path'] ?? '/';

if (!$address || !$uuid) {
    echo json_encode(["error" => "missing address/uuid", "index" => $index]);
    exit;
}

/* --- build a clean v2rayNG-friendly config (IGNORE original noisy fields) --- */
$out = [
    "log" => [ "loglevel" => "warning" ],

    "inbounds" => [
        [
            "tag" => "socks",
            "protocol" => "socks",
            "listen" => "127.0.0.1",
            "port" => 10808,
            "settings" => [ "udp" => true ]
        ],
        [
            "tag" => "http",
            "protocol" => "http",
            "listen" => "127.0.0.1",
            "port" => 10809
        ]
    ],

    "outbounds" => [],

    "routing" => [
        "domainStrategy" => "AsIs",
        "rules" => [
            [ "type" => "field", "ip" => ["geoip:private"], "outboundTag" => "direct" ]
        ]
    ],

    "dns" => [
        "queryStrategy" => "UseIPv4",
        "servers" => ["local", "8.8.8.8"]
    ]
];

// main proxy
$proxyOutbound = [
    "tag" => "proxy",
    "protocol" => "vless",
    "settings" => [
        "vnext" => [[
            "address" => $address,
            "port" => (int)$port,
            "users" => [[
                "id" => $uuid,
                "encryption" => "none"
            ]]
        ]]
    ],
    "streamSettings" => [
        "network" => $network
    ]
];
if ($flow) {
    $proxyOutbound["settings"]["vnext"][0]["users"][0]["flow"] = $flow; // xtls-rprx-vision
}
if ($security === 'reality') {
    $proxyOutbound["streamSettings"]["security"] = "reality";
    $proxyOutbound["streamSettings"]["realitySettings"] = array_filter([
        "serverName"  => $serverName,
        "publicKey"   => $publicKey,
        "shortId"     => $shortId,
        "fingerprint" => $fingerprint ?: "chrome",
        "show"        => false,
        "spiderX"     => $spiderX ?: "/"
    ], fn($v) => $v !== null && $v !== '');
    $proxyOutbound["streamSettings"]["tcpSettings"] = [ "header" => [ "type" => "none" ] ];
} elseif ($security === 'tls') {
    $proxyOutbound["streamSettings"]["security"] = "tls";
    $tlsSni = array_get($stream, 'tlsSettings.serverName', null);
    if ($tlsSni) $proxyOutbound["streamSettings"]["tlsSettings"]["serverName"] = $tlsSni;
}
if ($network === 'ws') {
    $ws = [];
    if ($wsHost) $ws['headers']['Host'] = $wsHost;
    if ($wsPath) $ws['path'] = $wsPath;
    if (!empty($ws)) $proxyOutbound["streamSettings"]["wsSettings"] = $ws;
}

$out["outbounds"][] = $proxyOutbound;
$out["outbounds"][] = [ "tag" => "direct", "protocol" => "freedom" ];
$out["outbounds"][] = [ "tag" => "block",  "protocol" => "blackhole" ];

// final
echo json_encode($out, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE);