<?php
/**
 * exir_extractor.php — RAW dump + decrypt + deep extract ALL configs (exact raw, no trimming)
 * UTF-8, PHP 7.4+
 *
 * خروجی‌ها داخل ./exir_out :
 * - settings_raw.json , profiles_raw.json , profiles_headers.json
 * - profiles_decrypted.json , profiles_timestamp.txt
 * - all_configs.txt  (همهٔ کانفیگ‌ها به‌صورت خام و خط‌به‌خط، بدون تکرار)
 * - hashes.txt       (هش‌های ثبت‌شده برای دِدیوپ)
 */

declare(strict_types=1);
mb_internal_encoding('UTF-8');
header('Content-Type: text/plain; charset=utf-8');

// ===== تنظیمات =====
const OUT_DIR          = __DIR__ . '/exir_out';
const RAW_ONLY_FIRST   = false;   // اگر true شود فقط RAW ذخیره/چاپ می‌شود و ادامه نمی‌دهد
const MS_SCAN_WINDOW   = 900;     // ±seconds اسکن میلی‌ثانیه‌ای برای ts (±15 دقیقه)
const S_SCAN_WINDOW    = 1800;    // ±seconds اسکن ثانیه‌ای برای ts (±30 دقیقه)
const HTTP_TIMEOUT     = 30;      // ثانیه
const HTTP_CONNECT_TO  = 10;      // ثانیه

// ===== کمک‌های I/O =====
function ensure_dir(string $dir, int $mode = 0775): void {
  if ($dir === '') return;
  if (!is_dir($dir)) { @mkdir($dir, $mode, true); @chmod($dir, $mode); }
}
function write_file(string $path, string $content): void {
  ensure_dir(dirname($path));
  file_put_contents($path, $content);
}
function append_line_unique_raw(string $path, string $line, string $hashPath): bool {
  ensure_dir(dirname($path));
  ensure_dir(dirname($hashPath));
  $hash = hash('sha256', $line, false);

  static $seen = null;
  static $loaded = null;
  if ($seen === null || $loaded !== $hashPath) {
    $seen = [];
    if (file_exists($hashPath)) {
      $fh = fopen($hashPath, 'r');
      if ($fh) {
        while (($h = fgets($fh)) !== false) {
          $h = trim($h);
          if ($h !== '') $seen[$h] = true;
        }
        fclose($fh);
      }
    }
    $loaded = $hashPath;
  }
  if (isset($seen[$hash])) return false;

  $fa = fopen($path, 'ab');
  $fh = fopen($hashPath, 'ab');
  if (!$fa || !$fh) return false;
  fwrite($fa, $line . "\n");   // خام و دقیق؛ فقط یک \n انتها
  fwrite($fh, $hash . "\n");
  fclose($fa); fclose($fh);
  $seen[$hash] = true;
  return true;
}
function pretty_json(string $s): string {
  $j = json_decode($s, true);
  return $j !== null ? json_encode($j, JSON_PRETTY_PRINT|JSON_UNESCAPED_UNICODE|JSON_UNESCAPED_SLASHES) : $s;
}

// ===== ساخت هدر safety =====
function safety(): string {
  $uuid = sprintf('%04x%04x-%04x-%04x-%04x-%04x%04x%04x',
    mt_rand(0, 65535), mt_rand(0, 65535),
    mt_rand(0, 65535),
    mt_rand(16384, 20479),
    mt_rand(32768, 49151),
    mt_rand(0, 65535), mt_rand(0, 65535), mt_rand(0, 65535)
  );
  $data = $uuid . '|' . (round(microtime(true) * 1000) + 5000);
  $key = random_bytes(8);
  $finalkey = hex2bin('8B7A69F8EDB5FF73') . $key;
  $enc = openssl_encrypt($data, "aes-128-cbc", $finalkey, OPENSSL_RAW_DATA, hex2bin('77A7F6B7DEFDB8C369F5CB6F0908EC95'));
  return base64_encode($enc . $key);
}

// ===== HTTP =====
function http_post_json(string $url, string $json, array $extraHeaders = []): array {
  $respHeaders = [];
  $ch = curl_init();
  $hdrs = array_merge([
    'cache-control: no-cache',
    'apikey: 7bf4991622b9ed24d7f16c2320f35c0d',
    'packagename: com.exir.vpn',
    'versioncode: 15',
    'tz: Europe/Berlin',
    'locale: IR',
    'safety: ' . safety(),
    'user-agent: okhttp/4.11.0',
    'content-type: application/json',
  ], $extraHeaders);

  curl_setopt_array($ch, [
    CURLOPT_URL => $url,
    CURLOPT_RETURNTRANSFER => true,
    CURLOPT_POST => true,
    CURLOPT_POSTFIELDS => $json,
    CURLOPT_HTTPHEADER => $hdrs,
    CURLOPT_CONNECTTIMEOUT => HTTP_CONNECT_TO,
    CURLOPT_TIMEOUT => HTTP_TIMEOUT,
    CURLOPT_SSL_VERIFYPEER => true,
    CURLOPT_SSL_VERIFYHOST => 2,
    CURLOPT_IPRESOLVE => CURL_IPRESOLVE_V4,
    CURLOPT_HEADERFUNCTION => function($ch, $hdr) use (&$respHeaders) {
      $len = strlen($hdr);
      $p = strpos($hdr, ':');
      if ($p !== false) {
        $name = strtolower(trim(substr($hdr,0,$p)));
        $val  = trim(substr($hdr,$p+1));
        $respHeaders[$name] = $val;
      }
      return $len;
    },
  ]);
  $body = curl_exec($ch);
  $err  = curl_error($ch);
  $code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
  curl_close($ch);
  return [$code, $body, $respHeaders, $err];
}

// ===== رمزگشایی AES-256-CBC با کلید مشتق از ts =====
function decrypt_profiles_payload(string $dataB64, string $ivB64, string $ts) {
  $key = substr(hash_hmac('sha256', $ts, "secret-key", true), 0, 32);
  $iv  = base64_decode($ivB64, true);
  $ct  = base64_decode($dataB64, true);
  if ($iv === false || $ct === false) return false;
  return openssl_decrypt($ct, 'AES-256-CBC', $key, OPENSSL_RAW_DATA, $iv);
}

// ===== حدس و دی‌کریپت ts =====
function guess_and_decrypt(string $dataB64, string $ivB64, array $profilesJson, array $hdrs): array {
  $cands = [];
  $json_ts_raw = isset($profilesJson['timestamp']) && is_string($profilesJson['timestamp']) ? $profilesJson['timestamp'] : null;
  if ($json_ts_raw) {
    $cands[] = $json_ts_raw;
    $cands[] = str_replace('-', '', $json_ts_raw);
    $cands[] = $json_ts_raw . ' 00:00:00';
    $cands[] = $json_ts_raw . 'T00:00:00Z';
  }
  if (isset($hdrs['date'])) {
    $srv = strtotime($hdrs['date']);
    if ($srv !== false) {
      $cands[] = gmdate('Y-m-d', $srv);
      $cands[] = gmdate('Y-m-d\TH:i:s\Z', $srv);
      $cands[] = (string)$srv;
      $cands[] = (string)($srv * 1000);
    }
  }
  $now = time();
  $cands[] = (string)$now;
  $cands[] = (string)($now * 1000);

  foreach ($cands as $ts) {
    if (!is_string($ts) || $ts==='') continue;
    $plain = decrypt_profiles_payload($dataB64, $ivB64, $ts);
    if ($plain !== false && $plain !== '') {
      $j = json_decode($plain, true);
      if ($j !== null) return [$ts, $plain];
    }
  }

  // اسکن اطراف چند پایه
  $bases = [];
  if ($json_ts_raw && ctype_digit($json_ts_raw)) $bases[] = (int)$json_ts_raw;
  if (isset($hdrs['date'])) {
    $srv = strtotime($hdrs['date']);
    if ($srv !== false) { $bases[] = $srv; $bases[] = $srv * 1000; }
  }
  $bases[] = time();
  $bases[] = (int)round(microtime(true) * 1000);

  // میلی‌ثانیه‌ای
  $MS_STEP = 200;
  foreach ($bases as $b) {
    $is_ms = ($b > 2000000000);
    $base_ms = $is_ms ? $b : ($b * 1000);
    for ($off=-MS_SCAN_WINDOW*1000; $off<=MS_SCAN_WINDOW*1000; $off += $MS_STEP) {
      $ts = (string)($base_ms + $off);
      $plain = decrypt_profiles_payload($dataB64, $ivB64, $ts);
      if ($plain !== false && $plain !== '') {
        $j = json_decode($plain, true);
        if ($j !== null) return [$ts, $plain];
      }
    }
  }
  // ثانیه‌ای
  foreach ($bases as $b) {
    $is_ms = ($b > 2000000000);
    $base_s = $is_ms ? intdiv($b,1000) : $b;
    for ($off=-S_SCAN_WINDOW; $off<=S_SCAN_WINDOW; $off++) {
      $ts_s  = (string)($base_s + $off);
      $ts_ms = (string)(($base_s + $off) * 1000);
      foreach ([$ts_s,$ts_ms] as $tsTry) {
        $plain = decrypt_profiles_payload($dataB64, $ivB64, $tsTry);
        if ($plain !== false && $plain !== '') {
          $j = json_decode($plain, true);
          if ($j !== null) return [$tsTry, $plain];
        }
      }
    }
  }
  return [null, null];
}

// ===== تشخیص اولیهٔ «شبیه کانفیگ» =====
function looks_like_config_string(string $s): bool {
  $t = $s;
  if (preg_match('#^(vless|vmess|ss|trojan|trojan\+tls|trojan\+ws)://#i', $t)) return true;
  if (strlen($t) > 0 && ($t[0] === '{' || $t[0] === '[')) {
    $j = json_decode($t, true);
    if ($j !== null) return true;
  }
  if (preg_match('/"inbounds"|\"outbounds\"|\"vnext\"|\"address\"|\"port\"/i', $t)) return true;
  return false;
}

// ===== دیکود/استخراج عمیق از رشته‌ها =====
function try_decode_variants(string $s, array &$out, int $depth = 0): void {
  if ($depth > 6) return;
  if ($s === '') return;

  if (looks_like_config_string($s)) {
    $out[] = $s;
  }
  // vmess://payload
  if (preg_match_all('#\bvmess://([A-Za-z0-9+/=_-]+)#i', $s, $mm)) {
    foreach ($mm[1] as $b) {
      $dec = @base64_decode($b, true);
      if ($dec !== false && $dec !== '') {
        if ($json = @json_decode($dec, true)) {
          collect_configs_deep($json, $out);
        } else {
          $gz = @gzdecode($dec);
          if ($gz !== false) try_decode_variants($gz, $out, $depth+1);
          else try_decode_variants($dec, $out, $depth+1);
        }
      }
    }
  }
  // ss://
  if (preg_match_all('#\bss://([A-Za-z0-9+/=_-]+)#i', $s, $mm2)) {
    foreach ($mm2[1] as $b) {
      $dec = @base64_decode($b, true);
      if ($dec !== false && $dec !== '') {
        $out[] = 'ss://'.$b;   // فرم اصلی
        if (preg_match('#@#', $dec)) $out[] = $dec; // روش:پس@هاست:پورت
      }
    }
  }
  // base64 عمومی
  if (strlen($s) > 50) {
    $d = @base64_decode($s, true);
    if ($d !== false && strlen($d) >= 20) {
      try_decode_variants($d, $out, $depth+1);
    }
  }
  // gz
  $g = @gzdecode($s);
  if ($g !== false && strlen($g) > 20) {
    try_decode_variants($g, $out, $depth+1);
  }
  // urldecode
  $u = @urldecode($s);
  if ($u !== $s) {
    try_decode_variants($u, $out, $depth+1);
  }
  // JSON خام
  if (($s[0] ?? '') === '{' || ($s[0] ?? '') === '[') {
    $j = json_decode($s, true);
    if ($j !== null) {
      collect_configs_deep($j, $out);
    }
  }
  // شکست خطوط/طولانی
  if (strpos($s, "\n") !== false || strlen($s) > 1000) {
    foreach (preg_split('/\r\n|\r|\n/', $s) as $line) {
      if ($line === '') continue;
      try_decode_variants($line, $out, $depth+1);
    }
  }
}

// ===== جمع‌آوری عمومی (عمیق) =====
function collect_configs_deep($node, array &$found): void {
  if (is_string($node)) {
    // خود رشته
    if (looks_like_config_string($node)) $found[] = $node;
    // تلاش‌های دیکد/کشف
    try_decode_variants($node, $found, 0);
    // اگر JSON است، داخلش را هم بگرد
    if (strlen($node) > 0 && ($node[0] === '{' || $node[0] === '[')) {
      $j = json_decode($node, true);
      if ($j !== null) collect_configs_deep($j, $found);
    }
    return;
  }
  if (is_array($node)) {
    foreach ($node as $k => $v) {
      // فیلدهای حامل رایج
      if (is_string($k) && in_array(strtolower($k),
          ['config','raw','payload','profile','profiles','splashconfig','data','iv','value','url','link'], true)) {
        if (is_string($v)) {
          // همیشه مقدار خام این فیلدها را هم ثبت کن (برای JSON config)
          $found[] = $v;
          try_decode_variants($v, $found, 0);
        }
      }
      collect_configs_deep($v, $found);
    }
    return;
  }
  if (is_object($node)) {
    foreach ($node as $k => $v) {
      if (is_string($k) && in_array(strtolower((string)$k),
          ['config','raw','payload','profile','profiles','splashconfig','data','iv','value','url','link'], true)) {
        if (is_string($v)) {
          $found[] = $v;
          try_decode_variants($v, $found, 0);
        }
      }
      collect_configs_deep($v, $found);
    }
    return;
  }
  // سایر انواع نادیده
}

// ===== اجرای اصلی =====
ensure_dir(OUT_DIR);

// (1) SETTINGS RAW
list($c1, $b1, $h1, $e1) = http_post_json(
  "https://exirpro.org:8443/api/v1/settings",
  '{"appVersionCode":"15","appVersionName":"1.4","deviceId":"8478e6ec3e2fa562","local":"","tz":"Europe/Paris"}'
);
echo "==== RAW RESPONSE 1 (settings) ====\n";
if ($e1) echo "cURL ERROR: $e1\n";
echo "HTTP: $c1\n";
echo ($b1 === false ? "(no body)\n" : $b1 . "\n\n");
write_file(OUT_DIR . '/settings_raw.json', $b1 === false ? '' : $b1);

// (2) PROFILES RAW
list($c2, $b2, $h2, $e2) = http_post_json(
  "https://exirpro.org:8443/api/v1/profiles/fetch/all?retry=0",
  '{}'
);
echo "==== RAW RESPONSE 2 (profiles) ====\n";
if ($e2) echo "cURL ERROR: $e2\n";
echo "HTTP: $c2\n";
echo ($b2 === false ? "(no body)\n" : $b2 . "\n\n");
write_file(OUT_DIR . '/profiles_raw.json', $b2 === false ? '' : $b2);
write_file(OUT_DIR . '/profiles_headers.json', json_encode($h2, JSON_PRETTY_PRINT|JSON_UNESCAPED_UNICODE|JSON_UNESCAPED_SLASHES));

if (RAW_ONLY_FIRST) {
  echo "RAW-only mode. Saved raw files under " . OUT_DIR . "\n";
  exit(0);
}

// (3) DECRYPT
$profilesJson = json_decode($b2 ?? '', true);
if (!$profilesJson || !isset($profilesJson['data'], $profilesJson['iv'])) {
  echo "profiles: data/iv missing; abort.\n";
  exit(0);
}
$dataB64 = (string)$profilesJson['data'];
$ivB64   = (string)$profilesJson['iv'];

list($hit_ts, $plain) = guess_and_decrypt($dataB64, $ivB64, $profilesJson, $h2);
if ($hit_ts === null) {
  echo "FAILED: could not decrypt — try widening scan windows.\n";
  exit(0);
}
echo "✓ DECRYPT HIT with ts='{$hit_ts}'\n";
write_file(OUT_DIR . '/profiles_timestamp.txt', $hit_ts . "\n");
write_file(OUT_DIR . '/profiles_decrypted.json', pretty_json($plain));

// (4) EXTRACT ALL configs (settings.splashConfig + profiles[*].config + any links/json)
$all = [];

// from settings.splashConfig (raw + deep)
$settingsObj = json_decode($b1 ?? '', true);
if (is_array($settingsObj) && isset($settingsObj['splashConfig']) && is_string($settingsObj['splashConfig'])) {
  $sc = $settingsObj['splashConfig'];
  $all[] = $sc;                                // خام
  $scJson = json_decode($sc, true);
  if ($scJson !== null) collect_configs_deep($scJson, $all);
  else try_decode_variants($sc, $all, 0);
}

// from decrypted payload — اول مستقیم رشتهٔ JSON
$plainJson = json_decode($plain, true);
if ($plainJson !== null) {
  // به‌صورت خاص: اگر آرایه‌ای از آیتم‌ها با فیلد `config` داریم، همان رشتهٔ config هر آیتم را خام ذخیره کن
  if (is_array($plainJson)) {
    foreach ($plainJson as $it) {
      if (is_array($it) && isset($it['config']) && is_string($it['config'])) {
        $all[] = $it['config'];                // خام
        // علاوه بر خام، رویش استخراج عمیق هم انجام بده
        $cjson = json_decode($it['config'], true);
        if ($cjson !== null) collect_configs_deep($cjson, $all);
        else try_decode_variants($it['config'], $all, 0);
      }
    }
  }
  // سپس کل ساختار را هم پیمایش عمیق کن تا چیز دیگری جا نماند
  collect_configs_deep($plainJson, $all);
} else {
  if (is_string($plain)) {
    $all[] = $plain;
    try_decode_variants($plain, $all, 0);
  }
}

// (5) WRITE exact & unique lines
$outFile  = OUT_DIR . '/all_configs.txt';
$hashFile = OUT_DIR . '/hashes.txt';
$totalIn  = count($all);
$added    = 0;

foreach ($all as $raw) {
  if (!is_string($raw)) continue;
  // «بدون هیچ تغییری» — حتی trim هم نمی‌زنیم
  $ok = append_line_unique_raw($outFile, $raw, $hashFile);
  if ($ok) $added++;
}

echo "== SUMMARY ==\n";
echo "total collected candidates: {$totalIn}\n";
echo "new unique configs written: {$added}\n";
echo "output file: {$outFile}\n";