<?php
/**
 * Unified script for us.usaapp-2435.com:
 * - Fetch servers (with headers), cache for 1 hour
 * - Rotate between servers via index.txt
 * - Extract first VLESS from inner "config"
 * - Output a clean, v2rayNG-compatible single JSON profile (NOT a vless:// link)
 */

header('Content-Type: application/json; charset=utf-8');

define('SRC_URL',    'http://us.usaapp-2435.com:8081/usa-1/all?package=com.hdesign.usavpn');
define('CACHE_FILE', __DIR__ . '/cache.json');  // cached list of servers
define('INDEX_FILE', __DIR__ . '/index.txt');   // rotation index
define('CACHE_TTL',  3600);                     // 1 hour

/* ---------------- helpers ---------------- */

function array_get($arr, $path, $default=null) {
    $cur = $arr;
    foreach (explode('.', $path) as $key) {
        if (is_array($cur) && array_key_exists($key, $cur)) { $cur = $cur[$key]; }
        else { return $default; }
    }
    return $cur;
}

/* --------------- fetch & cache --------------- */

function fetch_servers(): array {
    $ch = curl_init(SRC_URL);
    curl_setopt_array($ch, [
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_SSL_VERIFYPEER => false,
        CURLOPT_CUSTOMREQUEST  => 'GET',
        CURLOPT_TIMEOUT        => 20,
        CURLOPT_HTTPHEADER     => [
            "User-Agent: Dalvik/2.1.0 (Linux; U; Android 14; SM-S928B Build/UP1A.231005.007)",
            "Host: us.usaapp-2435.com:8081",
            "Connection: keep-alive",
            "Accept-Encoding: gzip, deflate, br"
        ],
    ]);

    $response = curl_exec($ch);
    if ($response === false) {
        curl_close($ch);
        return [];
    }
    $status = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);

    if ($status !== 200) return [];

    $data = json_decode($response, true);
    if (!is_array($data)) return [];

    $servers = $data['servers'] ?? [];
    if (!is_array($servers)) return [];

    // keep only items that have 'config'
    $out = [];
    foreach ($servers as $s) {
        if (is_array($s) && array_key_exists('config', $s)) $out[] = $s;
    }
    return $out;
}

function load_servers_with_cache(): array {
    $use_cache = false;
    if (file_exists(CACHE_FILE)) {
        $age = time() - filemtime(CACHE_FILE);
        if ($age < CACHE_TTL) $use_cache = true;
    }

    if ($use_cache) {
        $cached = json_decode(@file_get_contents(CACHE_FILE), true);
        return is_array($cached) ? $cached : [];
    }

    // refresh cache
    $servers = fetch_servers();
    if (!empty($servers)) {
        @file_put_contents(CACHE_FILE, json_encode($servers, JSON_UNESCAPED_SLASHES));
        return $servers;
    }

    // fallback to old cache if fetch failed
    if (file_exists(CACHE_FILE)) {
        $cached = json_decode(@file_get_contents(CACHE_FILE), true);
        if (is_array($cached)) return $cached;
    }
    return [];
}

/* --------------- rotation --------------- */

function next_index(int $count): int {
    $idx = -1;
    if (file_exists(INDEX_FILE)) {
        $raw = trim(@file_get_contents(INDEX_FILE));
        if ($raw !== '' && ctype_digit($raw)) $idx = (int)$raw;
    }
    if ($idx < 0 || $idx >= $count) $idx = -1;
    $idx = ($idx + 1) % $count;
    @file_put_contents(INDEX_FILE, (string)$idx);
    return $idx;
}

/* --------------- main --------------- */

$servers = load_servers_with_cache();
if (empty($servers)) {
    echo json_encode(["error" => "no servers available"]);
    exit;
}

$index  = next_index(count($servers));
$server = $servers[$index];

// inner "config" may be a JSON string or already an array
$innerRaw = $server['config'] ?? null;
$cfg = null;
if (is_string($innerRaw)) {
    $cfg = json_decode($innerRaw, true);
} elseif (is_array($innerRaw)) {
    $cfg = $innerRaw;
}
if (!is_array($cfg)) {
    echo json_encode([
        "error" => "failed to decode inner config",
        "index" => $index,
        "json_error" => json_last_error_msg()
    ]);
    exit;
}

/* ---- extract essentials from first VLESS outbound ---- */

$outbounds = $cfg['outbounds'] ?? [];
$proxy = null;
foreach ($outbounds as $ob) {
    if (($ob['protocol'] ?? '') === 'vless') { $proxy = $ob; break; }
}
if (!$proxy) {
    echo json_encode(["error" => "no VLESS outbound found", "index" => $index]);
    exit;
}

$vnext0     = array_get($proxy, 'settings.vnext.0', []);
$users0     = array_get($vnext0, 'users.0', []);
$address    = $vnext0['address'] ?? null;
$port       = $vnext0['port']    ?? 443;
$uuid       = $users0['id']      ?? null;
$flow       = $users0['flow']    ?? null;

$stream     = $proxy['streamSettings'] ?? [];
$network    = $stream['network']  ?? 'tcp';
$security   = $stream['security'] ?? 'none';

$reality    = $stream['realitySettings'] ?? [];
$serverName = $reality['serverName'] ?? null;
$publicKey  = $reality['publicKey']  ?? null;
$shortId    = $reality['shortId']    ?? null;
$fingerprint= $reality['fingerprint']?? 'chrome';
$spiderX    = $reality['spiderX']    ?? '/';

$wsSettings = $stream['wsSettings'] ?? [];
$wsHost     = array_get($wsSettings, 'headers.Host', null);
$wsPath     = $wsSettings['path'] ?? '/';

// required minimal fields
if (!$address || !$uuid) {
    echo json_encode(["error" => "missing address/uuid", "index" => $index]);
    exit;
}

/* ---- build a clean v2rayNG-compatible single profile ---- */

$out = [
    "log" => [ "loglevel" => "warning" ],
    "inbounds" => [
        [
            "tag" => "socks",
            "protocol" => "socks",
            "listen" => "127.0.0.1",
            "port" => 10808,
            "settings" => [ "udp" => true ]
        ],
        [
            "tag" => "http",
            "protocol" => "http",
            "listen" => "127.0.0.1",
            "port" => 10809
        ]
    ],
    "outbounds" => [],
    "routing" => [
        "domainStrategy" => "AsIs",
        "rules" => [
            [ "type" => "field", "ip" => ["geoip:private"], "outboundTag" => "direct" ]
        ]
    ],
    "dns" => [
        "queryStrategy" => "UseIPv4",
        "servers" => ["local", "8.8.8.8"]
    ]
];

// main proxy outbound
$proxyOutbound = [
    "tag" => "proxy",
    "protocol" => "vless",
    "settings" => [
        "vnext" => [[
            "address" => $address,
            "port"    => (int)$port,
            "users"   => [[
                "id" => $uuid,
                "encryption" => "none"
            ]]
        ]]
    ],
    "streamSettings" => [
        "network" => $network
    ]
];
if ($flow) {
    $proxyOutbound["settings"]["vnext"][0]["users"][0]["flow"] = $flow; // xtls-rprx-vision
}

// security options
if ($security === 'reality') {
    $proxyOutbound["streamSettings"]["security"] = "reality";
    $proxyOutbound["streamSettings"]["realitySettings"] = array_filter([
        "serverName"  => $serverName,
        "publicKey"   => $publicKey,
        "shortId"     => $shortId,
        "fingerprint" => $fingerprint ?: "chrome",
        "show"        => false,
        "spiderX"     => $spiderX ?: "/"
    ], fn($v) => $v !== null && $v !== '');
    // common for reality over tcp
    $proxyOutbound["streamSettings"]["tcpSettings"] = [ "header" => [ "type" => "none" ] ];
} elseif ($security === 'tls') {
    $proxyOutbound["streamSettings"]["security"] = "tls";
    $tlsSni = array_get($stream, 'tlsSettings.serverName', null);
    if ($tlsSni) $proxyOutbound["streamSettings"]["tlsSettings"]["serverName"] = $tlsSni;
}

// ws support
if ($network === 'ws') {
    $ws = [];
    if ($wsHost) $ws['headers']['Host'] = $wsHost;
    if ($wsPath) $ws['path'] = $wsPath;
    if (!empty($ws)) $proxyOutbound["streamSettings"]["wsSettings"] = $ws;
}

// finalize outbounds
$out["outbounds"][] = $proxyOutbound;
$out["outbounds"][] = [ "tag" => "direct", "protocol" => "freedom" ];
$out["outbounds"][] = [ "tag" => "block",  "protocol" => "blackhole" ];

// emit
echo json_encode($out, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE);