<?php
// app/bot.php — Sequential per-link processing with "wait for duplicate" rule,
// JSON endpoint extraction, stable signature for JSON, per-line cache (3600s),
// DC classification via DNS+ASN, thresholded per-DC outputs, and safety limits.
// Requires config.php (DEFAULTS, PATHS, DC_MAP, etc.)
declare(strict_types=1);

/* ===================== FS / JSON / DIRS ===================== */
function ensure_dirs(array $paths): void {
  foreach (['DATA_DIR','OUT_DIR','LOG_DIR','CACHE_DIR','CACHE_DNS','CACHE_ASN','CACHE_CFG'] as $k) {
    $d = $paths[$k] ?? null;
    if ($d && !is_dir($d)) @mkdir($d, 0775, true);
  }
  if (!is_file($paths['LINKS']))    @file_put_contents($paths['LINKS'],    json_encode([], JSON_UNESCAPED_UNICODE));
  if (!is_file($paths['SETTINGS']))  @file_put_contents($paths['SETTINGS'],  json_encode([], JSON_UNESCAPED_UNICODE));
  if (!is_file($paths['SUMMARY']))   @file_put_contents($paths['SUMMARY'],   json_encode(['time'=>null,'inputs'=>0,'unique_endpoints'=>0,'groups'=>[]], JSON_UNESCAPED_UNICODE));
  if (!is_dir($paths['OUT_DIR']))    @mkdir($paths['OUT_DIR'], 0775, true);
}
function load_json(string $f,$fb){ if(!is_file($f))return $fb; $s=@file_get_contents($f); $j=$s!==false?json_decode($s,true):null; return is_array($j)?$j:$fb; }
function save_json(string $f,$d){ @file_put_contents($f,json_encode($d,JSON_UNESCAPED_UNICODE|JSON_UNESCAPED_SLASHES|JSON_PRETTY_PRINT)); }

/* ===================== FILE CACHE (JSON files) ===================== */
function cache_get_file(string $dir, string $key, int $ttl){
  $f=rtrim($dir,'/').'/'.sha1($key).'.json';
  if(!is_file($f)) return null;
  if($ttl>0 && time()-filemtime($f)>$ttl) return null;
  $s=@file_get_contents($f);
  return $s!==false?json_decode($s,true):null;
}
function cache_set_file(string $dir, string $key, $val):void{
  $f=rtrim($dir,'/').'/'.sha1($key).'.json';
  @file_put_contents($f,json_encode($val,JSON_UNESCAPED_UNICODE|JSON_UNESCAPED_SLASHES));
}

/* ===================== STATE (sequential per-link) ===================== */
function state_file(array $paths): string { return rtrim($paths['DATA_DIR'],'/').'/state.json'; }
function load_state(array $paths): array {
  $f = state_file($paths);
  if (!is_file($f)) return ['idx'=>0, 'stale'=>[], 'last_new'=>[], 'check_count'=>[], 'watch_line'=>[]];
  $j = json_decode((string)@file_get_contents($f), true);
  if (!is_array($j)) return ['idx'=>0, 'stale'=>[], 'last_new'=>[], 'check_count'=>[], 'watch_line'=>[]];
  $j += ['idx'=>0, 'stale'=>[], 'last_new'=>[], 'check_count'=>[], 'watch_line'=>[]];
  return $j;
}
function save_state(array $paths, array $st): void {
  @file_put_contents(state_file($paths), json_encode($st, JSON_UNESCAPED_UNICODE|JSON_PRETTY_PRINT));
}

/* ===================== INPUT (read ONE link) ===================== */
function fetch_lines_from_link(string $url, int $timeout): array {
  $ctx=stream_context_create(['http'=>['timeout'=>$timeout],'https'=>['timeout'=>$timeout]]);
  $s=@file_get_contents($url,false,$ctx);
  if($s===false) return [];
  // If JSON array of items:
  if(preg_match('/^\s*\[/', $s)){
    $j=json_decode($s,true);
    if(is_array($j)){
      $out=[];
      foreach($j as $it){
        if(is_string($it)) $out[]=trim($it);
        elseif(is_array($it) && isset($it['server'])) $out[]=trim((string)$it['server']);
        elseif(is_array($it)) $out[]=trim(json_encode($it,JSON_UNESCAPED_UNICODE|JSON_UNESCAPED_SLASHES));
      }
      return array_values(array_filter($out, fn($x)=>$x!==''));
    }
  }
  // text lines
  $out=[];
  foreach(preg_split('/\r?\n/',$s) as $ln){ $ln=trim($ln); if($ln!=='') $out[]=$ln; }
  return $out;
}

/* ===================== Helpers ===================== */
function utf8_line(string $s): string {
  if (!mb_check_encoding($s,'UTF-8')) return mb_convert_encoding($s,'UTF-8','auto');
  return $s;
}

/* ===================== Stable Signature (for JSON dedup) ===================== */
// Use endpoints (address:port|sni|wshost|tls) to build a stable signature for JSON.
// Links use the raw line.
function stable_signature_for_line(string $line): string {
  $t = trim($line);
  if ($t === '') return '';
  if ($t[0] === '{' || $t[0] === '[') { // JSON
    $eps = parse_json_endpoints($t);
    if (is_array($eps) && !empty($eps)) {
      $rows = [];
      foreach ($eps as $e) {
        $scheme = strtolower($e['scheme'] ?? 'json');
        $addr   = strtolower((string)($e['address'] ?? ''));
        $port   = (string)($e['port'] ?? '');
        $sni    = strtolower((string)($e['sni'] ?? ''));
        $wsh    = strtolower((string)($e['wshost'] ?? ''));
        $tls    = !empty($e['tls']) ? '1' : '0';
        $rows[] = implode('|', [$scheme, "$addr:$port", $sni, $wsh, $tls]);
      }
      sort($rows, SORT_STRING);
      return 'jsonsig:'.sha1(implode(';', $rows));
    }
    $min = json_encode(json_decode($t, true), JSON_UNESCAPED_UNICODE|JSON_UNESCAPED_SLASHES);
    return 'jsonmin:'.sha1($min ?: $t);
  }
  return 'line:'.$t; // for non-JSON links
}

/* ===================== Per-line Cache (3600s) ===================== */
function line_cache_get(string $line, array $cfg){
  $dir=$cfg['PATHS']['CACHE_CFG']; $ttl=(int)($cfg['DEFAULTS']['cache_ttl_config']??3600);
  $key = stable_signature_for_line($line);
  return cache_get_file($dir, $key, $ttl);
}
function line_cache_set(string $line, array $value, array $cfg): void {
  $dir=$cfg['PATHS']['CACHE_CFG']; $key = stable_signature_for_line($line);
  cache_set_file($dir, $key, $value+['_ts'=>time()]);
}
function is_config_cache_valid_line(string $storedLine, array $cfg): bool {
  $dir=$cfg['PATHS']['CACHE_CFG']; $ttl=(int)($cfg['DEFAULTS']['cache_ttl_config']??3600);
  $key = stable_signature_for_line($storedLine);
  $file=rtrim($dir,'/').'/'.sha1($key).'.json';
  if(!is_file($file)) return false;
  if($ttl>0 && time()-filemtime($file)>$ttl) return false;
  return true;
}

/* ===================== Link Parsers ===================== */
function parse_link_minimal(string $line): ?array {
  $line = trim($line);
  // vless/trojan/ss
  if (stripos($line,'vless://')===0 || stripos($line,'trojan://')===0 || stripos($line,'ss://')===0) {
    if (preg_match('~@([^:/?#]+):(\d+)~',$line,$m)) {
      $host=$m[1]; $port=(int)$m[2];
      $sni=null; $wsh=null;
      if (preg_match('/[?&](sni|serverName)=([^&]+)/i',$line,$m2)) $sni=urldecode($m2[2]);
      if (preg_match('/[?&](host|wsHost|wshost)=([^&]+)/i',$line,$m3)) $wsh=urldecode($m3[2]);
      $scheme = stripos($line,'vless://')===0?'vless':(stripos($line,'trojan://')===0?'trojan':'ss');
      $tls = (bool)preg_match('/[?&](security|tls)=(tls|reality)/i', $line);
      if(!$tls){ $tls = in_array($port,[443,8443,2053,2083,2087],true); }
      return ['scheme'=>$scheme,'address'=>$host,'port'=>$port,'sni'=>$sni,'wshost'=>$wsh,'tls'=>$tls];
    }
    // ss base64
    if (stripos($line,'ss://')===0) {
      $v=substr($line,5); $v=preg_replace('~[#?].*$~','',$v);
      $b=@base64_decode(strtr($v,'-_','+/'), true);
      if ($b && preg_match('~@([^:/?#]+):(\d+)~',$b,$m)) {
        $host=$m[1]; $port=(int)$m[2];
        $tls = in_array($port,[443,8443,2053,2083,2087],true);
        return ['scheme'=>'ss','address'=>$host,'port'=>$port,'tls'=>$tls];
      }
    }
  }
  // vmess://
  if (stripos($line,'vmess://')===0) {
    $b64=substr($line,8); $b64=strtr($b64,'-_','+/'); $pad=strlen($b64)%4; if($pad)$b64.=str_repeat('=',4-$pad);
    $json=@base64_decode($b64,true);
    if ($json!==false) {
      $j=json_decode($json,true);
      if (is_array($j) && !empty($j['add']) && !empty($j['port'])) {
        $sni = $j['tlsSettings']['serverName'] ?? null;
        $wsh = $j['wsSettings']['headers']['Host'] ?? ($j['wsSettings']['host'] ?? null);
        $tls = !empty($j['tls']) || !empty($j['security']) || !empty($j['tlsSettings']) || in_array((int)$j['port'],[443,8443,2053,2083,2087],true);
        return ['scheme'=>'vmess','address'=>$j['add'],'port'=>(int)$j['port'],'sni'=>$sni,'wshost'=>$wsh,'tls'=>$tls];
      }
    }
  }
  // host:port
  if (preg_match('~^([^:/?#]+):(\d+)$~',$line,$m)) {
    $host=$m[1]; $port=(int)$m[2];
    $tls = in_array($port,[443,8443,2053,2083,2087],true);
    return ['scheme'=>'hostport','address'=>$host,'port'=>$port,'tls'=>$tls];
  }
  return null;
}

/* ===================== JSON → endpoints (address/sni/wshost/tls) ===================== */
function parse_json_endpoints(string $line): array {
  $j = json_decode($line, true);
  if (!is_array($j)) return [];
  $out = [];

  // root simple
  $addr0 = $j['add'] ?? $j['address'] ?? $j['server'] ?? null;
  $port0 = isset($j['port']) ? (int)$j['port'] : null;
  $tls0  = !empty($j['tls']) || !empty($j['security']) || !empty($j['tlsSettings']) || ($port0 && in_array($port0,[443,8443,2053,2083,2087],true));
  if ($addr0 && $port0) $out[] = ['scheme'=>'json','address'=>$addr0,'port'=>$port0,'tls'=>$tls0];

  // outbounds[]
  $outbounds = $j['outbounds'] ?? [];
  if (is_array($outbounds)) {
    foreach ($outbounds as $ob) {
      if (!is_array($ob)) continue;
      $proto = strtolower($ob['protocol'] ?? '');
      if (!in_array($proto, ['vless','vmess','trojan','shadowsocks','ss'], true)) continue;

      $ss = $ob['streamSettings'] ?? [];
      $sni = $ss['tlsSettings']['serverName'] ?? ($ss['realitySettings']['serverName'] ?? null);

      $wsh = null;
      if (!empty($ss['wsSettings']['headers']['Host'])) $wsh = $ss['wsSettings']['headers']['Host'];
      elseif (!empty($ss['wsSettings']['host'])) $wsh = $ss['wsSettings']['host'];
      elseif (!empty($ss['httpSettings']['host'])) { $h=$ss['httpSettings']['host']; $wsh=is_array($h)?($h[0]??null):$h; }
      elseif (!empty($ss['h2Settings']['host']))   { $h=$ss['h2Settings']['host'];   $wsh=is_array($h)?($h[0]??null):$h; }

      $tls = !empty($ss['security']) || !empty($ss['tlsSettings']) || !empty($ss['realitySettings']);

      if (in_array($proto,['vless','vmess'],true)) {
        $vnext = $ob['settings']['vnext'] ?? [];
        foreach ($vnext as $vn) {
          if (!is_array($vn)) continue;
          $addr=$vn['address'] ?? null; $port=isset($vn['port'])?(int)$vn['port']:null;
          if ($addr && $port) $out[]=['scheme'=>$proto,'address'=>$addr,'port'=>$port,'sni'=>$sni,'wshost'=>$wsh,'tls'=>$tls];
        }
      } else { // trojan/ss
        $servers = $ob['settings']['servers'] ?? [];
        foreach ($servers as $sv) {
          if (!is_array($sv)) continue;
          $addr=$sv['address'] ?? null; $port=isset($sv['port'])?(int)$sv['port']:null;
          if ($addr && $port) $out[]=['scheme'=>$proto,'address'=>$addr,'port'=>$port,'sni'=>$sni,'wshost'=>$wsh,'tls'=>$tls];
        }
      }
    }
  }
  return $out;
}

/* ===================== DNS / ASN cached ===================== */
function resolve_ipv4_cached(string $host, array $cfg): ?string{
  $dir=$cfg['PATHS']['CACHE_DNS']; $ttl=(int)($cfg['DEFAULTS']['cache_ttl_dns']??600);
  $key="dns:$host"; $c=cache_get_file($dir,$key,$ttl); if($c && !empty($c['ip'])) return $c['ip'];
  $recs=@dns_get_record($host, DNS_A);
  if(!empty($recs)){ foreach($recs as $r){ if(!empty($r['ip'])){ cache_set_file($dir,$key,['ip'=>$r['ip']]); return $r['ip']; } } }
  return null;
}
function asn_lookup_cached(string $ip, array $cfg): ?array{
  $dir=$cfg['PATHS']['CACHE_ASN']; $ttl=(int)($cfg['DEFAULTS']['cache_ttl_asn']??3600);
  $key="asn:$ip"; $c=cache_get_file($dir,$key,$ttl); if($c) return $c;
  $url="http://ip-api.com/json/".rawurlencode($ip)."?fields=status,as,org,query";
  $ctx=stream_context_create(['http'=>['timeout'=>5]]); $s=@file_get_contents($url,false,$ctx);
  if($s===false) return null;
  $j=json_decode($s,true); if(!is_array($j)||($j['status']??'')!=='success') return null;
  $asn=null; if(!empty($j['as']) && preg_match('~AS(\d+)~i',$j['as'],$m)) $asn=(int)$m[1];
  $out=['asn'=>$asn,'org'=>$j['org']??null]; cache_set_file($dir,$key,$out); return $out;
}

/* ===================== DC mapping ===================== */
function map_datacenter(?string $org, ?int $asn, array $dcmap): string{
  $txt=trim(($org??'').' '.($asn?'AS'.$asn:'')); foreach($dcmap as $label=>$regex){ if(preg_match($regex,$txt)) return $label; }
  if($org && preg_match('/(mobile|mci|irancell|rightel|telecom|isp|broadband|communications|cellular|adsl|fiber|home|residential)/i',$org)) return 'residential/isp';
  return 'other';
}

/* ===================== DC decision from hints ===================== */
function decide_dc_from_candidates(array $candidates, array $cfg): array {
  $dcmap = $cfg['DC_MAP'] ?? [];
  $ips=[]; $ip_asn_map=[]; $votes=[];
  foreach ($candidates as $label => $val) {
    if (!$val) continue;
    if (filter_var($val, FILTER_VALIDATE_IP)) $ips[$val] = $label.'_ip';
    else { $ip = resolve_ipv4_cached($val, $cfg); if ($ip) $ips[$ip] = $label; }
  }
  foreach($ips as $ip=>$reason){
    $ai=asn_lookup_cached($ip,$cfg); $asn=$ai['asn']??null; $org=$ai['org']??null;
    $ip_asn_map[]=['ip'=>$ip,'asn'=>$asn,'org'=>$org,'reason'=>$reason];
    $dc=map_datacenter($org,$asn,$dcmap); $votes[$dc]=($votes[$dc]??0)+1;
  }
  $decided='other';
  if(!empty($votes)){ arsort($votes); $decided=array_key_first($votes); }
  return [$decided, $ip_asn_map];
}

/* ===================== Process ONE line (classify and cache) ===================== */
function process_line(string $line, array $cfg): array {
  $line = utf8_line(trim($line));
  if ($line==='') return ['stored'=>'', '__dc'=>'other', '__ip_asn_map'=>[]];

  // cache hit?
  $cached = line_cache_get($line, $cfg);
  if ($cached && !empty($cached['__dc'])) {
    // touch
    line_cache_set($line, ['__dc'=>$cached['__dc'],'__ip_asn_map'=>$cached['__ip_asn_map'] ?? []], $cfg);
    return ['stored'=>$line, '__dc'=>$cached['__dc'], '__ip_asn_map'=>$cached['__ip_asn_map'] ?? []];
  }

  // link?
  $lk = parse_link_minimal($line);
  if ($lk) {
    [$dc, $ipasn] = decide_dc_from_candidates([
      'address'=>$lk['address']??null,
      'sni'=>$lk['sni']??null,
      'wshost'=>$lk['wshost']??null,
    ], $cfg);
    $res=['stored'=>$line,'__dc'=>$dc,'__ip_asn_map'=>$ipasn];
    line_cache_set($line,$res,$cfg);
    return $res;
  }

  // JSON?
  if (($line[0]??'')==='{' || ($line[0]??'')==='[') {
    $eps = parse_json_endpoints($line);
    if ($eps) {
      $addr=[]; $sni=[]; $wsh=[];
      foreach ($eps as $e) {
        if (!empty($e['address'])) $addr[]=$e['address'];
        if (!empty($e['sni']))     $sni[] =$e['sni'];
        if (!empty($e['wshost']))  $wsh[] =$e['wshost'];
      }
      $addr=array_values(array_unique($addr));
      $sni =array_values(array_unique($sni));
      $wsh =array_values(array_unique($wsh));

      $votes=[]; $all=[];
      foreach ($addr as $a){ [$d,$m]=decide_dc_from_candidates(['address'=>$a],$cfg); $votes[$d]=($votes[$d]??0)+1; $all=array_merge($all,$m); }
      foreach ($sni  as $a){ [$d,$m]=decide_dc_from_candidates(['sni'=>$a],$cfg);     $votes[$d]=($votes[$d]??0)+1; $all=array_merge($all,$m); }
      foreach ($wsh  as $a){ [$d,$m]=decide_dc_from_candidates(['wshost'=>$a],$cfg);  $votes[$d]=($votes[$d]??0)+1; $all=array_merge($all,$m); }

      $dc='other'; if(!empty($votes)){ arsort($votes); $dc=array_key_first($votes); }
      $res=['stored'=>$line,'__dc'=>$dc,'__ip_asn_map'=>$all];
      line_cache_set($line,$res,$cfg);
      return $res;
    } else {
      // Fallback: raw text
      $txt=strtolower($line); $dc='other';
      foreach (($cfg['DC_MAP'] ?? []) as $label=>$rx) if(preg_match($rx,$txt)){ $dc=$label; break; }
      $res=['stored'=>$line,'__dc'=>$dc,'__ip_asn_map'=>[]];
      line_cache_set($line,$res,$cfg);
      return $res;
    }
  }

  // unknown → text regex
  $txt=strtolower($line); $dc='other';
  foreach (($cfg['DC_MAP'] ?? []) as $label=>$rx) if(preg_match($rx,$txt)){ $dc=$label; break; }
  $res=['stored'=>$line,'__dc'=>$dc,'__ip_asn_map'=>[]];
  line_cache_set($line,$res,$cfg);
  return $res;
}

/* ===================== Write per-DC (merge & TTL-clean) ===================== */
function write_dc_file_merged(string $dc, array $newStoredLines, array $cfg): void{
  $paths=$cfg['PATHS']; $fname=rtrim($paths['OUT_DIR'],'/').'/'.preg_replace('/[^a-z0-9_\-]+/i','_',$dc).'.txt';
  $existing=[];
  if(is_file($fname)){
    $content=@file_get_contents($fname);
    if($content!==false){
      foreach(preg_split('/\r?\n/',$content) as $ln){
        $ln=trim($ln); if($ln==='') continue;
        if(is_config_cache_valid_line($ln,$cfg)) $existing[]=$ln; // keep only valid (not expired) lines
      }
    }
  }
  $merged=array_merge($existing,$newStoredLines);
  $seen=[]; $uniq=[];
  foreach($merged as $ln){ if(!isset($seen[$ln])){ $seen[$ln]=true; $uniq[]=$ln; } }
  $tmp=$fname.'.tmp'; @file_put_contents($tmp, implode("\n",$uniq)); @rename($tmp,$fname);
}

/* ===================== One sequential run ===================== */
function run_once(array $cfg): array{
  $paths=$cfg['PATHS']; $defs=$cfg['DEFAULTS']; ensure_dirs($paths);

  $settings  = load_json($paths['SETTINGS'],$defs);
  $http_to   = (int)($settings['http_timeout']??$defs['http_timeout']);
  $min_per_dc= (int)($settings['min_per_dc']??($defs['min_per_dc']??1));
  if($min_per_dc<1)$min_per_dc=1;

  // safety params
  $max_checks = (int)($settings['max_checks_per_link'] ?? $defs['max_checks_per_link'] ?? 40);
  $max_cycles = (int)($settings['max_stale_cycles'] ?? $defs['max_stale_cycles'] ?? 6);
  $max_age    = (int)($settings['force_advance_after_sec'] ?? $defs['force_advance_after_sec'] ?? 300);

  $links=load_json($paths['LINKS'],[]); if(!is_array($links)) $links=[];
  $n = count($links);
  if ($n===0) {
    $summary=['time'=>date('c'),'inputs'=>0,'unique_endpoints'=>0,'groups'=>[],'idx'=>null];
    save_json($paths['SUMMARY'],$summary); save_json($paths['OUT_DIR'].'/summary.json',$summary);
    return $summary;
  }

  // state/load
  $st = load_state($paths);
  $idx = (int)($st['idx'] ?? 0);
  if ($idx<0 || $idx>=$n) $idx=0;
  $url = $links[$idx];

  // defaults for per-link state
  $st['stale']       = $st['stale']       ?? [];
  $st['last_new']    = $st['last_new']    ?? [];
  $st['check_count'] = $st['check_count'] ?? [];
  $st['watch_line']  = $st['watch_line']  ?? [];

  $st['stale'][$idx]       = (int)($st['stale'][$idx] ?? 0);
  $st['last_new'][$idx]    = (int)($st['last_new'][$idx] ?? 0);
  $st['check_count'][$idx] = (int)($st['check_count'][$idx] ?? 0);
  $st['watch_line'][$idx]  = $st['watch_line'][$idx] ?? null;

  // fetch only this link
  $lines = fetch_lines_from_link($url, $http_to);
  $lines = array_values(array_unique(array_map('trim', $lines)));

  // classify lines (new vs cached)
  $new_lines = []; $cached_seen = false; $total_valid = 0;
  foreach ($lines as $ln) {
    $ln_utf = utf8_line($ln);
    if ($ln_utf==='') continue;
    $cached = line_cache_get($ln_utf, $cfg);
    if ($cached) {
      $cached_seen = true;
      // touch to extend TTL
      line_cache_set($ln_utf, $cached, $cfg);
    } else {
      $new_lines[] = $ln_utf;
    }
    $total_valid++;
  }

  // process new lines
  $out_by_dc = []; $processed = 0; $got_new = false;
  foreach ($new_lines as $ln_utf) {
    $res = process_line($ln_utf, $cfg);
    $stored = $res['stored'] ?? $ln_utf;
    $dc = $res['__dc'] ?? 'other';
    if (!isset($out_by_dc[$dc])) $out_by_dc[$dc] = [];
    $out_by_dc[$dc][] = $stored;
    // set per-line cache (3600s TTL controlled by reader)
    line_cache_set($stored, ['__dc'=>$dc,'__ip_asn_map'=>$res['__ip_asn_map']??[]], $cfg);
    $processed++; $got_new = true;
    $st['last_new'][$idx] = time();
  }

  // write outputs with threshold
  if (!empty($out_by_dc)) {
    $misc = [];
    foreach ($out_by_dc as $dc=>$arr) {
      if (count($arr) >= $min_per_dc) write_dc_file_merged($dc,$arr,$cfg);
      else $misc = array_merge($misc,$arr);
    }
    if (!empty($misc)) write_dc_file_merged('misc',$misc,$cfg);
  }

  /* ===================== ADVANCEMENT RULES ===================== */
  $advance = false;

  // (A) اگر تکراری دیدیم، برو بعدی
  if ($cached_seen) $advance = true;

  // (B) اگر ≥2 کانفیگ جدید در همین دور دیدیم، برو بعدی
  if (!$advance && count($new_lines) >= 2) $advance = true;

  // (C) اگر دقیقاً 1 خط معتبر داریم و همان هم «جدید» است ⇒ بمان و منتظر تکراری شو
  if (!$advance && count($new_lines) === 1 && $total_valid === 1) {
    $st['watch_line'][$idx] = $new_lines[0]; // watch
  } else {
    // خارج از این حالت، watch را پاک کن
    $st['watch_line'][$idx] = null;
  }

  // (D) اگر هیچ جدیدی نبود ولی در حال watch بودیم و الان cached_seen=true ⇒ تکراری دیده شد ⇒ برو بعدی
  if (!$advance && empty($new_lines) && !empty($st['watch_line'][$idx]) && $cached_seen) {
    $advance = true;
    $st['watch_line'][$idx] = null;
  }

  // (E) سقف ۴۰ بار چک
  if (!$advance) {
    $st['check_count'][$idx] ++;
    $max_checks = max(1,(int)$max_checks);
    if ($st['check_count'][$idx] >= $max_checks) {
      $advance = true;
      $st['check_count'][$idx] = 0;
      $st['watch_line'][$idx] = null;
    }
  } else {
    $st['check_count'][$idx] = 0;
  }

  // (F) ایمنی زمانی: اگر خیلی وقت است خبری نشده، برو بعدی
  if (!$advance) {
    $now = time();
    $last_new = (int)($st['last_new'][$idx] ?? 0);
    if ($last_new > 0 && ($now - $last_new) >= (int)$max_age) {
      $advance = true;
      $st['check_count'][$idx] = 0;
      $st['watch_line'][$idx] = null;
    }
  }

  // انجام جابجایی
  if ($advance) {
    $idx = ($idx + 1) % $n;
    // مقادیر برای ایندکس جدید در دور بعد ست می‌شود
  }

  // stale counter به‌روز
  if ($got_new) $st['stale'][$idx] = 0;
  else $st['stale'][$idx] = (int)($st['stale'][$idx] ?? 0) + 1;
  if ($st['stale'][$idx] >= (int)$max_cycles) {
    $idx = ($idx + 1) % $n;
    $st['stale'][$idx] = 0;
    $st['check_count'][$idx] = 0;
    $st['watch_line'][$idx] = null;
  }

  // save state & summary
  $st['idx'] = $idx;
  save_state($paths, $st);

  $summary=[
    'time'=>date('c'),
    'active_link'=>$url,
    'idx'=>$idx,
    'inputs'=>count($lines),
    'unique_endpoints'=>$processed,
    'groups'=>array_map(fn($a)=>count($a),$out_by_dc),
    'got_new'=>$got_new,
    'advance'=>$advance,
    'check_count'=>$st['check_count'][$idx] ?? 0,
    'watching'=> $st['watch_line'][$idx] ? true : false,
  ];
  save_json($paths['SUMMARY'],$summary);
  save_json($paths['OUT_DIR'].'/summary.json',$summary);
  return $summary;
}